﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections;
using System.Diagnostics;
using System.Text;
using System.Linq;

namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Manipulator;
    using System.Drawing;


    /// <summary>
    ///
    /// </summary>
    public struct KeyAanchor
    {
        IAnmKeyFrame _key;
        bool _bLeft;

        /// <summary>
        ///
        /// </summary>
        KeyAanchor( IAnmKeyFrame key, bool bLeft )
        {
            _key = key;
            _bLeft = bLeft;
        }

        public IAnmKeyFrame IAnmKeyFrame {get { return _key; }}
        public bool bLeft { get { return _bLeft; } }
        public bool IsValid { get { return this.IAnmKeyFrame != null; } }

        /// <summary>
        ///
        /// </summary>
        static public KeyAanchor MakeLeftSlope( IAnmKeyFrame key )
        {
            return new KeyAanchor( key, true );
        }

        /// <summary>
        ///
        /// </summary>
        static public KeyAanchor MakeRightSlope( IAnmKeyFrame key )
        {
            return new KeyAanchor( key, false );
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class TargetAnmAttributeSet
    {
        /// <summary>
        /// 描画用カーブ情報
        /// </summary>
        public class DrawAnmCurve
        {
            public IAnmAttribute AnmAttr { get; private set; }
            public Color DrawColor { get; private set; }

            public DrawAnmCurve(IAnmAttribute anmAttr, Color drawColor)
            {
                this.AnmAttr = anmAttr;
                this.DrawColor = drawColor;
            }
        }

        List<IAnmAttribute>   _attributeSet = new List<IAnmAttribute>();
        List<DrawAnmCurve>    _allCurveSet = new List<DrawAnmCurve>();
        List<DrawAnmCurve>    _targetCurveSet = new List<DrawAnmCurve>();
        List<IAnmKeyFrame>    _targetKeySet   = new List<IAnmKeyFrame>();
        List<IAnmKeyFrame>    _selectedKeySet = new List<IAnmKeyFrame>();
        List<KeyAanchor>      _anchorSelectedKeySet = new List<KeyAanchor>();
        Dictionary<IAnmCurve, IAnmAttribute> _curveAttributeMap = new Dictionary<IAnmCurve, IAnmAttribute>();

        // アンメーションカーブ操作クラス
        AnmAttributeManipulator _anmAttributeMnp = new AnmAttributeManipulator();
        // アンメーションカーブ操作クラス
        AnmCurveManipulator _anmCurveMnp         = new AnmCurveManipulator();
        // キーフレーム操作クラス
        AnmKeyFrameManipulator     _currentKeyFrameMnp  = new AnmKeyFrameManipulator();

        IAnimFrameSection _targetFrameSection = null;

        public event EventHandler OnTargetCurveChange = null;
        public event EventHandler OnSelectedKeySetChange = null;

        /// <summary>
        /// 操作対象アニメーションカーブ
        /// </summary>
        public IAnmAttribute[] _TargetAttributeSet
        {
            get { return _attributeSet.ToArray(); }
        }

        /// <summary>
        /// _TargetAttributeSet 中の 全てのカーブ
        /// </summary>
        public IAnmCurve[] _AllCurveSet
        {
            get { return _allCurveSet.ConvertAll((drawCurve) => drawCurve.AnmAttr.ICurrentAnimationCurve).ToArray(); }
        }

        /// <summary>
        /// カーブを保持するアトリビュート全体
        /// </summary>
        public IEnumerable<IAnmAttribute> _AllAnmAttributeSet => _allCurveSet.Select(x => x.AnmAttr);

        /// <summary>
        /// 操作対象アニメーションカーブ
        /// </summary>
        public IAnmCurve[] _TargetCurveSet
        {
            get { return _targetCurveSet.ConvertAll((drawCurve) => drawCurve.AnmAttr.ICurrentAnimationCurve).ToArray(); }
        }

        /// <summary>
        /// 操作対象のカーブを保持するアトリビュート全体
        /// </summary>
        public IEnumerable<IAnmAttribute> _TargetAnmAttributeSet => _targetCurveSet.Select(x => x.AnmAttr);


        /// <summary>
        /// 操作対象アニメーションカーブ
        /// </summary>
        public DrawAnmCurve[] _TargetDrawCurveSet
        {
            get { return _targetCurveSet.ToArray(); }
        }

        /// <summary>
        /// 操作対象キーフレーム
        /// </summary>
        public IAnmKeyFrame[] _TargetKeyFrameSet
        {
            get
            {
                // すべての対象カーブのキーを集めます。
                ArrayList       keySet = new ArrayList();
                foreach (DrawAnmCurve curve in _targetCurveSet)
                {
                    keySet.AddRange( curve.AnmAttr.ICurrentAnimationCurve.IKeyFrameSet );
                }
                AnmKeyFrameComparer keyComparer = new AnmKeyFrameComparer();
                keySet.Sort( keyComparer );
                return keySet.ToArray( typeof( IAnmKeyFrame ) ) as IAnmKeyFrame[];
            }
        }

        /// <summary>
        ///
        /// </summary>
        public KeyAanchor[] _SelectedAnchorSet
        {
            get
            {
                return _anchorSelectedKeySet.ToArray();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public IAnimFrameSection TargetAnimFrameSection
        {
            get { return _targetFrameSection; }
        }

        /// <summary>
        ///
        /// </summary>
        ISubScene _CurrentSubScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }

        /// <summary>
        /// 選択キーフレーム
        /// </summary>
        public IAnmKeyFrame[] _SelectedKeyFrameSet
        {
            get
            {
                List<IAnmKeyFrame> selKeySet = new List<IAnmKeyFrame>();
                foreach( DrawAnmCurve curve in _targetCurveSet )
                {
                    selKeySet.AddRange(curve.AnmAttr.ICurrentAnimationCurve.SelectedKeySet);
                }
                return selKeySet.ToArray();
            }
        }

        /// <summary>
        /// 親ウインドウ
        /// </summary>
        public LEToolWindow ParentWindow { get; set; }

        ///---------------------------- アトリビュート

        /// <summary>
        /// 対象アトリビュート
        /// </summary>
        public void SetTargetAttribute(IAnmAttribute[] targetSet, Func<IAnmAttribute, bool> condition)
        {
            this.ResetTargetAttribute();

            _attributeSet.AddRange(targetSet);

            // 操作対象カーブ
            {
                bool bModified = false;
                foreach (IAnmAttribute attr in targetSet)
                {
                    bModified |= AddTargetCurve_(_targetCurveSet, attr, condition);
                }

                if (bModified)
                {
                    ResetSelectKeyAll();
                    NotifyTargetCurveChange_();
                }
            }

            // すべてのカーブ
            foreach (IAnmAttribute attr in targetSet)
            {
                AddTargetCurve_(_allCurveSet, attr, condition);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void SetTargetAnimFrameSection(IAnimFrameSection frameSection)
        {
            _targetFrameSection = frameSection;
        }

        /// <summary>
        /// リセット
        /// </summary>
        public void ResetTargetAttribute()
        {
            ResetTargetCurve_();

            _allCurveSet.Clear();
            _attributeSet.Clear();
            _curveAttributeMap.Clear();
        }

        /// <summary>
        /// マップにカーブとアトリビュートのペアを登録します。
        /// </summary>
        static void StoreCurveAttributePairRecursive_( Dictionary<IAnmCurve, IAnmAttribute> map, IAnmAttribute attribute )
        {
            foreach (IAnmCurve curve in attribute?.Curves ?? new List<IAnmCurve>().ToArray())
            {
                // 登録
                map[curve] = attribute;
            }

            for ( int i = 0 ; i < attribute.NumSubAttribute ; i++ )
            {
                StoreCurveAttributePairRecursive_( map, attribute.FindSubAttributeByIdx( i ) );
            }
        }

        /// <summary>
        /// カーブの持ち主アトリビュートを検索します。
        ///
        /// ２回目以降の検索は、マップを利用して高速に検索します。
        /// マップはターゲットアトリビュートが変更になった時点で、クリアされます。
        /// </summary>
        public IAnmAttribute FindCurveOwnerAttribute( IAnmCurve curve )
        {
            // 必要があれば、マップを初期化します
            if( _curveAttributeMap.Count == 0 )
            {
                foreach (IAnmAttribute attribute in _attributeSet)
                {
                    StoreCurveAttributePairRecursive_(_curveAttributeMap, attribute);
                }
            }

            IAnmAttribute ownerAttribute = null;
            _curveAttributeMap.TryGetValue( curve, out ownerAttribute );

            return ownerAttribute;
        }



        ///---------------------------- カーブ

        /// <summary>
        ///
        /// </summary>
        public bool Contains(IAnmCurve curve)
        {
            return _targetCurveSet.Find((drawCurve) => object.ReferenceEquals(drawCurve.AnmAttr.ICurrentAnimationCurve, curve)) != null;
        }

        /// <summary>
        ///
        /// </summary>
        public void SetTargetCurve(IAnmAttribute[] attrSet, Func<IAnmAttribute, bool> condition)
        {
            ResetTargetCurve_();

            bool bModified = false;
            foreach (IAnmAttribute attr in attrSet)
            {
                bModified |= AddTargetCurve_(_targetCurveSet, attr, condition);
            }

            if (bModified)
            {
                ResetSelectKeyAll();
                NotifyTargetCurveChange_();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void ResetTargetCurve_()
        {
            BeginKeySelectChange();
            ResetSelectKeyAll();
            EndKeySelectChange();

            _targetCurveSet.Clear();
            NotifyTargetCurveChange_();
        }

        /// <summary>
        /// カーブの描画色を取得します。
        /// </summary>
        Color GetAttrDrawColor_(IAnmAttribute attr)
        {
            if (attr.ICurrentAnimationCurve == null)
            {
                return Color.Black;
            }

            if (attr.ICurrentAnimationCurve.IsReadOnlyLocked)
            {
                return Color.Black;
            }

            return AttributerUIHelper.GetCurveColor(attr, Color.LightGray);
        }

        /// <summary>
        /// 含んでいるか
        /// </summary>
        private bool ContainsAttrCurve_(List<DrawAnmCurve> curveSet, IAnmAttribute attr)
        {
            return curveSet.Find((drawCurve) => object.ReferenceEquals(drawCurve.AnmAttr.ICurrentAnimationCurve, attr.ICurrentAnimationCurve)) != null;
        }

        /// <summary>
        /// 追加
        /// </summary>
        private void AddAttrCurve_(List<DrawAnmCurve> curveSet, IAnmAttribute attr)
        {
            curveSet.Add(new DrawAnmCurve(attr, GetAttrDrawColor_(attr)));
        }

        /// <summary>
        /// 削除
        /// </summary>
        private void RemoveAttrCurve_(List<DrawAnmCurve> curveSet, IAnmAttribute attr)
        {
            curveSet.RemoveAll((drawCurve) => object.ReferenceEquals(drawCurve.AnmAttr.ICurrentAnimationCurve, attr.ICurrentAnimationCurve));
        }

        /// <summary>
        /// 対象カーブに設定します。
        /// </summary>
        bool AddTargetCurve_(List<DrawAnmCurve> curveSet, IAnmAttribute attr, Func<IAnmAttribute, bool> condition)
        {
            bool bModified = false;

            if (!condition(attr))
            {
                return false;
            }

            // アニメーションカーブを持つのならば、グラフビューの操作対象に設定する。
            if( attr.HasAnimationCurve )
            {
                AddAttrCurve_(curveSet, attr);
                bModified = true;
            }

            // サブアトリビュートに関して同様の処理を繰り返す。
            if( attr.HasSubAttribute )
            {
                for( int i = 0; i < attr.NumSubAttribute; i++ )
                {
                    bModified |= AddTargetCurve_(curveSet, attr.FindSubAttributeByIdx(i), condition);
                }
            }
            return bModified;
        }

        ///---------------------------- キー

        /// <summary>
        ///
        /// </summary>
        public void BeginKeySelectChange()
        {
            if( _CurrentSubScene != null )
            {
                _CurrentSubScene.BeginMassiveModify();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void EndKeySelectChange()
        {
            NotifyTargetKeyChange_();
            if( _CurrentSubScene != null )
            {
                _CurrentSubScene.EndMassiveModify();
            }
        }


        /// <summary>
        /// キーをトグル選択します。
        /// </summary>
        public void SelectKey( IAnmKeyFrame[] keySet )
        {
            SelectKey( keySet, false );
        }

        /// <summary>
        /// キーをトグル選択します。
        /// </summary>
        public void SelectKey( IAnmKeyFrame[] keySet, bool bToggleSelect )
        {
            Debug.Assert( keySet != null );
            if( _CurrentSubScene != null )
            {
                foreach( IAnmKeyFrame key in keySet )
                {
                    _anmCurveMnp.BindTarget( key.OwnerIAnmCurve );
                    if( bToggleSelect && key.IsSelected )
                    {
                        _anmCurveMnp.RemoveSelectedSet( key );
                    }
                    else
                    {
                        _anmCurveMnp.AddSelectedSet( key );
                    }
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void SelectKeyAnchor( KeyAanchor[] keySet )
        {
            SelectKeyAnchor( keySet, false );
        }

        /// <summary>
        ///
        /// </summary>
        public void SelectKeyAnchor( KeyAanchor[] keySet, bool bToggleSelect )
        {
            if( bToggleSelect )
            {
                foreach( KeyAanchor keyAanchor in keySet )
                {
                    if( _anchorSelectedKeySet.Contains( keyAanchor ) )
                    {
                        _anchorSelectedKeySet.Remove( keyAanchor );
                    }
                    else
                    {
                        _anchorSelectedKeySet.Add( keyAanchor );
                    }
                }
            }
            else
            {
                _anchorSelectedKeySet.AddRange( keySet );
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void ResetSelectedTangentAnchor()
        {
            _anchorSelectedKeySet.Clear();
        }

        /// <summary>
        ///
        /// </summary>
        public void DeleteSlectedKey()
        {
            if( _CurrentSubScene != null )
            {
                List<IAnmCurve>		ownerCurveSet = new List<IAnmCurve>();

                // 選択中のキーを消去します
                foreach( IAnmKeyFrame key in _SelectedKeyFrameSet )
                {
                    if( !ownerCurveSet.Contains( key.OwnerIAnmCurve ) )
                    {
                        ownerCurveSet.Add( key.OwnerIAnmCurve );
                    }
                    _currentKeyFrameMnp.BindTarget( key );
                    _currentKeyFrameMnp.RemoveFromOwner( false );
                }

                // すべてのキーを消去した後に、まとめてカーブを再評価します。
                foreach( IAnmCurve ownerCurve in ownerCurveSet )
                {
                    _anmCurveMnp.BindTarget( ownerCurve );
                    _anmCurveMnp.Update( true );
                }

                ResetSelectKeyAll();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void ResetSelectKeyAll()
        {
            if( _CurrentSubScene != null )
            {
                ResetSelectedTangentAnchor();
                foreach( DrawAnmCurve anmCurve in _targetCurveSet )
                {
                    _anmAttributeMnp.BindTarget(anmCurve.AnmAttr);
                    _anmAttributeMnp.ResetSelectedSet();
                }
            }
        }



        ///---------------------------- 通知

        /// <summary>
        /// ターゲットの変更を通知する
        /// </summary>
        void NotifyTargetKeyChange_()
        {
            if( OnSelectedKeySetChange != null )
            {
                OnSelectedKeySetChange( this, null );
            }
        }

        /// <summary>
        ///
        /// </summary>
        void NotifyTargetCurveChange_()
        {
            if( OnTargetCurveChange != null )
            {
                OnTargetCurveChange( this, null );
            }
        }
    }
}
