﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Drawing;
using System.Drawing.Imaging;

namespace LayoutEditor
{
    public sealed class CursorMgr
    {
        /// <summary>
        /// IconInfo構造体を取得する
        /// </summary>
        [DllImport("user32.dll")]
        [return: MarshalAs(UnmanagedType.Bool)]
        public static extern bool GetIconInfo(IntPtr hIcon, ref IconInfo pIconInfo);

        /// <summary>
        /// IconInfo構造体からHIconを得る
        /// </summary>
        [DllImport("user32.dll")]
        public static extern IntPtr CreateIconIndirect(ref IconInfo icon);

        [DllImport("user32.dll")]
        private static extern IntPtr LoadCursorFromFile(string lpFileName);

        [DllImport("gdi32.dll", SetLastError = true)]
        private static extern bool DeleteObject(IntPtr hObject);

        /// <summary>
        /// 透明度を考慮して、Cursor から Bitmap を生成する。
        /// 参考：https://stackoverflow.com/questions/4451839/how-to-render-a-transparent-cursor-to-bitmap-preserving-alpha-channel
        /// </summary>
        private static Bitmap CreateBitmapFromCursor_(Cursor cur)
        {
            IconInfo ii = new IconInfo();
            try
            {
                GetIconInfo(cur.Handle, ref ii);

                if (ii.hbmColor != IntPtr.Zero)
                {
                    Bitmap bmp = Bitmap.FromHbitmap(ii.hbmColor);

                    BitmapData bd = bmp.LockBits(new Rectangle(0, 0, bmp.Width, bmp.Height), ImageLockMode.ReadOnly, bmp.PixelFormat);
                    Bitmap dstBitmap = new Bitmap(bd.Width, bd.Height, bd.Stride, PixelFormat.Format32bppArgb, bd.Scan0);
                    bmp.UnlockBits(bd);

                    return new Bitmap(dstBitmap);
                }
                else
                {
                    // 境界がアンチエイリアスされません。それほど目立たないので妥協します。
                    Bitmap cursorBmp = new Bitmap(cur.Size.Width, cur.Size.Height, PixelFormat.Format32bppPArgb);
                    using (Graphics g = Graphics.FromImage(cursorBmp))
                    {
                        cur.Draw(g, new Rectangle(0, 0, cursorBmp.Width, cursorBmp.Height));
                    }

                    return new Bitmap(cursorBmp);
                }
            }
            finally
            {
                DeleteObject(ii.hbmColor);
                DeleteObject(ii.hbmMask);
            }
        }

        /// <summary>
        /// Bitmapからを作成する
        /// </summary>
        public static Cursor CreateCursor(Bitmap bmp, int xHotspot, int yHotspot)
        {
            // システムのカーソルを取得
            Cursor defaultCursor = System.Windows.Forms.Cursors.Default;
            Bitmap cursorBmp = CreateBitmapFromCursor_(defaultCursor);

            // アルファを抜いた領域を求める
            int alphaRangeMinX = int.MaxValue;
            int alphaRangeMinY = int.MaxValue;
            int alphaRangeMaxX = 0;
            int alphaRangeMaxY = 0;
            for (int x = 0; x < cursorBmp.Width; x++)
            {
                for (int y = 0; y < cursorBmp.Height; y++)
                {
                    Color c = cursorBmp.GetPixel(x, y);
                    if (c.A != 0)
                    {
                        alphaRangeMinX = Math.Min(alphaRangeMinX, x);
                        alphaRangeMinY = Math.Min(alphaRangeMinY, y);
                        alphaRangeMaxX = Math.Max(alphaRangeMaxX, x);
                        alphaRangeMaxY = Math.Max(alphaRangeMaxY, y);
                    }
                }
            }

            int alphaRectWidth = alphaRangeMaxX + 1;
            int alphaRectHeight = alphaRangeMaxY + 1;

            // アルファ矩形＋追加画像のサイズでアイコンを作る
            int width = alphaRectWidth + bmp.Width;
            int height = Math.Max(alphaRectHeight, bmp.Height);

            Bitmap iconBmp = new Bitmap(width, height, PixelFormat.Format32bppPArgb);
            using (Graphics g = Graphics.FromImage(iconBmp))
            {
                g.DrawImage(cursorBmp, 0, 0);
                g.DrawImage(bmp, new Point(alphaRectWidth, alphaRectHeight - bmp.Height)); // 追加アイコンの位置はカーソルのbottomに合わせる
            }

            // HIconへのポインタ
            IntPtr ptr = iconBmp.GetHicon();

            // アイコン情報
            IconInfo tmp = new IconInfo();

            // アイコン情報に画像を与える
            GetIconInfo(ptr, ref tmp);

            // アイコンのポイント位置を指定
            tmp.xHotspot = xHotspot;
            tmp.yHotspot = yHotspot;

            // アイコンはカーソルである事を示す
            tmp.fIcon = false;

            // アイコンを作成
            ptr = CreateIconIndirect(ref tmp);
            return new Cursor(ptr);
        }
    }

    ///<summary>IconInfo構造体</summary>
    public struct IconInfo
    {

        public bool fIcon;      // iconならtrue,curならfalse
        public int xHotspot;    // x座標のHotspot
        public int yHotspot;    // y座標のHotspot
        public IntPtr hbmMask;  // アイコン画像のマスク( 上半分がANDマスク、下半分がXORマスク )
        public IntPtr hbmColor; // カラービットマップ、白黒のアイコンならなしでもよい
    }

}
