﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Diagnostics;
using System.Text;
using System.Windows.Forms;
using System.Drawing.Drawing2D;

namespace LayoutEditor.Controls
{
    using Forms.ToolWindows;
    using LECore;
    using LECore.Manipulator;
    using LECore.Structures;
    using LECore.Structures.Core;
    using src.Controls;

    public partial class VertexColorEditPanel : UserControl
        , IAnimationMarkUsable
    {
        enum TargetColorKind
        {
            None,
            LT,
            RT,
            LB,
            RB,
            All,
            L,
            T,
            R,
            B,
        }

        public delegate void OnColorChange(Color LT, Color RT, Color LB, Color RB);
        public event OnColorChange ColorChange = null;

        #region フィールド

        Color _colorLT = Color.Yellow;
        Color _colorRT = Color.Green;
        Color _colorLB = Color.Blue;
        Color _colorRB = Color.Red;

        TargetColorKind _targetColor = TargetColorKind.None;
        Label _activeLabel = null;

        Point _posMouseDownForDragDrop;

        readonly GraphicsPath _graphicsPath = new GraphicsPath();
        readonly Brush _brush = new HatchBrush(HatchStyle.LargeCheckerBoard, Color.White, Color.DarkGray);
        readonly Image _image;

        readonly ColorButton _colorButton = new ColorButton();
        readonly ColorPickerAdapter _colorPickerAdapter;

        bool _useAnimationMark = false;

        AnmCurveManipulator _anmCurveMnp = new AnmCurveManipulator();

        #endregion

        #region プロパティ

        /// <summary>
        /// 操作対象カラーを設定します。
        /// </summary>
        TargetColorKind _TargetColorKind
        {
            set
            {
                if (_targetColor != value)
                {
                    _targetColor = value;
                }
            }
        }

        /// <summary>
        /// 中心の色
        /// </summary>
        Color _CenterColor
        {
            get
            {
                return Color.FromArgb(
                                      (_colorLT.A + _colorRT.A + _colorLB.A + _colorRB.A) / 4,
                                      (_colorLT.R + _colorRT.R + _colorLB.R + _colorRB.R) / 4,
                                      (_colorLT.G + _colorRT.G + _colorLB.G + _colorRB.G) / 4,
                                      (_colorLT.B + _colorRT.B + _colorLB.B + _colorRB.B) / 4);
            }
        }

        ///
        private Color ColorMixture(Color colorA, Color colorB)
        {
            return Color.FromArgb
                ((colorA.A + colorB.A) / 2,
                 (colorA.R + colorB.R) / 2,
                 (colorA.G + colorB.G) / 2,
                 (colorA.B + colorB.B) / 2);
        }

        /// <summary>
        /// 操作対象カラー
        /// </summary>
        Color _TargetColor
        {
            set
            {
                SetTargetColor_(_targetColor, value);
            }

            get
            {
                return GetTargetColor_(_targetColor);
            }
        }

        /// <summary>
        /// アニメーションマークのリスト
        /// </summary>
        public AnimationMark[] AnimationMarkArray
        {
            get
            {
                List<AnimationMark> list = new List<AnimationMark>();
                list.Add(_pnlMarkLT);
                list.Add(_pnlMarkRT);
                list.Add(_pnlMarkLB);
                list.Add(_pnlMarkRB);

                return list.ToArray();
            }
        }

        #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public VertexColorEditPanel()
        {
            InitializeComponent();


            this.SetStyle(ControlStyles.SupportsTransparentBackColor, true);
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, true);
            this.SetStyle(ControlStyles.AllPaintingInWmPaint, true);

            _lblLT.Tag = TargetColorKind.LT;
            _lblRT.Tag = TargetColorKind.RT;
            _lblLB.Tag = TargetColorKind.LB;
            _lblRB.Tag = TargetColorKind.RB;
            _lblC.Tag = TargetColorKind.All;
            _lblL.Tag = TargetColorKind.L;
            _lblT.Tag = TargetColorKind.T;
            _lblR.Tag = TargetColorKind.R;
            _lblB.Tag = TargetColorKind.B;

            _colorButton.EnableAlpha = true;
            _colorPickerAdapter = new ColorPickerAdapter(_colorButton);
            _colorPickerAdapter.ColorEdit += Event_ColorEditForm_ColorEdit;
            _colorPickerAdapter.ColorPickerClosed += Event_ColorPickerAdapter_ColorPickerClosed;

            _image = new Bitmap(_pnlColor.Width, _pnlColor.Height);

            // コンテキストメニューを追加します
            RegistContextMenu("-", null, null);
            RegistContextMenu(StringResMgr.Get("ANIMCONTEXTMENU_ADD_R_KEY"), null, Event_AddKey, new VertexColorEditPanelArgs(0));
            RegistContextMenu(StringResMgr.Get("ANIMCONTEXTMENU_ADD_G_KEY"), null, Event_AddKey, new VertexColorEditPanelArgs(1));
            RegistContextMenu(StringResMgr.Get("ANIMCONTEXTMENU_ADD_B_KEY"), null, Event_AddKey, new VertexColorEditPanelArgs(2));
            RegistContextMenu(StringResMgr.Get("ANIMCONTEXTMENU_ADD_A_KEY"), null, Event_AddKey, new VertexColorEditPanelArgs(3));

            // AnimationMarkの持つコンテキストメニューを関連付けます
            _lblLT.BindAnimationMark(_pnlMarkLT);
            _lblRT.BindAnimationMark(_pnlMarkRT);
            _lblLB.BindAnimationMark(_pnlMarkLB);
            _lblRB.BindAnimationMark(_pnlMarkRB);
        }

        /// <summary>
        /// 色を設定します。
        /// </summary>
        public void Set(Color LT, Color RT, Color LB, Color RB)
        {
            bool bChanged = false;
            if (_colorLT != LT)
            {
                _colorLT = LT;
                bChanged |= true;
            }

            if (_colorRT != RT)
            {
                _colorRT = RT;
                bChanged |= true;
            }

            if (_colorLB != LB)
            {
                _colorLB = LB;
                bChanged |= true;
            }

            if (_colorRB != RB)
            {
                _colorRB = RB;
                bChanged |= true;
            }

            if (bChanged)
            {
                RedrawPctBox_();
            }
        }

        /// <summary>
        /// コンテキストメニューを追加します。
        /// </summary>
        public void RegistContextMenu(string name, Image image, EventHandler handler, EventArgs args = null)
        {
            if (handler != null)
            {
                _pnlMarkLT.RegistContextMenu(name, image, (s, e) => handler(s, CreateEventArgs_(args, _pnlMarkLT)));
                _pnlMarkRT.RegistContextMenu(name, image, (s, e) => handler(s, CreateEventArgs_(args, _pnlMarkRT)));
                _pnlMarkLB.RegistContextMenu(name, image, (s, e) => handler(s, CreateEventArgs_(args, _pnlMarkLB)));
                _pnlMarkRB.RegistContextMenu(name, image, (s, e) => handler(s, CreateEventArgs_(args, _pnlMarkRB)));
            }
            else
            {
                _pnlMarkLT.RegistContextMenu(name, image, null);
                _pnlMarkRT.RegistContextMenu(name, image, null);
                _pnlMarkLB.RegistContextMenu(name, image, null);
                _pnlMarkRB.RegistContextMenu(name, image, null);
            }
        }
        private EventArgs CreateEventArgs_(EventArgs args, src.Controls.AnimationMark targetMark)
        {
            if (args == null)
            {
                return EventArgs.Empty;
            }

            VertexColorEditPanelArgs vcepArgs = args as VertexColorEditPanelArgs;
            if (vcepArgs != null)
            {
                vcepArgs.TargetMark = targetMark;
                return vcepArgs;
            }

            return args;
        }

        /// <summary>
        /// ViewManagerへのメッセージハンドラを設定します。
        /// </summary>
        public void SetViewManagerMessageHandler(Action<ViewManagerMessage> handler)
        {
            _pnlMarkLT.SendMessageToViewManager = handler;
            _pnlMarkRT.SendMessageToViewManager = handler;
            _pnlMarkLB.SendMessageToViewManager = handler;
            _pnlMarkRB.SendMessageToViewManager = handler;
        }

        /// <summary>
        /// 操作対象のアトリビュートを設定します
        /// </summary>
        public void SetTargetAttribute(IVertexColor4Holder holder)
        {
            Debug.Assert(holder != null);

            _pnlMarkLT.SetTargetAttribute(holder.VtxCol_LTIAnmAttr);
            _pnlMarkRT.SetTargetAttribute(holder.VtxCol_RTIAnmAttr);
            _pnlMarkLB.SetTargetAttribute(holder.VtxCol_LBIAnmAttr);
            _pnlMarkRB.SetTargetAttribute(holder.VtxCol_RBIAnmAttr);
        }

        /// <summary>
        /// アニメーションマークの色を更新します
        /// </summary>
        public void UpdateMarkColor()
        {
            _pnlMarkLT.UpdateMarkColor();
            _pnlMarkRT.UpdateMarkColor();
            _pnlMarkLB.UpdateMarkColor();
            _pnlMarkRB.UpdateMarkColor();
        }

        /// <summary>
        /// コンテキストメニューを更新します
        /// </summary>
        public void UpdateContextMenu()
        {
            _pnlMarkLT.UpdateContextMenu();
            _pnlMarkRT.UpdateContextMenu();
            _pnlMarkLB.UpdateContextMenu();
            _pnlMarkRB.UpdateContextMenu();
        }

        /// <summary>
        ///
        /// </summary>
        private void SetActive(Label lbl)
        {
            TargetColorKind kind = (TargetColorKind)lbl.Tag;
            _TargetColorKind = kind;

            lbl.BorderStyle = BorderStyle.FixedSingle;
            TryToShowPopupForm_(Control.MousePosition);
            SetActiveLabel_(lbl);
        }

        #region IAnimationMarkUsableメンバ

        /// <summary>
        /// アニメーションマークを使うか？
        /// </summary>
        public bool UseAnimationMark
        {
            get { return _useAnimationMark; }
            set
            {
                _useAnimationMark = value;
                _pnlMarkLT.Visible = value;
                _pnlMarkRT.Visible = value;
                _pnlMarkLB.Visible = value;
                _pnlMarkRB.Visible = value;
                _lblLT.UseContextMenu = value;
                _lblRT.UseContextMenu = value;
                _lblLB.UseContextMenu = value;
                _lblRB.UseContextMenu = value;
            }
        }

        /// <summary>
        /// アニメーションマークを更新します
        /// </summary>
        public void UpdateAnimationMark()
        {
            UpdateMarkColor();
            UpdateContextMenu();
        }

        #endregion

        #region イベント・ハンドラ

        /// <summary>
        /// 再描画
        /// </summary>
        private void UpdateImage_()
        {
            Graphics g = Graphics.FromImage(_image);

            Rectangle rect = new Rectangle(0, 0, _pnlColor.Width, _pnlColor.Height);
            _graphicsPath.Reset();
            _graphicsPath.AddRectangle(rect);

            // 格子を書く
            g.FillRectangle(_brush, rect);

            using (PathGradientBrush gb = new PathGradientBrush(_graphicsPath))
            {
                //パスグラデーションの中心の色を白にする
                gb.CenterColor = _CenterColor;

                //パス内の点に対応している色を指定する
                gb.SurroundColors = new Color[] { _colorLT, _colorRT, _colorRB, _colorLB };

                //四角を描く
                g.FillRectangle(gb, rect);
            }

            _pnlColor.BackgroundImage = _image;
            _pnlColor.Invalidate();
        }

        /// <summary>
        /// マウスダウン
        /// </summary>
        private void Event_PctColor_MouseDown(object sender, MouseEventArgs e)
        {
            _TargetColorKind = TargetColorKind.None;
        }

        /// <summary>
        /// カラーピッカー:色変更イベントハンドラ
        /// </summary>
        void Event_ColorEditForm_ColorEdit(object sender, ColorEditEventArgs e)
        {
            if (e.EditFixed)
            {
                _TargetColor = e.Color.ToSystemColor();
            }
        }

        /// <summary>
        ///
        /// </summary>
        void Event_ColorPickerAdapter_ColorPickerClosed(object sender, EventArgs e)
        {
            SetActiveLabel_(null);
            TryToShowPopupForm_(Control.MousePosition);
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_Lbl_MouseClick(object sender, MouseEventArgs e)
        {
            _posMouseDownForDragDrop = new Point(e.X, e.Y);

            Label lbl = sender as Label;
            SetActive(lbl);
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_Lbl_MouseMove(object sender, MouseEventArgs e)
        {
            if ((e.Button & MouseButtons.Left) == 0)
            {
                _posMouseDownForDragDrop = new Point(e.X, e.Y);
            }
            else
            {
                // ドラックが検知されたら...
                if (Math.Abs(e.X - _posMouseDownForDragDrop.X) > SystemInformation.DragSize.Width ||
                      Math.Abs(e.Y - _posMouseDownForDragDrop.Y) > SystemInformation.DragSize.Height)
                {
                    Label lbl = sender as Label;

                    TargetColorKind srcKind = (TargetColorKind)lbl.Tag;
                    ColorData colorData = new ColorData(new FloatColor(GetTargetColor_(srcKind)), "");
                    lbl.DoDragDrop(new Tuple<ColorData, ColorEditPanel.DragDropAction>(colorData, _selectDragDropAction()), DragDropEffects.Copy);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_Lbl_MouseHover(object sender, EventArgs e)
        {
            if (ColorPickerDialog.IsShowingDialog)
            {
                Label lbl = sender as Label;
                SetActive(lbl);
            }
        }

        /// <summary>
        /// すべてを白に設定。
        /// </summary>
        private void Event_BtnSetColorWhite_Click(object sender, EventArgs e)
        {
            _colorLT = CopyColor_(_colorLT, Color.White);
            _colorRT = CopyColor_(_colorRT, Color.White);
            _colorLB = CopyColor_(_colorLB, Color.White);
            _colorRB = CopyColor_(_colorRB, Color.White);

            RedrawPctBox_();

            NotifyColorChangeEvent_();
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_Lbl_DragEnter(object sender, DragEventArgs e)
        {
            if (ColorData.CheckDropItemHasColorDataWithAction(e))
            {
                e.Effect = e.AllowedEffect;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }

        }

        /// <summary>
        ///
        /// </summary>
        private void Event_Lbl_DragDrop(object sender, DragEventArgs e)
        {
            Tuple<ColorData, ColorEditPanel.DragDropAction> data;
            if (ColorData.TryToGetColorDataFromDropItem(e, out data))
            {
                Label dstLabel = sender as Label;
                TargetColorKind dstKind = (TargetColorKind)dstLabel.Tag;

                FloatColor targetColor = new FloatColor(GetTargetColor_(dstKind));
                FloatColor color = ColorEditPanel.getCopyColor(targetColor, data.Item1.Color, data.Item2);

                SetTargetColor_(dstKind, color.ToSystemColor());
            }
        }

        /// <summary>
        /// キーを追加します
        /// </summary>
        private void Event_AddKey(object sender, EventArgs e)
        {
            VertexColorEditPanelArgs args = e as VertexColorEditPanelArgs;

            foreach (IAnmAttribute target in args.TargetMark.Target)
            {
                IAnmAttribute attr = target.FindSubAttributeByIdx(args.Index);

                _anmCurveMnp.BindTarget(attr.ICurrentAnimationCurve);
                _anmCurveMnp.MakeKeyFrame(GlobalTime.Inst.Time, attr.Value);
            }
        }

        #endregion

        #region private

        /// <summary>
        /// 有効な成分だけをコピーした色を取得します。
        /// </summary>
        Color CopyColor_(Color copyDstColor, Color copySrcColor)
        {
            Color color = _chkColorEditEnabled.Checked ? copySrcColor : copyDstColor;
            byte a = _chkAlphaEditEnabled.Checked ? copySrcColor.A : copyDstColor.A;

            return Color.FromArgb(a, color);
        }

        /// <summary>
        ///
        /// </summary>
        void SetTargetColor_(TargetColorKind kind, Color color)
        {
            bool bChanged = false;

            switch (kind)
            {
                case TargetColorKind.LT:
                    _colorLT = CopyColor_(_colorLT, color); bChanged = true; break;
                case TargetColorKind.RT:
                    _colorRT = CopyColor_(_colorRT, color); bChanged = true; break;
                case TargetColorKind.LB:
                    _colorLB = CopyColor_(_colorLB, color); bChanged = true; break;
                case TargetColorKind.RB:
                    _colorRB = CopyColor_(_colorRB, color); bChanged = true; break;
                case TargetColorKind.All:
                    _colorLT = CopyColor_(_colorLT, color);
                    _colorRT = CopyColor_(_colorRT, color);
                    _colorLB = CopyColor_(_colorLB, color);
                    _colorRB = CopyColor_(_colorRB, color);
                    bChanged = true; break;

                case TargetColorKind.L:
                    _colorLT = CopyColor_(_colorLT, color);
                    _colorLB = CopyColor_(_colorLB, color);
                    bChanged = true;
                    break;

                case TargetColorKind.T:
                    _colorLT = CopyColor_(_colorLT, color);
                    _colorRT = CopyColor_(_colorRT, color);
                    bChanged = true;
                    break;

                case TargetColorKind.R:
                    _colorRT = CopyColor_(_colorRT, color);
                    _colorRB = CopyColor_(_colorRB, color);
                    bChanged = true;
                    break;

                case TargetColorKind.B:
                    _colorLB = CopyColor_(_colorLB, color);
                    _colorRB = CopyColor_(_colorRB, color);
                    bChanged = true;
                    break;

                default:
                    break;
            }

            if (bChanged)
            {
                RedrawPctBox_();
                NotifyColorChangeEvent_();
            }
        }

        /// <summary>
        ///
        /// </summary>
        Color GetTargetColor_(TargetColorKind kind)
        {
            switch (kind)
            {
                case TargetColorKind.LT:
                    return _colorLT;
                case TargetColorKind.RT:
                    return _colorRT;
                case TargetColorKind.LB:
                    return _colorLB;
                case TargetColorKind.RB:
                    return _colorRB;
                case TargetColorKind.All:
                    return _CenterColor;

                case TargetColorKind.L:
                    return ColorMixture(_colorLT, _colorLB);

                case TargetColorKind.T:
                    return ColorMixture(_colorLT, _colorRT);

                case TargetColorKind.R:
                    return ColorMixture(_colorRT, _colorRB);

                case TargetColorKind.B:
                    return ColorMixture(_colorLB, _colorRB);

                default:
                    return Color.Empty;
            }
        }

        /// <summary>
        /// PctBoxを再描画します。
        /// </summary>
        void RedrawPctBox_()
        {
            UpdateImage_();
        }

        /// <summary>
        ///
        /// </summary>
        void TryToShowPopupForm_(Point positon)
        {
            ColorPickerDialog.EndConnection(_colorPickerAdapter);
            if (_targetColor != TargetColorKind.None)
            {

                // 色を設定して
                _colorButton.Color.FromSystemColor(_TargetColor);
                _colorPickerAdapter.NotifyUpdate();

                // 表示
                // クリックして、表示
                _colorButton.PerformClick();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetActiveLabel_(Label lable)
        {
            // 設定済みならリセットします。
            if (_activeLabel != null &&
                _activeLabel != lable)
            {
                _activeLabel.BorderStyle = BorderStyle.None;
                _TargetColorKind = TargetColorKind.None;
            }

            _activeLabel = lable;
        }

        /// <summary>
        /// カラー変更イベントを通知します。
        /// </summary>
        void NotifyColorChangeEvent_()
        {
            if (ColorChange != null)
            {
                ColorChange(_colorLT, _colorRT, _colorLB, _colorRB);
            }
        }

        /// <summary>
        /// ドラッグドロップ時のアクションを決める。
        /// </summary>
        private ColorEditPanel.DragDropAction _selectDragDropAction()
        {
            if (_chkColorEditEnabled.Checked && _chkAlphaEditEnabled.Checked)
            {
                return ColorEditPanel.DragDropAction.Copy_Both;
            }
            else if (_chkColorEditEnabled.Checked)
            {
                return ColorEditPanel.DragDropAction.Copy_RGB;
            }
            else if (_chkAlphaEditEnabled.Checked)
            {
                return ColorEditPanel.DragDropAction.Copy_Alpha;
            }
            else
            {
                return ColorEditPanel.DragDropAction.Copy_None;
            }
        }

        #endregion private
    }

    /// <summary>
    /// イベントパラメータ型
    /// </summary>
    public sealed class VertexColorEditPanelArgs : EventArgs
    {
        private src.Controls.AnimationMark _targetMark;
        private readonly int _index;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public VertexColorEditPanelArgs(src.Controls.AnimationMark targetMark, int index)
        {
            _targetMark = targetMark;
            _index = index;
        }
        public VertexColorEditPanelArgs(int index)
        {
            _index = index;
        }

        /// <summary>
        /// 操作対象のアニメーションマーク
        /// </summary>
        public src.Controls.AnimationMark TargetMark
        {
            get { return _targetMark; }
            set { _targetMark = value; }
        }

        /// <summary>
        /// 色番号
        /// </summary>
        public int Index
        {
            get { return _index; }
        }
    }
}
