﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using System.Windows.Forms.Design;



namespace LayoutEditor.Controls
{
    using LECore;
    using LECore.Util;

    /// <summary>
    /// 値テキストボックスクラス。
    /// </summary>
    public class UINumericUpDown : NumericUpDown, FixEditingState
    {
        /// <summary>イベント発行制御</summary>
        protected static bool _invokeEvent = true;
        /// <summary>内部テキストボックス</summary>
        private readonly TextBox _internalTextBox;
        Decimal               _lastValue;
        const int _MaxAdjustmentDecimalPlace = 7;

        /// <summary>
        /// クラス変数
        /// </summary>
        public static bool InvokeEvent
        {
            get { return _invokeEvent; }
            set { _invokeEvent = value; }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public UINumericUpDown()
        {
            //デフォルト設定をコンストラクタで上書きすると、
            //フォームデザイナでうまく機能しない（設定がリセットされてしまう）ので廃止しました。
            base.TextAlign = HorizontalAlignment.Right;

            this.Enter += Event_UINumericUpDown2_Enter_;

            // 内部コントロールを取得します。
            _internalTextBox = (TextBox)base.Controls[1];

            _lastValue = Value;
        }

        /// <summary>
        /// フォーカスを取得した場合、テキストを全選択します。
        /// </summary>
        void Event_UINumericUpDown2_Enter_( object sender, EventArgs e )
        {
            _internalTextBox.SelectAll();
        }

        /// <summary>
        /// テキストが全選択されているか
        /// </summary>
        public bool IsSelectedAll()
        {
            return _internalTextBox.Text?.Length > 0 &&
                _internalTextBox.SelectionLength == _internalTextBox.Text.Length;
        }

        /// <summary>
        /// テキストを全選択する
        /// </summary>
        public void SelectAll()
        {
            _internalTextBox.SelectAll();
        }

        /// <summary>
        /// 値更新。
        /// </summary>
        protected bool UpdateValue
        {
            get{
                // this.Value の取得で OnValueChanged が呼ばれ、
                // _lastValue の値が変わる時がある
                var value = _lastValue;
                return this.Value != value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void SetValue(Decimal newValue)
        {
            try
            {
                UINumericUpDown.InvokeEvent = false;
                this.Value = newValue;
            }
            finally
            {
                UINumericUpDown.InvokeEvent = true;
            }
        }

        /// <summary>
        /// 小数点位置を調整します。
        /// </summary>
        public void AdjustDecimalPlace(int initialPlace)
        {
            int decimalPlaces = initialPlace;
            while ((Decimal.Parse(this.Value.ToString("F" + decimalPlaces)) != this.Value) && decimalPlaces < _MaxAdjustmentDecimalPlace)
            {
                decimalPlaces++;
            }

            this.DecimalPlaces = decimalPlaces;
        }

        /// <summary>
        ///
        /// </summary>
        public override Color BackColor
        {
            get
            {
                return _internalTextBox != null ?
                    _internalTextBox.BackColor : SystemColors.Window;
            }

            set
            {
                _internalTextBox.BackColor = value;

#if false
                using( Graphics gc = CreateGraphics())
                {
                    InvokePaint( this, new PaintEventArgs( gc, Bounds));
                }
#else
                Invalidate();
#endif
            }
        }

        /// <summary>
        /// 編集中の値を確定します。
        /// </summary>
        public void FixValue()
        {
            // 値更新
            bool changed = UpdateValue;

            this.UpdateEditText();

            // イベント発行
            if (changed)
            {
                OnValueChanged(EventArgs.Empty);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool IsInputKey(Keys keyData)
        {
            if ((keyData & Keys.Alt) != Keys.Alt)
            {
                Keys keys = keyData & Keys.KeyCode;
                if (keys == Keys.Return)
                {
                    return true;
                }
            }
            return base.IsInputKey(keyData);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            FixValue();

            base.OnLostFocus(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Return:
                {
                    // 値更新
                    bool changed = UpdateValue;

                    // 全文字を選択
                    this.Select(0, 10);

                    OnValueChanged( EventArgs.Empty );

                    // イベント発行
                    if (changed)
                    {
                        OnValueChanged(EventArgs.Empty);
                    }
                    break;
                }
                default:
                    base.OnKeyDown(e);
                    break;
            }
        }

        protected override void OnValueChanged( EventArgs e )
        {
            _lastValue = Value;
            if( CanRaiseEvents )
            {
                base.OnValueChanged( e );
            }
        }

        /// <summary>
        /// イベントを発生するかどうか
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get { return _invokeEvent & base.CanRaiseEvents; }
        }

        private void InitializeComponent()
        {
            ( (System.ComponentModel.ISupportInitialize)( this ) ).BeginInit();
            this.SuspendLayout();
            ( (System.ComponentModel.ISupportInitialize)( this ) ).EndInit();
            this.ResumeLayout( false );
        }
    }

    /// <summary>
    /// 連続的な値変更イベントハンドラデリゲート。
    /// </summary>
    public delegate void SequentialValueChangedEventHandler(object sender, SequentialValueChangedEventArgs e);

    /// <summary>
    /// 連続的な値変更イベントデータクラス。
    /// </summary>
    public sealed class SequentialValueChangedEventArgs : EventArgs
    {
        // 変更中
        private readonly bool _changing;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public SequentialValueChangedEventArgs(bool changing)
        {
            _changing = changing;
        }

        /// <summary>
        /// 変更中かどうか。
        /// </summary>
        public bool Changing
        {
            get { return _changing; }
        }
    }
}
