﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Diagnostics;
using System.Data;
using System.Windows.Forms;

namespace LayoutEditor.Controls
{
    using Forms.ToolWindows;
    using LECore.Structures;
    using LECore.Structures.Core;
    using src.Controls;
    using System.Collections.Generic;

    /// <summary>
    /// Fvec2EditUpDownPanel の概要の説明です。
    /// </summary>
    public class Fvec2EditUpDownPanel : System.Windows.Forms.UserControl
        , IAnimationMarkUsable
    {
        private LayoutEditor.Controls.UINumericUpDown _nudX;
        private LayoutEditor.Controls.UINumericUpDown _nudY;
        private src.Controls.BindAnimationLabel _lblX;
        private src.Controls.BindAnimationLabel _lblY;
        private src.Controls.AnimationMark _pnlMarkX;
        private src.Controls.AnimationMark _pnlMarkY;
        private UI.UIButton _btnLink;
        private int _initialDecimalPlaces;
        private Color _InitialForeColor;
        private IContainer components;
        private LayoutEditor.Controls.DockContentToolTip toolTip1;

        public event FVec2EditEventHandler OnFVec2Edit   = null;
        private bool _roundByStepValue = false;
        private bool _link = false;
        private bool _useAnimationMark = false;

        private Rectangle lblXBounds;
        private Rectangle tbxXBounds;
        private Rectangle lblYBounds;
        private Rectangle tbxYBounds;
        private Rectangle btnBounds;

        public Fvec2EditUpDownPanel()
        {
            InitializeComponent();

            lblXBounds = _lblX.Bounds;
            tbxXBounds = _nudX.Bounds;
            lblYBounds = _lblY.Bounds;
            tbxYBounds = _nudY.Bounds;
            btnBounds = _btnLink.Bounds;

            _initialDecimalPlaces = _nudX.DecimalPlaces;
            _InitialForeColor = _nudX.ForeColor;

            Link = false;

            toolTip1.SetToolTip(_btnLink, StringResMgr.Get("FVEC2EDIT_LINK_TIP"));

            // AnimationMarkの持つコンテキストメニューを関連付けます
            _lblX.BindAnimationMark(_pnlMarkX);
            _lblY.BindAnimationMark(_pnlMarkY);
        }

        public static bool InvokeEvent
        {
            get { return UINumericUpDown.InvokeEvent; }
            set { UINumericUpDown.InvokeEvent = value; }
        }

        public FVec2 FVec2Value
        {
            get{ return ConvertToFVec2_();}
            set
            {
                UINumericUpDown.InvokeEvent = false;
                SafeSetToFVec2_( _nudX, value.X );
                SafeSetToFVec2_( _nudY, value.Y );
                UINumericUpDown.InvokeEvent = true;
            }
        }

        /// <summary>
        /// ステップ値で丸めるか？
        /// </summary>
        public bool RoundByStepValue
        {
            get{ return _roundByStepValue;}
            set{ _roundByStepValue = value;}
        }

        /// <summary>
        /// 増加値
        /// </summary>
        public float Increment
        {
            get
            {
                Debug.Assert( _nudX.Increment == _nudY.Increment );
                return Decimal.ToSingle( _nudX.Increment );
            }
            set
            {
                _nudX.Increment = Convert.ToDecimal( value );
                _nudY.Increment = Convert.ToDecimal( value );
            }
        }

        /// <summary>
        ///
        /// </summary>
        public Color TextBoxBackColor
        {
            set
            {
                _nudX.BackColor = value;
                _nudY.BackColor = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int DecimalPlaces
        {
            get
            {
                return _nudX.DecimalPlaces;
            }
            set
            {
                _nudX.DecimalPlaces = value;
                _nudY.DecimalPlaces = value;
                _initialDecimalPlaces = value;
            }
        }

        private bool _vertical;

        [DefaultValue(0)]
        public bool Vertical
        {
            get
            {
                return _vertical;
            }
            set
            {
                if (_vertical != value)
                {
                    _vertical = value;
                    _alignControls();
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public int TextBoxWidth
        {
            get
            {
                return _nudX.Width;
            }
            set
            {
                if (_nudX.Width != value)
                {
                    _nudX.Width = value;
                    _alignControls();
                }

            }
        }

        private void _alignControls()
        {
            int diffX = Vertical ? lblXBounds.X - lblYBounds.X:TextBoxWidth - tbxXBounds.Width;
            int diffY = Vertical ? _nudX.Height + 2: 0;

            _lblY.Location = new Point(lblYBounds.X + diffX, lblYBounds.Y + diffY);
            _nudY.Location = new Point(tbxYBounds.X + diffX, tbxYBounds.Y + diffY);
            _nudY.Width = TextBoxWidth;
            _pnlMarkY.Location = new Point(_nudY.Location.X - (_pnlMarkY.Width + 2), _nudY.Location.Y);

            _btnLink.Location = new Point(_nudY.Right + btnBounds.Left - tbxYBounds.Right, btnBounds.Y);
        }

        /// <summary>
        ///
        /// </summary>
        public string LabelX
        {
            get { return _lblX.Text; }
            set
            {
                _lblX.Text = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string LabelY
        {
            get { return _lblY.Text; }
            set
            {
                _lblY.Text = value;
            }
        }

        /// <summary>
        /// X が有効か
        /// </summary>
        public bool XEnabled
        {
            get { return _nudX.Enabled; }
            set
            {
                _nudX.Enabled = value;
            }
        }

        /// <summary>
        /// Y が有効か
        /// </summary>
        public bool YEnabled
        {
            get { return _nudY.Enabled; }
            set
            {
                _nudY.Enabled = value;
            }
        }

        /// <summary>
        /// リンク が有効か
        /// </summary>
        public bool Link
        {
            get { return _btnLink.Visible; }
            set
            {
                _btnLink.Visible = value;
            }
        }

        /// <summary>
        /// アニメーションマークのリスト
        /// </summary>
        public AnimationMark[] AnimationMarkArray
        {
            get
            {
                List<AnimationMark> list = new List<AnimationMark>();
                list.Add(_pnlMarkX);
                list.Add(_pnlMarkY);

                return list.ToArray();
            }
        }

        FVec2 ConvertToFVec2_()
        {
            return new FVec2( Decimal.ToSingle( _nudX.Value ), Decimal.ToSingle( _nudY.Value ) );
        }

        void SafeSetToFVec2_( UINumericUpDown udCtrl, float fval )
        {
            Decimal   decVal = Convert.ToDecimal( fval );
            decVal = Math.Min( decVal, udCtrl.Maximum );
            decVal = Math.Max( decVal, udCtrl.Minimum );
            udCtrl.Value = decVal;

            // 小数点の調整
            udCtrl.AdjustDecimalPlace(_initialDecimalPlaces);
            udCtrl.ForeColor = (udCtrl.DecimalPlaces == _initialDecimalPlaces) ? _InitialForeColor : Color.RoyalBlue;
        }

        /// <summary>
        /// ViewManagerへのメッセージハンドラを設定します。
        /// </summary>
        public void SetViewManagerMessageHandler(Action<ViewManagerMessage> handler)
        {
            _pnlMarkX.SendMessageToViewManager = handler;
            _pnlMarkY.SendMessageToViewManager = handler;
        }

        /// <summary>
        /// コンテキストメニューを追加します。
        /// </summary>
        public void RegistContextMenu(string name, Image image, EventHandler handler)
        {
            _pnlMarkX.RegistContextMenu(name, image, handler);
            _pnlMarkY.RegistContextMenu(name, image, handler);
        }

        /// <summary>
        /// 操作対象のアトリビュートを設定します
        /// </summary>
        public void SetTargetAttribute(IAnmAttribute attr)
        {
            Debug.Assert(attr.NumSubAttribute == 2);

            _pnlMarkX.SetTargetAttribute(attr.FindSubAttributeByIdx(0));
            _pnlMarkY.SetTargetAttribute(attr.FindSubAttributeByIdx(1));
        }
        public void SetTargetAttribute(IAnmAttribute[] attrs)
        {
            _pnlMarkX.ClearTargetAttribute();
            _pnlMarkY.ClearTargetAttribute();

            foreach (IAnmAttribute attr in attrs)
            {
                Debug.Assert(attr.NumSubAttribute == 2);

                _pnlMarkX.AddTargetAttribute(attr.FindSubAttributeByIdx(0));
                _pnlMarkY.AddTargetAttribute(attr.FindSubAttributeByIdx(1));
            }
        }

        /// <summary>
        /// アニメーションマークの色を更新します
        /// </summary>
        public void UpdateMarkColor()
        {
            _pnlMarkX.UpdateMarkColor();
            _pnlMarkY.UpdateMarkColor();
        }

        /// <summary>
        /// コンテキストメニューを更新します
        /// </summary>
        public void UpdateContextMenu()
        {
            _pnlMarkX.UpdateContextMenu();
            _pnlMarkY.UpdateContextMenu();
        }

        /// <summary>
        /// 連動ボタンを更新します
        /// </summary>
        public void UpdateLinkButton(bool state)
        {
            _link = state;
            _btnLink.Image = state ? Properties.Resources.Link_On : Properties.Resources.Link_Off;
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if(components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        #region IAnimationMarkUsableメンバ

        /// <summary>
        /// アニメーションマークを利用するか
        /// </summary>
        public bool UseAnimationMark
        {
            get { return _useAnimationMark; }
            set
            {
                _useAnimationMark = value;
                _pnlMarkX.Visible = value;
                _pnlMarkY.Visible = value;
                _lblX.UseContextMenu = value;
                _lblY.UseContextMenu = value;
            }
        }

        /// <summary>
        /// アニメーションマークを更新します
        /// </summary>
        public void UpdateAnimationMark()
        {
            UpdateMarkColor();
            UpdateContextMenu();
        }

        #endregion

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            this._pnlMarkX = new LayoutEditor.src.Controls.AnimationMark();
            this._pnlMarkY = new LayoutEditor.src.Controls.AnimationMark();
            this._btnLink = new LayoutEditor.Controls.UI.UIButton();
            this._nudX = new LayoutEditor.Controls.UINumericUpDown();
            this._nudY = new LayoutEditor.Controls.UINumericUpDown();
            this.toolTip1 = new LayoutEditor.Controls.DockContentToolTip(this.components);
            this._lblX = new LayoutEditor.src.Controls.BindAnimationLabel();
            this._lblY = new LayoutEditor.src.Controls.BindAnimationLabel();
            ((System.ComponentModel.ISupportInitialize)(this._nudX)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this._nudY)).BeginInit();
            this.SuspendLayout();
            //
            // _pnlMarkX
            //
            this._pnlMarkX.Location = new System.Drawing.Point(21, 3);
            this._pnlMarkX.Name = "_pnlMarkX";
            this._pnlMarkX.Size = new System.Drawing.Size(5, 19);
            this._pnlMarkX.TabIndex = 0;
            this._pnlMarkX.Visible = false;
            //
            // _pnlMarkY
            //
            this._pnlMarkY.BackColor = System.Drawing.SystemColors.Control;
            this._pnlMarkY.Location = new System.Drawing.Point(110, 3);
            this._pnlMarkY.Name = "_pnlMarkY";
            this._pnlMarkY.Size = new System.Drawing.Size(5, 19);
            this._pnlMarkY.TabIndex = 0;
            this._pnlMarkY.Visible = false;
            //
            // _btnLink
            //
            this._btnLink.Image = global::LayoutEditor.Properties.Resources.Link_Off;
            this._btnLink.Location = new System.Drawing.Point(186, 4);
            this._btnLink.Name = "_btnLink";
            this._btnLink.Size = new System.Drawing.Size(18, 18);
            this._btnLink.TabIndex = 4;
            this._btnLink.Click += new System.EventHandler(this.Event_ClickedLinkButton);
            //
            // _nudX
            //
            this._nudX.DecimalPlaces = 2;
            this._nudX.Increment = new decimal(new int[] {
            1,
            0,
            0,
            131072});
            this._nudX.Location = new System.Drawing.Point(28, 3);
            this._nudX.Maximum = new decimal(new int[] {
            10000,
            0,
            0,
            0});
            this._nudX.Minimum = new decimal(new int[] {
            10000,
            0,
            0,
            -2147483648});
            this._nudX.Name = "_nudX";
            this._nudX.RightToLeft = System.Windows.Forms.RightToLeft.No;
            this._nudX.Size = new System.Drawing.Size(62, 19);
            this._nudX.TabIndex = 2;
            this._nudX.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this._nudX.ValueChanged += new System.EventHandler(this.Event_NudX_ValueChanged);
            //
            // _nudY
            //
            this._nudY.DecimalPlaces = 2;
            this._nudY.Increment = new decimal(new int[] {
            1,
            0,
            0,
            131072});
            this._nudY.Location = new System.Drawing.Point(117, 3);
            this._nudY.Maximum = new decimal(new int[] {
            10000,
            0,
            0,
            0});
            this._nudY.Minimum = new decimal(new int[] {
            10000,
            0,
            0,
            -2147483648});
            this._nudY.Name = "_nudY";
            this._nudY.Size = new System.Drawing.Size(62, 19);
            this._nudY.TabIndex = 3;
            this._nudY.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this._nudY.ValueChanged += new System.EventHandler(this.Event_NudY_ValueChanged);
            //
            // _lblX
            //
            this._lblX.Location = new System.Drawing.Point(2, 7);
            this._lblX.Name = "_lblX";
            this._lblX.Size = new System.Drawing.Size(19, 16);
            this._lblX.TabIndex = 0;
            this._lblX.Text = "X :";
            this._lblX.TextAlign = System.Drawing.ContentAlignment.TopRight;
            this._lblX.UseAnimationMark = false;
            this._lblX.UseContextMenu = false;
            //
            // _lblY
            //
            this._lblY.Location = new System.Drawing.Point(91, 7);
            this._lblY.Name = "_lblY";
            this._lblY.Size = new System.Drawing.Size(19, 16);
            this._lblY.TabIndex = 1;
            this._lblY.Text = "Y :";
            this._lblY.TextAlign = System.Drawing.ContentAlignment.TopRight;
            this._lblY.UseAnimationMark = false;
            this._lblY.UseContextMenu = false;
            //
            // Fvec2EditUpDownPanel
            //
            this.Controls.Add(this._pnlMarkX);
            this.Controls.Add(this._pnlMarkY);
            this.Controls.Add(this._btnLink);
            this.Controls.Add(this._nudX);
            this.Controls.Add(this._nudY);
            this.Controls.Add(this._lblX);
            this.Controls.Add(this._lblY);
            this.Name = "Fvec2EditUpDownPanel";
            this.Size = new System.Drawing.Size(209, 25);
            ((System.ComponentModel.ISupportInitialize)(this._nudX)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this._nudY)).EndInit();
            this.ResumeLayout(false);

        }
        #endregion

        private void Event_NudX_ValueChanged(object sender, System.EventArgs e)
        {
            if( OnFVec2Edit != null )
            {
                if (_link)
                {
                    bool isYSelected = _nudY.IsSelectedAll();

                    FVec2Value = new FVec2(_nudX.Value, _nudX.Value);

                    // X から Y にタブ移動したとき Y の変更により Y の全選択が取り消されないようにする
                    if (isYSelected)
                    {
                        _nudY.SelectAll();
                    }

                    OnFVec2Edit(this, new FVec2EditEventArgs("XY", FVec2Value));
                }
                else
                {
                    OnFVec2Edit(this, new FVec2EditEventArgs("X", FVec2Value));
                }
            }
        }

        private void Event_NudY_ValueChanged(object sender, System.EventArgs e)
        {
            if( OnFVec2Edit != null )
            {
                if (_link)
                {
                    bool isXSelected = _nudX.IsSelectedAll();

                    FVec2Value = new FVec2(_nudY.Value, _nudY.Value);

                    // Y から X にタブ移動したとき X の変更により X の全選択が取り消されないようにする
                    if (isXSelected)
                    {
                        _nudX.SelectAll();
                    }

                    OnFVec2Edit(this, new FVec2EditEventArgs("XY", FVec2Value));
                }
                else
                {
                    OnFVec2Edit(this, new FVec2EditEventArgs("Y", FVec2Value));
                }
            }
        }

        private void Event_ClickedLinkButton(object sender, EventArgs e)
        {
            UpdateLinkButton(!_link);
        }
    }

    #region FVec2EditEvent
    public delegate void FVec2EditEventHandler( object sender, FVec2EditEventArgs args );

    /// <summary>
    /// イベントパラメータ型
    /// </summary>
    public sealed class FVec2EditEventArgs : EventArgs
    {
        private readonly FVec2      _val;
        private string _name = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public FVec2EditEventArgs( string name, FVec2 val )
        {
            _name = name;
            _val = val;
        }

        /// <summary>
        ///
        /// </summary>
        public FVec2 Value
        {
            get { return _val; }
        }

        /// <summary>
        ///
        /// </summary>
        public string Name
        {
            get { return _name; }
        }
    }
    #endregion FVec2EditEvent
}
