﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;

namespace LayoutEditor.Controls
{
    using Forms.ToolWindows;
    using LECore.Structures;
    using LECore.Structures.Core;
    using src.Controls;
    using System.Collections.Generic;

    /// <summary>
    /// カラー編集パネルクラス。
    /// </summary>
    [ToolboxBitmap(typeof(UserControl))]
    public sealed class ColorEditPanel : UserControl
        , IAnimationMarkUsable
    {
        //--------------------------------------------------
        #region デザイナ変数

        private LayoutEditor.Controls.ColorButton btnColor;
        private LayoutEditor.Controls.FloatTextBox tbxR;
        private LayoutEditor.Controls.FloatTextBox tbxG;
        private LayoutEditor.Controls.FloatTextBox tbxB;
        private LayoutEditor.Controls.FloatTextBox tbxA;
        private System.Windows.Forms.Panel pnlRGBA;
        private src.Controls.BindAnimationLabel lblR;
        private src.Controls.BindAnimationLabel lblG;
        private src.Controls.BindAnimationLabel lblB;
        private src.Controls.BindAnimationLabel lblA;
        private FlowLayoutPanel _flowLayoutPanel;
        private src.Controls.AnimationMark pnlMarkR;
        private src.Controls.AnimationMark pnlMarkB;
        private src.Controls.AnimationMark pnlMarkG;
        private src.Controls.AnimationMark pnlMarkA;
        private IContainer components;
        #endregion

        //--------------------------------------------------
        #region デザイナコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            this.lblR = new LayoutEditor.src.Controls.BindAnimationLabel();
            this.lblG = new LayoutEditor.src.Controls.BindAnimationLabel();
            this.lblB = new LayoutEditor.src.Controls.BindAnimationLabel();
            this.lblA = new LayoutEditor.src.Controls.BindAnimationLabel();
            this.pnlRGBA = new System.Windows.Forms.Panel();
            this.pnlMarkR = new LayoutEditor.src.Controls.AnimationMark();
            this.pnlMarkA = new LayoutEditor.src.Controls.AnimationMark();
            this.pnlMarkB = new LayoutEditor.src.Controls.AnimationMark();
            this.pnlMarkG = new LayoutEditor.src.Controls.AnimationMark();
            this.tbxR = new LayoutEditor.Controls.FloatTextBox();
            this.tbxG = new LayoutEditor.Controls.FloatTextBox();
            this.tbxB = new LayoutEditor.Controls.FloatTextBox();
            this.tbxA = new LayoutEditor.Controls.FloatTextBox();
            this._flowLayoutPanel = new System.Windows.Forms.FlowLayoutPanel();
            this.btnColor = new LayoutEditor.Controls.ColorButton();
            this.pnlRGBA.SuspendLayout();
            this._flowLayoutPanel.SuspendLayout();
            this.SuspendLayout();
            //
            // lblR
            //
            this.lblR.Location = new System.Drawing.Point(14, -2);
            this.lblR.Name = "lblR";
            this.lblR.Size = new System.Drawing.Size(20, 12);
            this.lblR.TabIndex = 0;
            this.lblR.Text = "R";
            this.lblR.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            //
            // lblG
            //
            this.lblG.Location = new System.Drawing.Point(52, -2);
            this.lblG.Name = "lblG";
            this.lblG.Size = new System.Drawing.Size(20, 12);
            this.lblG.TabIndex = 2;
            this.lblG.Text = "G";
            this.lblG.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            //
            // lblB
            //
            this.lblB.Location = new System.Drawing.Point(91, -2);
            this.lblB.Name = "lblB";
            this.lblB.Size = new System.Drawing.Size(20, 12);
            this.lblB.TabIndex = 4;
            this.lblB.Text = "B";
            this.lblB.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            //
            // lblA
            //
            this.lblA.Location = new System.Drawing.Point(131, -2);
            this.lblA.Name = "lblA";
            this.lblA.Size = new System.Drawing.Size(20, 12);
            this.lblA.TabIndex = 6;
            this.lblA.Text = "A";
            this.lblA.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
            //
            // pnlRGBA
            //
            this.pnlRGBA.Controls.Add(this.pnlMarkR);
            this.pnlRGBA.Controls.Add(this.pnlMarkA);
            this.pnlRGBA.Controls.Add(this.pnlMarkB);
            this.pnlRGBA.Controls.Add(this.pnlMarkG);
            this.pnlRGBA.Controls.Add(this.lblR);
            this.pnlRGBA.Controls.Add(this.tbxR);
            this.pnlRGBA.Controls.Add(this.lblG);
            this.pnlRGBA.Controls.Add(this.tbxG);
            this.pnlRGBA.Controls.Add(this.lblB);
            this.pnlRGBA.Controls.Add(this.tbxB);
            this.pnlRGBA.Controls.Add(this.lblA);
            this.pnlRGBA.Controls.Add(this.tbxA);
            this.pnlRGBA.Location = new System.Drawing.Point(74, 0);
            this.pnlRGBA.Margin = new System.Windows.Forms.Padding(0);
            this.pnlRGBA.Name = "pnlRGBA";
            this.pnlRGBA.Size = new System.Drawing.Size(159, 32);
            this.pnlRGBA.TabIndex = 1;
            //
            // pnlMarkR
            //
            this.pnlMarkR.Location = new System.Drawing.Point(1, 11);
            this.pnlMarkR.Name = "pnlMarkR";
            this.pnlMarkR.Size = new System.Drawing.Size(5, 19);
            this.pnlMarkR.TabIndex = 0;
            this.pnlMarkR.Visible = false;
            //
            // pnlMarkA
            //
            this.pnlMarkA.Location = new System.Drawing.Point(117, 11);
            this.pnlMarkA.Name = "pnlMarkA";
            this.pnlMarkA.Size = new System.Drawing.Size(5, 19);
            this.pnlMarkA.TabIndex = 0;
            this.pnlMarkA.Visible = false;
            //
            // pnlMarkB
            //
            this.pnlMarkB.Location = new System.Drawing.Point(78, 11);
            this.pnlMarkB.Name = "pnlMarkB";
            this.pnlMarkB.Size = new System.Drawing.Size(5, 19);
            this.pnlMarkB.TabIndex = 0;
            this.pnlMarkB.Visible = false;
            //
            // pnlMarkG
            //
            this.pnlMarkG.Location = new System.Drawing.Point(39, 11);
            this.pnlMarkG.Name = "pnlMarkG";
            this.pnlMarkG.Size = new System.Drawing.Size(5, 19);
            this.pnlMarkG.TabIndex = 0;
            this.pnlMarkG.Visible = false;
            //
            // tbxR
            //
            this.tbxR.Format = "f0";
            this.tbxR.Location = new System.Drawing.Point(7, 10);
            this.tbxR.MaxValue = 255F;
            this.tbxR.MinValue = 0F;
            this.tbxR.Name = "tbxR";
            this.tbxR.Size = new System.Drawing.Size(30, 21);
            this.tbxR.TabIndex = 1;
            this.tbxR.Text = "255";
            this.tbxR.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.tbxR.Value = 255F;
            //
            // tbxG
            //
            this.tbxG.Format = "f0";
            this.tbxG.Location = new System.Drawing.Point(46, 10);
            this.tbxG.MaxValue = 255F;
            this.tbxG.MinValue = 0F;
            this.tbxG.Name = "tbxG";
            this.tbxG.Size = new System.Drawing.Size(30, 21);
            this.tbxG.TabIndex = 3;
            this.tbxG.Text = "255";
            this.tbxG.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.tbxG.Value = 255F;
            //
            // tbxB
            //
            this.tbxB.Format = "f0";
            this.tbxB.Location = new System.Drawing.Point(85, 10);
            this.tbxB.MaxValue = 255F;
            this.tbxB.MinValue = 0F;
            this.tbxB.Name = "tbxB";
            this.tbxB.Size = new System.Drawing.Size(30, 21);
            this.tbxB.TabIndex = 5;
            this.tbxB.Text = "255";
            this.tbxB.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.tbxB.Value = 255F;
            //
            // tbxA
            //
            this.tbxA.Format = "f0";
            this.tbxA.Location = new System.Drawing.Point(124, 10);
            this.tbxA.MaxValue = 255F;
            this.tbxA.MinValue = 0F;
            this.tbxA.Name = "tbxA";
            this.tbxA.Size = new System.Drawing.Size(30, 21);
            this.tbxA.TabIndex = 7;
            this.tbxA.Text = "255";
            this.tbxA.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
            this.tbxA.Value = 255F;
            //
            // _flowLayoutPanel
            //
            this._flowLayoutPanel.Controls.Add(this.btnColor);
            this._flowLayoutPanel.Controls.Add(this.pnlRGBA);
            this._flowLayoutPanel.Dock = System.Windows.Forms.DockStyle.Fill;
            this._flowLayoutPanel.Location = new System.Drawing.Point(0, 0);
            this._flowLayoutPanel.Name = "_flowLayoutPanel";
            this._flowLayoutPanel.Size = new System.Drawing.Size(236, 32);
            this._flowLayoutPanel.TabIndex = 2;
            //
            // btnColor
            //
            this.btnColor.AllowDrop = true;
            this.btnColor.EnableAlpha = true;
            this.btnColor.Location = new System.Drawing.Point(3, 3);
            this.btnColor.Name = "btnColor";
            this.btnColor.Size = new System.Drawing.Size(68, 28);
            this.btnColor.TabIndex = 0;
            this.btnColor.DragDrop += new System.Windows.Forms.DragEventHandler(this.Event_BtnColor_DragDrop);
            this.btnColor.DragEnter += new System.Windows.Forms.DragEventHandler(this.Event_BtnColor_DragEnter);
            this.btnColor.DragOver += new System.Windows.Forms.DragEventHandler(this.Event_BtnColor_DragOver);
            this.btnColor.MouseDown += new System.Windows.Forms.MouseEventHandler(this.Event_BtnColor_MouseDown);
            this.btnColor.MouseMove += new System.Windows.Forms.MouseEventHandler(this.Event_BtnColor_MouseMove);
            //
            // ColorEditPanel
            //
            this.AllowDrop = true;
            this.Controls.Add(this._flowLayoutPanel);
            this.Font = new System.Drawing.Font("Tahoma", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.Name = "ColorEditPanel";
            this.Size = new System.Drawing.Size(236, 32);
            this.pnlRGBA.ResumeLayout(false);
            this.pnlRGBA.PerformLayout();
            this._flowLayoutPanel.ResumeLayout(false);
            this.ResumeLayout(false);

        }
        #endregion

        //--------------------------------------------------
        #region フィールド
        /// <summary>
        /// ドラッグ・ドロップ処理のためのマウスダウン位置。
        /// </summary>
        Point _posMouseDownForDragDrop;
        /// <summary>
        /// ドラッグ・ドロップ時の実行アクション。
        /// </summary>
        DragDropAction _dragDropAction;
        /// <summary>
        /// カラーピッカーアダプタ
        /// </summary>
        private ColorPickerAdapter _adapter = null;
        /// <summary>
        /// マーク表示
        /// </summary>
        private bool _useAnimationMark = false;
        #endregion

        //--------------------------------------------------
        #region イベント
        /// <summary>カラー編集イベント</summary>
        public event ColorEditEventHandler ColorEdit = null;
        #endregion イベント

        //--------------------------------------------------
        #region enum
        [Flags]
        public enum DragDropAction
        {
            Copy_None = 0,
            Copy_RGB = 1,
            Copy_Alpha = 2,
            Copy_Both = 3,
        }
        #endregion enum

        //--------------------------------------------------
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorEditPanel()
        {
            InitializeComponent();

            // カラーピッカーアダプタ作成
            _adapter = new ColorPickerAdapter(btnColor);
            _adapter.AddControl(tbxR);
            _adapter.AddControl(tbxG);
            _adapter.AddControl(tbxB);
            _adapter.AddControl(tbxA);
            _adapter.ColorEdit += new ColorEditEventHandler(Event_ColorPicker_ColorEdit);

            // イベント登録
            tbxR.ValueChanged += new EventHandler(Event_TbxRGBA_ValueChanged);
            tbxG.ValueChanged += new EventHandler(Event_TbxRGBA_ValueChanged);
            tbxB.ValueChanged += new EventHandler(Event_TbxRGBA_ValueChanged);
            tbxA.ValueChanged += new EventHandler(Event_TbxRGBA_ValueChanged);

            // AnimationMarkの持つコンテキストメニューを関連付けます
            lblR.BindAnimationMark(pnlMarkR);
            lblG.BindAnimationMark(pnlMarkG);
            lblB.BindAnimationMark(pnlMarkB);
            lblA.BindAnimationMark(pnlMarkA);
        }

        /// <summary>
        /// カラー。
        /// </summary>
        [Description("カラー")]
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public FloatColor Color
        {
            get { return btnColor.Color; }
            set
            {
                if (btnColor.Color != value)
                {
                    btnColor.Color = value;

                    UpdateColorModeUIState(value);
                }
            }
        }

        /// <summary>
        /// カラー。
        /// </summary>
        [DefaultValue(typeof(bool), "true")]
        [Description("アルファが有効か")]
        public bool IsAlphaEnabled
        {
            get { return btnColor.EnableAlpha; }
            set
            {
                if (btnColor.EnableAlpha != value)
                {
                    btnColor.EnableAlpha = value;
                    tbxA.Visible = value;
                    lblA.Visible = value;
                    pnlMarkA.Visible = value & UseAnimationMark;
                }
            }
        }

        /// <summary>
        /// カラーボタン。
        /// </summary>
        [DefaultValue(typeof(bool), "true")]
        [Description("カラーボタン")]
        public bool ColorButtonVisible
        {
            get { return btnColor.Visible; }
            set
            {
                if (btnColor.Visible != value)
                {
                    btnColor.Visible = value;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public Color TextBoxBackColor
        {
            set
            {
                tbxR.BackColor = value;
                tbxG.BackColor = value;
                tbxB.BackColor = value;
                tbxA.BackColor = value;
            }
        }

        /// <summary>
        /// アニメーションマークのリスト
        /// </summary>
        [Description("アニメーションマークのリスト")]
        public AnimationMark[] AnimationMarkArray
        {
            get
            {
                List<AnimationMark> list = new List<AnimationMark>();
                list.Add(pnlMarkR);
                list.Add(pnlMarkG);
                list.Add(pnlMarkB);
                if (IsAlphaEnabled)
                {
                    list.Add(pnlMarkA);
                }

                return list.ToArray();
            }
        }

        #region IAnimationMarkUsableメンバ

        /// <summary>
        /// アニメーションマークを表示するか。
        /// </summary>
        [Description("アニメーションマークを表示するか")]
        public bool UseAnimationMark
        {
            get { return _useAnimationMark; }
            set
            {
                _useAnimationMark = value;
                pnlMarkR.Visible = value;
                pnlMarkG.Visible = value;
                pnlMarkB.Visible = value;
                pnlMarkA.Visible = value & IsAlphaEnabled;
                lblR.UseContextMenu = value;
                lblG.UseContextMenu = value;
                lblB.UseContextMenu = value;
                lblA.UseContextMenu = value;
            }
        }

        /// <summary>
        /// アニメーションマークを更新します
        /// </summary>
        public void UpdateAnimationMark()
        {
            UpdateMarkColor();
            UpdateContextMenu();
        }

        #endregion

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        //--------------------------------------------------
        #region イベントハンドラ
        /// <summary>
        /// カラー編集ハンドラ。
        /// </summary>
        private void OnColorEdit(ColorEditEventArgs e)
        {
            if (this.ColorEdit != null)
            {
                this.ColorEdit(this, e);
            }
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_TbxRGBA_ValueChanged(object sender, EventArgs e)
        {
            ColorEditEventArgs.Modified modified = ColorEditEventArgs.Modified.None;

            if( sender == tbxR ) { modified |= ColorEditEventArgs.Modified.R; }
            if( sender == tbxG ) { modified |= ColorEditEventArgs.Modified.G; }
            if( sender == tbxB ) { modified |= ColorEditEventArgs.Modified.B; }
            if( sender == tbxA ) { modified |= ColorEditEventArgs.Modified.A; }


            if (btnColor.Color.ByteConvert)
            {
                Color color = System.Drawing.Color.FromArgb((int)tbxA.Value, (int)tbxR.Value, (int)tbxG.Value, (int)tbxB.Value);
                float   r = ((modified & ColorEditEventArgs.Modified.R) != 0) ? tbxR.Value / 255.0f : btnColor.Color.R;
                float   g = ((modified & ColorEditEventArgs.Modified.G) != 0) ? tbxG.Value / 255.0f : btnColor.Color.G;
                float   b = ((modified & ColorEditEventArgs.Modified.B) != 0) ? tbxB.Value / 255.0f : btnColor.Color.B;
                float   a = ((modified & ColorEditEventArgs.Modified.A) != 0) ? tbxA.Value / 255.0f : btnColor.Color.A;

                FloatColor newColor = new FloatColor(r, g, b, a);
                newColor.ByteConvert = true;
                btnColor.Color = newColor;
            }
            else
            {
                btnColor.Color = new FloatColor(tbxR.Value, tbxG.Value, tbxB.Value, tbxA.Value);
            }

            // イベント発行
            OnColorEdit( new ColorEditEventArgs( btnColor.Color, true, modified));
            // アダプタに通知
            _adapter.NotifyUpdate();
        }

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_ColorPicker_ColorEdit(object sender, ColorEditEventArgs e)
        {
            if (e.EditFixed)
            {
                // 自分のカラー変更
                this.Color = e.Color;

                // イベント発行
                OnColorEdit(e);
            }
        }

        /// <summary>
        /// カラーの表示モードに基づいて UI の状態を更新。
        /// </summary>
        private void UpdateColorModeUIState(FloatColor color)
        {
            if (color.ByteConvert)
            {
                btnColor.Enabled = true;
                Color systemColor = color.ToSystemColor();
                tbxR.SetValue(systemColor.R);
                tbxR.Format = "f0";
                tbxG.SetValue(systemColor.G);
                tbxG.Format = "f0";
                tbxB.SetValue(systemColor.B);
                tbxB.Format = "f0";
                tbxA.SetValue(systemColor.A);
                tbxA.Format = "f0";

                tbxR.MaxValue = 255.0f;
                tbxG.MaxValue = 255.0f;
                tbxB.MaxValue = 255.0f;
                tbxA.MaxValue = 255.0f;
            }
            else
            {
                btnColor.Enabled = false;
                tbxR.SetValue(color.R);
                tbxR.Format = "f2";
                tbxG.SetValue(color.G);
                tbxG.Format = "f2";
                tbxB.SetValue(color.B);
                tbxB.Format = "f2";
                tbxA.SetValue(color.A);
                tbxA.Format = "f2";

                tbxR.MaxValue = float.MaxValue;
                tbxG.MaxValue = float.MaxValue;
                tbxB.MaxValue = float.MaxValue;
                tbxA.MaxValue = float.MaxValue;
            }
        }

        //-----------------------------------------------------
        #region ドラッグ・ドロップ関連
        /// <summary>
        ///
        /// </summary>
        private void ColorEditPanel_DragEnter( object sender, DragEventArgs e )
        {
            if( ColorData.CheckDropItemHasColorData( e ) )
            {
                e.Effect = DragDropEffects.Copy;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_BtnColor_MouseDown( object sender, MouseEventArgs e )
        {
            _posMouseDownForDragDrop = new Point( e.X, e.Y );
            _dragDropAction = _selectDragDropAction( e.X );
        }

        /// <summary>
        ///
        /// </summary>
    private void Event_BtnColor_MouseMove( object sender, MouseEventArgs e )
        {
            // ボタン領域外で、ドラッグされたら...
            //
            // ボタン領域内で、DoDragDrop()を実行すると、
            // ボタンが押下状態から正しく復帰しない。
      //if( btnColor.Bounds.Width < e.X || btnColor.Bounds.Height < e.Y )
      if( e.X < 0 || e.Y < 0 ||
          btnColor.Bounds.Width < e.X || btnColor.Bounds.Height < e.Y )
            {
                if( ( e.Button & MouseButtons.Left ) == 0 )
                {
                    _posMouseDownForDragDrop = new Point( e.X, e.Y );
                }
                else
                {
                    // ドラックが検知されたら...
                    if( Math.Abs( e.X - _posMouseDownForDragDrop.X ) > SystemInformation.DragSize.Width ||
              Math.Abs( e.Y - _posMouseDownForDragDrop.Y ) > SystemInformation.DragSize.Height )
                    {
                        // ドラッグを開始する。
                        btnColor.DoDragDrop(new Tuple<ColorData, DragDropAction>(new ColorData(this.Color, ""), _dragDropAction), DragDropEffects.Copy);
                    }
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_BtnColor_DragDrop( object sender, DragEventArgs e )
        {
            Tuple<ColorData, DragDropAction> data;
            if ( ColorData.TryToGetColorDataFromDropItem( e, out data ) )
            {
                bool byteConvert = this.Color.ByteConvert;
                FloatColor color = getCopyColor( this.Color, data.Item1.Color, data.Item2, IsAlphaEnabled );
                color.ByteConvert = byteConvert;
                this.Color = color;
                OnColorEdit( new ColorEditEventArgs( color, true ) );
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_BtnColor_DragEnter( object sender, DragEventArgs e )
        {
            // DragEnterでアクションをクリアしておき、
            // DragOverでマウスカーソルアイコンの変更を行う
            _dragDropAction = 0;
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_BtnColor_DragOver(object sender, DragEventArgs e)
        {
            Tuple<ColorData, DragDropAction> data;
            if (ColorData.TryToGetColorDataFromDropItem(e, out data))
            {
                // ActionがBothの場合はコピー可能
                if (data.Item2 == DragDropAction.Copy_Both)
                {
                    e.Effect = e.AllowedEffect;
                    return;
                }

                // アクションがBoth以外の場合は座標位置で判定
                var clientPoint = this.btnColor.PointToClient(new Point(e.X, e.Y)); // btnColor基準に座標変換
                var action = _selectDragDropAction(clientPoint.X);
                if (_dragDropAction != action)
                {
                    _dragDropAction = action;

                    if (ColorData.CheckDropItemHasColorData(e, action))
                    {
                        e.Effect = e.AllowedEffect;
                    }
                    else
                    {
                        e.Effect = DragDropEffects.None;
                    }
                }
            }
        }

        #endregion ドラッグ・ドロップ関連


        #endregion イベントハンドラ

        #region アニメーションマーク関連

        /// <summary>
        /// ViewManagerへのメッセージハンドラを設定します。
        /// </summary>
        public void SetViewManagerMessageHandler(Action<ViewManagerMessage> handler)
        {
            pnlMarkR.SendMessageToViewManager = handler;
            pnlMarkG.SendMessageToViewManager = handler;
            pnlMarkB.SendMessageToViewManager = handler;
            pnlMarkA.SendMessageToViewManager = handler;
        }

        /// <summary>
        /// コンテキストメニューを追加します。
        /// </summary>
        public void RegistContextMenu(string name, Image image, EventHandler handler)
        {
            pnlMarkR.RegistContextMenu(name, image, handler);
            pnlMarkG.RegistContextMenu(name, image, handler);
            pnlMarkB.RegistContextMenu(name, image, handler);
            pnlMarkA.RegistContextMenu(name, image, handler);
        }

        /// <summary>
        /// 操作対象のアトリビュートを設定します
        /// </summary>
        public void SetTargetAttribute(IAnmAttribute attr)
        {
            Debug.Assert(attr.NumSubAttribute == 4);

            pnlMarkR.SetTargetAttribute(attr.FindSubAttributeByIdx(0));
            pnlMarkG.SetTargetAttribute(attr.FindSubAttributeByIdx(1));
            pnlMarkB.SetTargetAttribute(attr.FindSubAttributeByIdx(2));
            pnlMarkA.SetTargetAttribute(attr.FindSubAttributeByIdx(3));
        }
        public void SetTargetAttribute(IAnmAttribute[] attrs)
        {
            pnlMarkR.ClearTargetAttribute();
            pnlMarkG.ClearTargetAttribute();
            pnlMarkB.ClearTargetAttribute();
            pnlMarkA.ClearTargetAttribute();

            foreach (IAnmAttribute attr in attrs)
            {
                Debug.Assert(attr.NumSubAttribute == 4);

                pnlMarkR.AddTargetAttribute(attr.FindSubAttributeByIdx(0));
                pnlMarkG.AddTargetAttribute(attr.FindSubAttributeByIdx(1));
                pnlMarkB.AddTargetAttribute(attr.FindSubAttributeByIdx(2));
                pnlMarkA.AddTargetAttribute(attr.FindSubAttributeByIdx(3));
            }
        }

        /// <summary>
        /// アニメーションマークの色を更新します
        /// </summary>
        public void UpdateMarkColor()
        {
            pnlMarkR.UpdateMarkColor();
            pnlMarkG.UpdateMarkColor();
            pnlMarkB.UpdateMarkColor();
            pnlMarkA.UpdateMarkColor();
        }

        /// <summary>
        /// コンテキストメニューを更新します
        /// </summary>
        public void UpdateContextMenu()
        {
            pnlMarkR.UpdateContextMenu();
            pnlMarkG.UpdateContextMenu();
            pnlMarkB.UpdateContextMenu();
            pnlMarkA.UpdateContextMenu();
        }

        #endregion

        //--------------------------------------------------
        #region ローカル関数
        /// <summary>
        /// ドラッグドロップ時のアクションを決める。
        /// </summary>
        private DragDropAction _selectDragDropAction( int xPos )
        {
            if (!IsAlphaEnabled || xPos <= this.btnColor.Width / 2)
            {
                return DragDropAction.Copy_RGB;
            }
            else
            {
                return DragDropAction.Copy_Alpha;
            }
        }

        /// <summary>
        /// アクションに応じてコピーする色を作成する。
        /// </summary>
        public static FloatColor getCopyColor(FloatColor src, FloatColor target, DragDropAction action, bool alpha = true)
        {
            FloatColor color;

            if (action == DragDropAction.Copy_Both)
            {
                color = new FloatColor(
                    target.R,
                    target.G,
                    target.B,
                    alpha ? target.A : src.A
                );
            }
            else if (action == DragDropAction.Copy_RGB)
            {
                color = new FloatColor(
                    target.R,
                    target.G,
                    target.B,
                    src.A
                );
            }
            else if (action == DragDropAction.Copy_Alpha)
            {
                color = new FloatColor(
                    src.R,
                    src.G,
                    src.B,
                    target.A
                );
            }
            else
            {
                color = new FloatColor(
                    src.R,
                    src.G,
                    src.B,
                    src.A
                );
            }

            return color;
        }

        #endregion ローカル関数


    }

    #region ColorEditEvent
    /// <summary>
    /// カラー編集イベントハンドラデリゲート。
    /// </summary>
    public delegate void ColorEditEventHandler(object sender, ColorEditEventArgs e);

    /// <summary>
    /// カラー編集イベントデータクラス。
    /// </summary>
    public sealed class ColorEditEventArgs : EventArgs
    {
        // カラー
        private readonly FloatColor _color;
        // 編集確定フラグ
        private readonly bool _editFixed;

        private Modified _modifiedFlag = Modified.All;

        //
        [Flags]
        public enum Modified
        {
            None = 0x00000000,
            R    = 0x00000001,
            G    = 0x00000002,
            B    = 0x00000004,
            A    = 0x00000008,
            All  = 0x0000000F,
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorEditEventArgs( FloatColor color, bool editFixed)
        {
            _color     = color;
            _editFixed = editFixed;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ColorEditEventArgs( FloatColor color, bool editFixed, Modified modifiedFlag)
        {
            _color     = color;
            _editFixed = editFixed;
            _modifiedFlag = modifiedFlag;
        }

        /// <summary>
        /// カラー。
        /// </summary>
        public FloatColor Color
        {
            get { return _color; }
        }

        /// <summary>
        /// 編集確定。
        /// </summary>
        public bool EditFixed
        {
            get { return _editFixed; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsModifiedR { get { return ( this._modifiedFlag & Modified.R ) != 0; }}
        public bool IsModifiedG { get { return ( this._modifiedFlag & Modified.G ) != 0; }}
        public bool IsModifiedB { get { return ( this._modifiedFlag & Modified.B ) != 0; }}
        public bool IsModifiedA { get { return ( this._modifiedFlag & Modified.A ) != 0; }}
    }
    #endregion
}
