﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.Drawing.Drawing2D;

using System.ComponentModel;
using System.Diagnostics;


namespace LayoutEditor.Controls
{
    using MathUtil = LECore.Util.MathUtil;

    /// <summary>
    /// 縮小可能パネル
    ///
    /// 現在は、水平方向分割(Orientation == Horizontal)
    /// での利用しか想定されていません。
    /// </summary>
    public class CollapsiblePanel :
        UISplitContainer,
        ILESerializableControl
    {
        /// <summary>
        ///
        /// </summary>
        Size			_sizeBeforeCollapse = Size.Empty;
        string _titleString = string.Empty;
        Bitmap _bmpExpanded = null;
        Bitmap _bmpCollapsed = null;
        Label _dummyLabel = null;

        /// <summary>
        /// Collapsed プロパティの値が変更された場合に発生します。
        /// </summary>
        public event EventHandler CollapsedChanged;

        //----------------------------------------------------
        #region プロパティ
        /// <summary>
        /// パネル１サイズ
        /// </summary>
        public Size Panel1Size
        {
            set
            {
                if( Orientation == Orientation.Horizontal )
                {
                    SplitterDistance = value.Height;
                }
                else
                {
                    SplitterDistance = value.Width;
                }
                Panel1.Size = value;
            }
            get { return Panel1.Size; }
        }

        /// <summary>
        /// タイトル文字列
        /// Panel1に描画されます。
        /// </summary>
        [Localizable(true)]
        public string TitleString
        {
            set { _titleString = value; }
            get { return _titleString; }
        }

        /// <summary>
        /// エラー表示に使う、ダミーラベル
        /// </summary>
        public Control HeaderContorol
        {
            get { return _dummyLabel; }
        }

        /// <summary>
        /// Panel2が縮小しているかどうか取得、設定します。
        /// </summary>
        bool Collapsed
        {
            set
            {
                if( !Collapsed )
                {
                    _sizeBeforeCollapse = this.Size;
                    this.Size = Panel1.Size;
                    this.Panel2Collapsed = true;
                }
                else
                {
                    this.Size = new Size(this.Width, _sizeBeforeCollapse.Height);
                    this.Panel2Collapsed = false;
                }

                if (this.CollapsedChanged != null)
                {
                    this.CollapsedChanged(this, new EventArgs());
                }
            }

            get
            {
                return this.Panel2Collapsed;
            }
        }
        #endregion プロパティ

        /// <summary>
        /// コンポーネントの初期化
        /// </summary>
        private void InitializeComponent()
        {
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager( typeof( CollapsiblePanel ) );
            this.SuspendLayout();
            //
            // CollapsiblePanel
            //
            this.FixedPanel = System.Windows.Forms.FixedPanel.Panel1;
            resources.ApplyResources( this, "$this" );
            //
            //
            //
            this.Panel1.BackColor = System.Drawing.SystemColors.ButtonFace;
            this.Panel1.Click += new System.EventHandler( this.CollapsiblePanel_Panel1_Click );
            this.Panel1.Paint += new System.Windows.Forms.PaintEventHandler( this.CollapsiblePanel_Panel1_Paint );
            this.ResumeLayout( false );

        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CollapsiblePanel()
        {
            InitializeComponent();

            _dummyLabel = new Label();
            this.Panel1.Controls.Add(_dummyLabel);
            _dummyLabel.Location = new Point(16, 0);
            _dummyLabel.Size = new Size(1, this.Panel1.Height);
            _dummyLabel.BackColor = Color.Transparent;

            _bmpExpanded = ImageResMgr.GetManifestResourceBitmap( "Expanded.bmp" );
            _bmpCollapsed = ImageResMgr.GetManifestResourceBitmap( "Collapsed.bmp" );

            _bmpExpanded.MakeTransparent( Color.FromArgb( 255, 0, 255 ) );
            _bmpCollapsed.MakeTransparent( Color.FromArgb( 255, 0, 255 ) );
        }


        //----------------------------------------------------
        #region イベントハンドラ
        /// <summary>
        /// パネル１クリック
        /// </summary>
        private void CollapsiblePanel_Panel1_Click( object sender, EventArgs e )
        {
            Collapsed = !Collapsed;
            this.Panel1.Invalidate();
        }

        /// <summary>
        /// panel1描画ハンドラ
        /// タイトル文字列を描画します。
        /// </summary>
        private void CollapsiblePanel_Panel1_Paint( object sender, PaintEventArgs e )
        {

            Bitmap bmpIcon = ( Collapsed ) ? _bmpCollapsed : _bmpExpanded;

            e.Graphics.DrawImage( bmpIcon, new Point( 0, ( this.Panel1Size.Height - bmpIcon.Size.Height ) / 2 ) );

            PointF vDiff = new PointF( bmpIcon.Size.Width * 2.0f, ( this.Panel1Size.Height - Font.Size ) * 0.5f );
            e.Graphics.DrawString( _titleString, this.Font, SystemBrushes.ControlText, vDiff );
        }
        #endregion イベントハンドラ


        //----------------------------------------------------
        #region ILESerializableControl メンバ


        /// <summary>
        /// 状態を読み込みます。
        /// 明示的にインターフェースを実装し、
        /// 基底クラスの処理をオーバーライドします。
        /// </summary>
        void ILESerializableControl.LoadState( LEControlUserDataChunk[] dataSet )
        {
            // 開閉状態を復元する前に、必要なデータを保存しておきます。
            _sizeBeforeCollapse = this.Size;
            Size panel1Size = this.Panel1Size;

            base.LoadState( dataSet );

            if( Collapsed )
            {
                this.Size = panel1Size;
            }
        }
        #endregion


    }
}
