﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using LayoutEditor.Forms;
using LayoutEditor.Structures.SerializableObject;
using System.Diagnostics;
using System.Windows.Forms;
using LECore;

namespace LayoutEditor
{
    /// <summary>
    /// アプリケーションメッセージ管理クラス
    /// </summary>
    public class AppMessageReporter
    {
        /// <summary>
        /// エラーメッセージのキャッシュです。
        /// </summary>
        class MessageCache
        {
            public readonly string CategoryName;
            public string Text;

            public MessageCache(string categoryName)
            {
                CategoryName = categoryName;
            }

            public bool IsEmpty
            {
                get { return string.IsNullOrEmpty(this.Text); }
            }

            public string GetMessage()
            {
                string text = "  " + this.Text.Replace(Environment.NewLine, Environment.NewLine + "  ");
                return string.Format("* {0}{1}{2}", this.CategoryName, Environment.NewLine, text);
            }
        }

        /// <summary>
        /// メッセージダイアログ
        /// </summary>
        readonly MessageReportDlg _messageReportDlg;

        /// <summary>
        /// AppSettings読み込み関数
        /// </summary>
        Func<AppSetting> _getAppSettingFunc = null;

        Stack<MessageCache> _messageStack = new Stack<MessageCache>();

        /// <summary>
        /// ダイアログ表示の無効化
        /// </summary>
        public bool DialogDisabled { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AppMessageReporter(Func<AppSetting> func)
        {
            _getAppSettingFunc = func;
            _messageReportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, true);
        }

        /// <summary>
        /// ダイアログの親フォームを設定します。
        /// </summary>
        public void InitializeReportDlgOwner(Form owner)
        {
            Debug.Assert(_messageReportDlg.Owner == null);
            _messageReportDlg.Owner = owner;
        }

        /// <summary>
        /// メッセージの集約を開始します。
        /// </summary>
        public void BeginPacking(string categoryName)
        {
            _messageStack.Push(new MessageCache(categoryName));
        }

        /// <summary>
        /// 集約を終了します。
        /// </summary>
        /// <param name="bDoReportMessage">メッセージを報告するかどうか。</param>
        public void EndPacking(bool forceToShow)
        {
            if (_messageStack.Count > 0)
            {
                MessageCache packedMessage = _messageStack.Pop();
                if (!packedMessage.IsEmpty)
                {
                    if (_messageStack.Count == 0 || forceToShow)
                    {
                        ShowDialog_(
                            new MessageArgs() {
                                title = "Error Report",
                                msg = packedMessage.GetMessage(),
                            }
                        );
                    }
                    else
                    {
                        if (_messageStack.Count > 0)
                        {
                            string packedText = packedMessage.GetMessage();

                            MessageCache parentChache = _messageStack.Peek();
                            if (!string.IsNullOrEmpty(parentChache.Text) &&
                                !parentChache.Text.EndsWith(Environment.NewLine))
                            {
                                parentChache.Text += Environment.NewLine;
                            }

                            parentChache.Text += packedText;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// メッセージ表示関数
        /// </summary>
        public void ReportMsgHandler(object sender, MessageArgs args)
        {
            Debug.Assert(_messageReportDlg != null);

            AppSetting settings = _getAppSettingFunc();

            MessageReportDlg.MessageTypes type = (args as LayoutEditorMessageArgs)?.messageType ?? MessageReportDlg.MessageTypes.Always;

            // 表示無視フラグが設定されていなか確認します。
            bool bIgnore = settings?.IgnoreSimilarMessage?.Contains(type) ?? false;
            if (!bIgnore)
            {
                if (_messageStack.Count <= 0)
                {
                    ShowDialog_(args);

                    // 以降の表示を無視すべきか判断し、表示無視フラグとして記憶します。
                    if (_messageReportDlg.IgnoreSimilarMessage)
                    {
                        if (settings?.IgnoreSimilarMessage != null)
                        {
                            settings.IgnoreSimilarMessage.Add(type);
                        }
                    }
                }
                else
                {
                    MessageCache currentChache = _messageStack.Peek();
                    if (string.IsNullOrEmpty(args.title))
                    {
                        currentChache.Text += args.msg + Environment.NewLine;
                    }
                    else
                    {
                        currentChache.Text += string.Format("[{0}]...{1}" + Environment.NewLine, args.title, args.msg);
                    }
                }
            }
            else
            {
                args.dialogResult = DialogResult.OK;
            }
        }

        /// <summary>
        /// 表示無視フラグをリセットします。
        /// </summary>
        public void ResetIgnoreState()
        {
            AppSetting settings = _getAppSettingFunc();
            if (settings?.IgnoreSimilarMessage != null)
            {
                settings.IgnoreSimilarMessage.Clear();
            }
        }

        /// <summary>
        /// ダイアログを表示します。
        /// </summary>
        public void ShowDialog_(MessageArgs args)
        {
            if (!DialogDisabled)
            {
                if (!_messageReportDlg.Visible)
                {
                    _messageReportDlg.Title = args.title;
                    _messageReportDlg.Message = args.msg;
                    _messageReportDlg.ChkIgnoreSimilarMessageEnabled = (args as LayoutEditorMessageArgs)?.chkIgnoreSimilarMessageEnabled ?? false;
                    _messageReportDlg.ChkIgnoreSimilarMessageChecked = false;
                    _messageReportDlg.ButtonType = (args as LayoutEditorMessageArgs)?.buttonType ?? MessageReportDlg.ButtonTypes.Ok;
                    args.dialogResult = _messageReportDlg.ShowDialog();
                }
                else
                {
                    // 同一インスタンスに対して ShowDialog ができないので後回しにする。
                    EventHandler waitDialogClosed = null;
                    waitDialogClosed = (s, e) => {
                        if (!_messageReportDlg.Visible)
                        {
                            Application.Idle -= waitDialogClosed;
                            ShowDialog_(args);
                        }
                    };
                    Application.Idle += waitDialogClosed;
                }
            }
        }
    }
}
