﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Drawing;
using System.Linq;

namespace LECore.Structures
{
    /// <summary>
    /// シーン変更イベント引数。
    /// </summary>
    public class SceneModifyEventArgs
    // 意味もなさそうなので、System.EventArgs から派生しない。
    // : System.EventArgs
    {
        /// <summary>
        /// 選択アイテムクリックイベントのパラメータ
        /// </summary>
        public class SelectedPaneClickedEventArgs
        {
            readonly IPane _pane;
            readonly PointF _posClicked;

            public IPane IPane { get { return _pane; } }
            public PointF PosClicked { get { return _posClicked; } }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public SelectedPaneClickedEventArgs( IPane pane, PointF posClicked )
            {
                _pane = pane;
                _posClicked = posClicked;
            }
        }

        /// <summary>
        /// 操作対象の種類
        /// </summary>
        public enum Kind
        {
            PaneModify = LECore.Structures.Core.LEDataNode.EventKind.Modify,                // ペイン
            PaneAnimModify = LECore.Structures.Core.LEDataNode.EventKind.AnimationModify,            // ペインアニメーション
            PaneAnimAddRemove = LECore.Structures.Core.LEDataNode.EventKind.AnimationAddRemove,         // ペインアニメーション追加・削除
            PaneAnimModeModify = LECore.Structures.Core.LEDataNode.EventKind.AnimationModeModify,   // ペインアニメーション管理モード変更
            PaneAttrModify = LECore.Structures.Core.LEDataNode.EventKind.AttributeModify,         // ペインアトリビュート
            PaneAdd,                   // ペイン追加
            PaneRemove,                // ペイン削除
            TextboxFontChanged,        // フォント変更
            TextboxContentsChanged,    // 文字内容の変更
            GroupModify,               // グループ
            GroupMake,                 // グループ
            GroupRemove,               // グループ
            GroupReorder,              // グループ
            GroupAddPane,              // グループ
            GroupRemovePane,           // グループ
            GroupReorderPane,          // グループ
            HierarchyModify,           // イベント通知用階層（Reordr,Make,Removeを一まとめにした種類）
            HierarchyReordr,
            HierarchyMake,             // 階層
            HierarchyRemove,           // 階層
            SelectedSetModify,         // 選択セットが変更された
            SelectedSetClicked,        // 選択セットがクリックされた
            CurrentSubSceneChanged,    // カレント・サブシーン
            SubSceneRemoved,           // サブシーンが削除された
            ClipBoardModify,           // クリップボード
            TextureManager,            // テクスチャマネージャ
            FontManager,               // フォントマネージャ
            AnimSectionTagSet,         // アニメーションフレーム区間タグセット
            AnimSectionTagTarget,      // アニメーションフレーム区間タグセット
            AnimShareInfomarionManager, // アニメーション共有情報マネージャ
            PartsSettings,              // 部品情報
            PartsLayout,                // 部品ペイン
            PartsLayoutFntResAdjusted,  // 部品ペイン（上書き要因でフォントが登録された）
            PartsLayoutTexResAdjusted,  // 部品ペイン（上書き要因でテクスチャが登録された）
            MaterialEditMode,           // 部品ペイン
            CurrentPaneModify,          // カレントペインが変更された
            SubScenePropertyModify,     // サブシーンのプロパティ(背景や拡張ユーザ情報等)
            StateMachineModify,         // ステートマシンが更新された
            StateMachineSelectionChanged,// ステートマシンが更新された

            // TODO: 経路が他と違うので ViewManagerMessageKind に変更する
            PlatformChanged,            // プラットフォームが変更された
        }

        #region プロパティ

        public Kind Target
        {
            get { return _kind; }
        }

        public List<object> Paramaters
        {
            get {
                // マージされているものも含める
                if (_mergedArgs.Any())
                {
                    _paramaters = LECore.Util.GenericUtil.GetNodeRecursively(this, x => x._mergedArgs).SelectMany(x => x._paramaters).Distinct().ToList();
                    _mergedArgs.Clear();
                }
                return _paramaters;
            }
        }

        #endregion プロパティ

        readonly Kind _kind;
        List<object> _paramaters = new List<object>();
        readonly List<SceneModifyEventArgs> _mergedArgs = new List<SceneModifyEventArgs>();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="actionType"></param>
        /// <param name="targetType"></param>
        public SceneModifyEventArgs
            (
            Kind targetType,
            object paramaters
            )
        {
            _kind = targetType;
            _paramaters.Add( paramaters );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="actionType"></param>
        public SceneModifyEventArgs
            (
            Kind targetType
            )
        {
            _kind = targetType;
        }

        /// <summary>
        /// パラメータに列挙子を追加します。
        /// </summary>
        public void  AddParameters(IEnumerable<object> args)
        {
            Debug.Assert(_paramaters.Count <= 0);
            Debug.Assert(_mergedArgs.Count <= 0);
            _paramaters.AddRange(args);
        }

        /// <summary>
        /// クリップボード変更イベントの引数を生成します。
        /// </summary>
        /// <param name="paramater">新しくクリップボードに入力されたデータ</param>
        /// <returns>イベント引数の実体</returns>
        static public SceneModifyEventArgs
            CreateClipBoardModifyEventArgs( object paramater )
        {
            return new SceneModifyEventArgs( Kind.ClipBoardModify, paramater );
        }

        /// <summary>
        /// 同一種類のコマンドか調査します。
        /// </summary>
        public bool IsSameType( SceneModifyEventArgs another )
        {
            return ( _kind == another.Target );
        }

        /// <summary>
        /// 2つのコマンドをマージします。
        /// 同一種類のコマンドのみ機能します。
        /// </summary>
        public void MergeCmd( SceneModifyEventArgs another )
        {
            if( another != this && this.IsSameType( another ) )
            {
                this._mergedArgs.Add(another);
            }
            else
            {
                Debug.Assert( false );
            }
        }
    }

    /// <summary>
    /// シーン更新イベントハンドラ
    /// </summary>
    public delegate void OnSceneModifyHandler( object sender, SceneModifyEventArgs e );

    /// <summary>
    /// シーン変更通知を受け取るクラスが実装するインタフェースです。
    /// </summary>
    public interface ISceneModifyListener
    {
        void OnSceneModifyHandler( object sender, SceneModifyEventArgs e );
    }
}
