﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Drawing;
using System.Diagnostics;

namespace LECore.Structures
{
    using Core;

    /// <summary>
    /// シーン内で、選択されたアイテムを表現するクラス。
    ///
    /// </summary>
    internal class SelectedSet :
       ISelectedSet
    {
        #region ----------------- 型定義 -----------------


        #endregion ----------------- 型定義 -----------------

        #region ----------------- フィールド -----------------
        // 選択ペインセット
        ArrayList            _paneSet                   = new ArrayList();
        // 選択変更イベントリスナ
        ArrayList            _selChangeEventListeners   = new ArrayList();

        // 変更イベントの通知を強制的に無効にします。
        // 大量の変更をまとめて行う場合に使用します。
        // 現在は外部からの操作インタフェースは提供していません。
        Stack                _changeEventOccuredStack = new Stack();

        private readonly ISubScene _owner;

        #endregion // ----------------- フィールド -----------------

        #region ----------------- プロパティ -----------------
        /// <summary>
        /// 選択ペインの配列を取得します。
        /// </summary>
        ///
        public IPane[]       IPaneArray
        {
            get
            {
                return PaneArray as IPane[];
            }
        }

        /// <summary>
        /// 選択ペインの配列を取得します。
        /// </summary>
        public Pane[] PaneArray
        {
            get
            {
                Pane[] paneSet = (Pane[])_paneSet.ToArray( typeof( Pane ) );
                return paneSet;
            }
        }

        /// <summary>
        /// 選択ペインの配列を取得します。
        /// </summary>
        public Pane[] PaneParentArray
        {
            get
            {
                ArrayList selectedSet = _paneSet.Clone() as ArrayList;

                // foreach 中で _paneSet 自身を変更できないので...
                Pane[] paneSet = (Pane[])selectedSet.ToArray( typeof( Pane ) );
                foreach( Pane pane in paneSet )
                {
                    // 親ペインが存在し...
                    Pane parent = pane.Parent as Pane;
                    while( parent != null )
                    {
                        // 親ペインが選択されている場合
                        if( selectedSet.Contains( parent ) )
                        {
                            // 自分は選択セットから外れる
                            selectedSet.Remove( pane );
                        }
                        parent = parent.Parent as Pane;
                    }
                }
                return selectedSet.ToArray( typeof( Pane ) ) as Pane[];
            }
        }

        /// <summary>
        /// 子ノードのセットを収集します。
        /// </summary>
        static void ColloectChildPanes_( ArrayList paneSet, Pane srcPane )
        {
            foreach( Pane childPane in srcPane.Children )
            {
                if( !paneSet.Contains( childPane ) )
                {
                    paneSet.Add( childPane );
                }

                // 再帰的に呼び出します。
                ColloectChildPanes_( paneSet, childPane );
            }
        }

        /// <summary>
        /// 選択ペインの子供を含むセットを取得します。
        /// </summary>
        public IPane[] SelectedPaneArrayWithChildren
        {
            get
            {
                ArrayList       paneWithChildren = new ArrayList();

                paneWithChildren.AddRange( _paneSet );

                foreach( Pane pane in _paneSet )
                {
                     ColloectChildPanes_( paneWithChildren, pane );
                }

                IPane[] paneSet = paneWithChildren.ToArray( typeof( IPane ) ) as IPane[];
                return paneSet;
            }
        }

        /// <summary>
        /// メンバを持っているか調べます。
        /// </summary>
        public bool Empty
        {
            get{ return _paneSet.Count == 0;}
        }

        /// <summary>
        /// 変更イベントが強制無効化状態か取得します。
        /// </summary>
        bool                 _ChangeEventForceDisable
        {
            get   { return _changeEventOccuredStack.Count != 0; }
        }

        #endregion // ----------------- プロパティ -----------------


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SelectedSet(ISubScene owner)
        {
            _owner = owner;
        }

        /// <summary>
        /// ペインが選択セットに含まれているか判定します。
        /// </summary>
        public bool Contains( Pane pane )
        {
            return _paneSet.Contains( pane );
        }

        #region ----------------- SelectedSetModify への 大規模変更通知 -----------------
        /// <summary>
        /// 変更開始
        /// </summary>
        public void BeginMassiveModify()
        {
            _changeEventOccuredStack.Push( false );
        }

        /// <summary>
        /// 変更終了
        /// BeginMassiveModify() を呼んでから、本関数コールまでの間に
        /// 更新イベント呼び出しが発生していた場合、変更イベントを通知します。
        /// </summary>
        public void EndMassiveModify()
        {
            Debug.Assert( _changeEventOccuredStack.Count > 0 );
            bool bEventOccured = (bool)_changeEventOccuredStack.Pop();

            if( bEventOccured )
            {
                NotifyChangeEvent_();
            }
        }
        #endregion ----------------- SelectedSetModify への 大規模変更通知 -----------------

        #region ----------------- ペイン登録、削除 -----------------
        /// <summary>
        /// ペインを登録します。
        /// 重複するペインの登録は無視されます。
        /// </summary>
        public void AddPane( Pane pane )
        {
            if( !_paneSet.Contains( pane ))
            {
                if (pane != _owner.RootIPane)
                {
                    _paneSet.Add(pane);
                    pane.IsSelected = true;

                    // 変更イベントをリスナに通知します。
                    NotifyChangeEvent_();
                }
            }
        }

        /// <summary>
        /// 複数のペインを登録します。
        ///
        /// AddPane()では、更新イベントが多数発生してしまい、
        /// パフォーマンス的に問題が発生する場合があります。
        /// </summary>
        /// <param name="panes"></param>
        public void AddPanes( Pane[] panes )
        {
            BeginMassiveModify();

            foreach( Pane pane in panes )
            {
                AddPane( pane );
            }

            EndMassiveModify();
        }

        /// <summary>
        /// ペインを選択セットから取り除きます。
        /// </summary>
        public void RemovePane( Pane pane )
        {
            if( _paneSet.Contains( pane ) )
            {
                pane.IsSelected = false;
                _paneSet.Remove( pane );

                // 変更イベントをリスナに通知します。
                NotifyChangeEvent_();
            }
        }

        /// <summary>
        /// 選択セットをリセットします。
        /// </summary>
        public void Reset()
        {
            if( _paneSet.Count != 0 )
            {
                // 選択リセット
                foreach( Pane pane in _paneSet )
                {
                    pane.IsSelected = false;
                }
                _paneSet.Clear();

                NotifyChangeEvent_();
            }
        }

        /// <summary>
        /// 同一階層に属するペインの重複選択を削除します。
        /// </summary>
        public void RemoveFamilyPane()
        {
            BeginMassiveModify();

            // foreach 中で _paneSet 自身を変更できないので...
            Pane[] tempPaenSet = (Pane[])_paneSet.ToArray( typeof( Pane ) );

            foreach( Pane pane in tempPaenSet )
            {
                // 親ペインが存在し...
                Pane parent = pane.Parent as Pane;
                while( parent != null )
                {
                    // 親ペインが選択されている場合
                    if( _paneSet.Contains( parent ) )
                    {
                        // 自分は選択セットから外れる
                        RemovePane( pane );
                    }
                    parent = parent.Parent as Pane;
                }
            }

            EndMassiveModify();
        }
        #endregion ----------------- ペイン登録、削除 -----------------

        #region ----------------- 更新イベントリスナ登録 -----------------
        /// <summary>
        /// 更新イベント通知クラスを登録します。
        /// </summary>
        public void AddListener( ISelectedSetChangedEventListener listener )
        {
            _selChangeEventListeners.Add( listener );
        }

        /// <summary>
        /// 更新イベント通知クラスを削除します。
        /// </summary>
        public void RemoveListener( ISelectedSetChangedEventListener listener )
        {
            _selChangeEventListeners.Remove( listener );
        }
        #endregion ----------------- 更新イベントリスナ登録 -----------------

        #region ----------------- イベント通知 -----------------
        public void NotifySelectedSetClickedEvent( IPane pane, PointF point )
        {
            foreach (ISelectedSetChangedEventListener listener in _selChangeEventListeners)
            {
                listener.OnSelectedSetClickedEventHandler(pane, point);
            }
        }
        #endregion ---------------- イベント通知 -------------------

        #region ----------------- 非公開メソッド -----------------
        /// <summary>
        /// 変更イベントをリスナに通知します。
        /// </summary>
        void NotifyChangeEvent_()
        {
            // 強制無効化されていたら、通知を行いません。
            if( (bool)_ChangeEventForceDisable )
            {
                // イベントが発生したことをスタックに記録します。
                _changeEventOccuredStack.Pop();
                _changeEventOccuredStack.Push( true );
                return;
            }

            if( _selChangeEventListeners.Count != 0 )
            {
                IPane[]  selectedPaneSet = (IPane[])_paneSet.ToArray( typeof( IPane ) );

                foreach( ISelectedSetChangedEventListener listener in _selChangeEventListeners )
                {
                    // listener.OnSelctedSetChanged( this );
                    listener.OnSelectedSetChangedEventHandler( selectedPaneSet );

                }
            }

        }


        #endregion // ----------------- 非公開メソッド -----------------

        #region ----------------- ISelectedSet メンバ -----------------



        #endregion ----------------- ISelectedSet メンバ -----------------

        #region ----------------- ICollidable メンバ -----------------

        public RectangleF BoundingVolumeInWorld
        {
            get
            {
                // すべてのペインの領域の和を計算し返します。
                if( _paneSet.Count != 0 )
                {
                    Pane         pane =  _paneSet[0] as Pane;
                    RectangleF   rect = pane.BoundingVolumeInWorld;

                    // 2個目移行の領域を加算していきます。
                    for( int i = 1; i < _paneSet.Count; i++ )
                    {
                        pane =  _paneSet[i] as Pane;
                        rect = RectangleF.Union( rect, pane.BoundingVolumeInWorld );
                    }
                    return rect;
                }
                else
                {
                    return RectangleF.Empty;
                }
            }
        }

        /// <summary>
        /// 点が選択セット内のペインに含まれるか判定します。
        /// </summary>
        public bool Contains( FVec2    posInScene )
        {
            foreach( Pane    pane in  _paneSet )
            {
                if( pane.Contains( posInScene ) )
                {
                    return true;
                }
            }
            return false;
        }

        #endregion // ----------------- ICollidable メンバ -----------------

        #region ----------------- IDrawable -----------------

        /// <summary>
        /// 制御点を描画します。
        /// </summary>
        void DrawControlPoint_( IRenderer renderer , float px, float py )
        {
            float ctrlSize = 4.0f;
            float halfSize = ctrlSize * 0.5f;
            renderer.DrawRectangle( px - halfSize, py - halfSize, 0, ctrlSize, ctrlSize );
        }

        /// <summary>
        /// 制御点群を描画します。
        /// </summary>
        void DrawControlPointSet_( IRenderer renderer, RectangleF selectedRect )
        {
            renderer.PushMtx();

            renderer.Trans( selectedRect.X, selectedRect.Y );
            renderer.Scale( 1.0f, -1.0f );

            renderer.Trans( 0.0f, -selectedRect.Height );

            float hw = selectedRect.Width * 0.5f;
            float hh = selectedRect.Height * 0.5f;

            DrawControlPoint_( renderer, 0, 0 );
            DrawControlPoint_( renderer, 0.0f, hh);
            DrawControlPoint_( renderer, 0.0f, selectedRect.Height );

            DrawControlPoint_( renderer, hw, 0.0f );
            DrawControlPoint_( renderer, hw, selectedRect.Height );

            DrawControlPoint_( renderer, selectedRect.Width , 0.0f );
            DrawControlPoint_( renderer, selectedRect.Width , hh );
            DrawControlPoint_( renderer, selectedRect.Width , selectedRect.Height );

            renderer.PopMtx();
        }

        /// <summary>
        /// 描画します。
        /// </summary>>
        public void Draw( IRenderer renderer, DrawableOption option )
        {
            if( Empty )
            {
                return;
            }

            renderer.Color = Color.Red;
            RectangleF selectedRect = this.BoundingVolumeInWorld;


            renderer.DrawRectangle(
                selectedRect.X,
                selectedRect.Y,
                0,
                selectedRect.Width,
                selectedRect.Height );

            DrawControlPointSet_( renderer, selectedRect );
        }
        #endregion ----------------- IDrawable -----------------
    }
}
