﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Drawing;

namespace LECore.Structures
{
    using System.Collections.Generic;
    using LECore.Structures.LECoreInterface;
    using LECore.Structures.SerializableObject.Lyt;
    using Core;

    /// <summary>
    /// Pictrue のアニメーションアトリビュート
    /// </summary>
    internal class PictureAttribute
        : AnmAttribute
    {
        static readonly AnmAttrDescripter[]  SubAttrDescs =
        {
            new  AnmAttrDescripter( AttributeType.ByteRGBA4, "vtx_col_LT", null, RGBAColor.White ),
            new  AnmAttrDescripter( AttributeType.ByteRGBA4, "vtx_col_RT", null, RGBAColor.White ),
            new  AnmAttrDescripter( AttributeType.ByteRGBA4, "vtx_col_LB", null, RGBAColor.White ),
            new  AnmAttrDescripter( AttributeType.ByteRGBA4, "vtx_col_RB", null, RGBAColor.White )
        };

        #region ------------- プロパティ -------------

        // 主にセーブロードで使用
        public AnmAttribute     VtxCol_LTAnmAttr {get{ return FindAttributeByIdx( 0 );}}
        public AnmAttribute     VtxCol_RTAnmAttr {get{ return FindAttributeByIdx( 1 );}}
        public AnmAttribute     VtxCol_LBAnmAttr {get{ return FindAttributeByIdx( 2 );}}
        public AnmAttribute     VtxCol_RBAnmAttr {get{ return FindAttributeByIdx( 3 );}}


        public RGBAColor     VtxCol_LT
        {
            get{ return VtxCol_LTAnmAttr.GetAsRBGA();}
            set{ VtxCol_LTAnmAttr.SetValue( value );}
        }

        public RGBAColor     VtxCol_RT
        {
            get{ return VtxCol_RTAnmAttr.GetAsRBGA();}
            set{ VtxCol_RTAnmAttr.SetValue( value );}
        }

        public RGBAColor     VtxCol_LB
        {
            get{ return VtxCol_LBAnmAttr.GetAsRBGA();}
            set{ VtxCol_LBAnmAttr.SetValue( value );}
        }

        public RGBAColor     VtxCol_RB
        {
            get{ return VtxCol_RBAnmAttr.GetAsRBGA();}
            set{ VtxCol_RBAnmAttr.SetValue( value );}
        }
        #endregion ------------- プロパティ -------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PictureAttribute( LEDataNode owner )
            :
            base
            (
            owner,
            SubAttrDescs,
            new AnmAttrDescripter( AttributeType.Combined, "picture", null, null )
            )
        {

        }
    }

    /// <summary>
    /// Picture の概要の説明です。
    /// </summary>
    internal class Picture
        :LEDataNode,
         IPicture,
         IVertexColor4Settable,
         IPaneExParamater,
         IRevHWMaterialHolder,
         IDrawable,
         IDisposable
    {
        public const string PictureNodeName = "Picture";

        #region フィールド
        readonly IPane              _ownerPane           = null;

        bool                        _useDetailedMaterial = false;
        ShapeType                   _shapeType           = ShapeType.NormalQuad;
        readonly Material           _material            = null;
        readonly RevHWMaterial      _revHWMaterial       = null;
        readonly ProceduralShape    _proceduralShape     = null;

        uint                        _circleSlice         = 16;
        uint                        _sphereSlice         = 4;
        uint                        _sphereStack         = 2;
        uint                        _roundRadius         = 1;
        uint                        _roundSlice          = 3;

        float                       _calcRoundRectWidth  = 0;
        float                       _calcRoundRectHeight = 0;
        bool                        _updateMesh          = false;

        // 矩形以外のメッシュデータ
        float[]                     _meshPositions       = null;
        float[]                     _meshTexCoords       = null;
        int[]                       _meshIndices         = null;

        readonly PictureAttribute   _pictureAttribute;
        readonly TexCoord4[]        _texCoord4Set     = new TexCoord4[Material.MaxNumTexCoord];
        int                         _numTecCoord      = 0;
        #endregion フィールド

        #region プロパティ
        /// <summary>
        /// テクスチャマネージャ
        /// </summary>
        ITextureMgr _ITextureMgr
        {
            get { return OwnerPane.OwnerSubScene.ITextureMgr; }
        }
         #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Picture( Pane ownerPane )
            :base( ownerPane, PictureNodeName )
        {
            Debug.Assert( ownerPane != null );

            _ownerPane        = ownerPane;
            _pictureAttribute = new PictureAttribute( this );

            _material      = new Material( this, ownerPane.PaneName );
            _revHWMaterial = new RevHWMaterial( this, _material );

            _proceduralShape = new ProceduralShape(ownerPane);

            // ownerPane.MakePicturePane( this );
            ownerPane.BindPaneExData( this );

            for( int i = 0; i < _texCoord4Set.Length; i++ )
            {
                _texCoord4Set[i] = new TexCoord4();
            }

            this.NumTexCoord = 1;
        }

        /// <summary>
        /// シェイプ形状に応じたメッシュデータの更新処理
        /// </summary>
        private void UpdateMeshData()
        {
            // 通常の四角形か任意形状の時は何もしない。
            if (ShapeType == ShapeType.NormalQuad)
            {
                return;
            }

            List<Gfx.PrimitiveShapeVertex> vertexBuffer = new List<Gfx.PrimitiveShapeVertex>();
            List<int> indexBuffer = new List<int>();
            float scaleAdaption = 1.0f;

            switch (ShapeType)
            {
                case ShapeType.GfxPrimitiveRoundRect:
                    // 短辺を 1.0 として矩形を作成する。
                    if (OwnerPane.RenderingWidth > OwnerPane.RenderingHeight)
                    {
                        float aspectRatio = Math.Abs((float)OwnerPane.RenderingWidth / (float)OwnerPane.RenderingHeight);
                        float roundRatio = Math.Abs((float)RoundRadius / (float)OwnerPane.RenderingHeight);
                        Gfx.PrimitiveShape.CreateRoundRect(vertexBuffer, indexBuffer, 1.0f, 1.0f, roundRatio / aspectRatio, roundRatio, (int)RoundSlice);
                    }
                    else
                    {
                        float aspectRatio = Math.Abs((float)OwnerPane.RenderingHeight / (float)OwnerPane.RenderingWidth);
                        float roundRatio = Math.Abs((float)RoundRadius / (float)OwnerPane.RenderingWidth);
                        Gfx.PrimitiveShape.CreateRoundRect(vertexBuffer, indexBuffer, 1.0f, 1.0f, roundRatio, roundRatio / aspectRatio, (int)RoundSlice);
                    }
                    _calcRoundRectWidth = OwnerPane.RenderingWidth;
                    _calcRoundRectHeight = OwnerPane.RenderingHeight;
                    scaleAdaption = 1.0f;
                    break;
                case ShapeType.GfxPrimitiveCircle:
                    Gfx.PrimitiveShape.CreateCircle(vertexBuffer, indexBuffer, (int)CircleSlice);
                    scaleAdaption = 0.5f;
                    break;
                    /*
                case 2:
                    Gfx.PrimitiveShape.CreateCube(vertexBuffer, indexBuffer);
                    scaleAdaption = 1.0f;
                    break;
                case 3:
                    Gfx.PrimitiveShape.CreateSphere(vertexBuffer, indexBuffer, (int)SphereSlice, (int)SphereStack);
                    scaleAdaption = 0.5f;
                    break;
                case 4:
                    Gfx.PrimitiveShape.CreateHemiSphere(vertexBuffer, indexBuffer, 32);
                    scaleAdaption = 0.5f;
                    break;
                case 5:
                    Gfx.PrimitiveShape.CreateCylinder(vertexBuffer, indexBuffer, 16);
                    scaleAdaption = 0.5f;
                    break;
                    */
                default:
                    Gfx.PrimitiveShape.CreateSphere(vertexBuffer, indexBuffer, 32, 32);
                    scaleAdaption = 0.5f;
                    break;
            }

            _meshPositions = new float[vertexBuffer.Count * 3];
            _meshTexCoords = new float[vertexBuffer.Count * 2];
            int posIndex = 0;
            int uvIndex = 0;

            foreach (var vertex in vertexBuffer)
            {
                _meshPositions[posIndex++] = vertex.x * scaleAdaption;
                _meshPositions[posIndex++] = vertex.y * scaleAdaption;
                _meshPositions[posIndex++] = vertex.z * scaleAdaption;

                _meshTexCoords[uvIndex++] = vertex.u;
                _meshTexCoords[uvIndex++] = vertex.v;
            }

            _meshIndices = new int[indexBuffer.Count];
            int indexIndex = 0;
            foreach (var index in indexBuffer)
            {
                _meshIndices[indexIndex++] = index;
            }
        }

        #region IDisposable メンバ

        public void Dispose()
        {
            _pictureAttribute.Dispose();
            _material.Dispose();
            _revHWMaterial.Dispose();
        }

        #endregion

        #region IPicture
        /// <summary>
        /// 詳細なマテリアル設定を使用するか
        /// </summary>
        public bool     UseDetailedMaterial
        {
            get{ return _useDetailedMaterial;}
            set
            {
                if( _useDetailedMaterial != value )
                {
                    _useDetailedMaterial = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// シェイプタイプ
        /// </summary>
        public ShapeType ShapeType
        {
            get { return _shapeType; }
            set
            {
                if (_shapeType != value)
                {
                    _shapeType = value;

                    if (_shapeType == 0)
                    {
                        _meshPositions = null;
                        _meshTexCoords = null;
                        _meshIndices = null;
                    }
                    else
                    {
                        _updateMesh = true;
                    }

                    NotifyChangeToScene_();
                }
            }
        }

        public Material Material
        {
            get{ return _material;}
        }

        public IMaterial IMaterial
        {
            get{ return _material;}
        }

        public IProceduralShape IProceduralShape
        {
            get { return _proceduralShape; }
        }

        public RGBAColor VtxCol_LT
        {
            get { return _pictureAttribute.VtxCol_LT; }
            set { if (_pictureAttribute.VtxCol_LT != value) { _pictureAttribute.VtxCol_LT = value; } }
        }

        public RGBAColor VtxCol_RT
        {
            get { return _pictureAttribute.VtxCol_RT; }
            set { if (_pictureAttribute.VtxCol_RT != value) { _pictureAttribute.VtxCol_RT = value; } }
        }

        public RGBAColor VtxCol_LB
        {
            get { return _pictureAttribute.VtxCol_LB; }
            set { if (_pictureAttribute.VtxCol_LB != value) { _pictureAttribute.VtxCol_LB = value; } }
        }

        public RGBAColor VtxCol_RB
        {
            get { return _pictureAttribute.VtxCol_RB; }
            set { if (_pictureAttribute.VtxCol_RB != value) { _pictureAttribute.VtxCol_RB = value; } }
        }
        #region アニメーションアトリビュート取得関連

        public IAnmAttribute VtxCol_LTIAnmAttr { get{ return VtxCol_LTAnmAttr; }}
        public IAnmAttribute VtxCol_RTIAnmAttr { get{ return VtxCol_RTAnmAttr; }}
        public IAnmAttribute VtxCol_LBIAnmAttr { get{ return VtxCol_LBAnmAttr; }}
        public IAnmAttribute VtxCol_RBIAnmAttr { get{ return VtxCol_RBAnmAttr; }}

        public AnmAttribute VtxCol_LTAnmAttr { get{ return _pictureAttribute.VtxCol_LTAnmAttr; }}
        public AnmAttribute VtxCol_RTAnmAttr { get{ return _pictureAttribute.VtxCol_RTAnmAttr; }}
        public AnmAttribute VtxCol_LBAnmAttr { get{ return _pictureAttribute.VtxCol_LBAnmAttr; }}
        public AnmAttribute VtxCol_RBAnmAttr { get{ return _pictureAttribute.VtxCol_RBAnmAttr; }}

        #endregion アニメーションアトリビュート取得関連

        /// <summary>
        /// 使用するテクスチャ座標数を取得、設定します。
        /// </summary>
        public int NumTexCoord
        {
            get{ return _numTecCoord;}
            set
            {
                Debug.Assert( value <= Material.MaxNumTexCoord );

                _numTecCoord = value;

                Material.UserMaxNumTexCoord = _numTecCoord;

                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// 全てのテクスチャ座標に値を設定します。
        /// </summary>
        public void SetAllTexCoord4( float u, float v)
        {
            for( int index = 0; index < Material.MaxNumTexCoord; index++ )
            {
                TexCoord4 coord = _texCoord4Set[index];
                coord.RT = new FVec2(    u, 0.0F);
                coord.LB = new FVec2( 0.0F,   v);
                coord.RB = new FVec2(    u,   v);
            }
        }

        /// <summary>
        /// 指定番号のテクスチャ座標を取得します。
        /// </summary>
        public TexCoord4 GetTexCoord4( int texCoordIdx )
        {
            TexCoord4 retCoord = null;
            if( texCoordIdx < _numTecCoord )
            {
                retCoord = _texCoord4Set[texCoordIdx];
            }
            return retCoord;
        }

        /// <summary>
        /// 指定番号のテクスチャ座標を設定します。
        /// </summary>
        public void SetTexCoord4( TexCoord4 srcTexCoord, int texCoordIdx )
        {
            TexCoord4    dstTexCoord = GetTexCoord4( texCoordIdx );
            if( dstTexCoord != null )
            {
                dstTexCoord.Set( srcTexCoord );
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// テクチャマップ列を取得します。
        /// </summary>
        public IMaterialTexMap[] IMaterialTexMapSet
        {
            get { return this.IMaterial.IMaterialTexMapSet; }
        }

        #endregion IPicture

        #region IShapeParam

        /// <summary>
        /// 円シェイプ分割数
        /// </summary>
        public uint CircleSlice
        {
            get { return _circleSlice; }
            set
            {
                if (_circleSlice != value)
                {
                    _circleSlice = value;
                    _updateMesh = true;

                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 球シェイプ横分割数
        /// </summary>
        public uint SphereSlice
        {
            get { return _sphereSlice; }
            set
            {
                if (_sphereSlice != value)
                {
                    _sphereSlice = value;
                    _updateMesh = true;

                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 球シェイプ縦分割数
        /// </summary>
        public uint SphereStack
        {
            get { return _sphereStack; }
            set
            {
                if (_sphereStack != value)
                {
                    _sphereStack = value;
                    _updateMesh = true;

                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 角丸四角形の角の半径
        /// </summary>
        public uint RoundRadius
        {
            get { return _roundRadius; }
            set
            {
                if (_roundRadius != value)
                {
                    _roundRadius = value;
                    _updateMesh = true;

                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 角丸四角形の角の分割数
        /// </summary>
        public uint RoundSlice
        {
            get { return _roundSlice; }
            set
            {
                if (_roundSlice != value)
                {
                    _roundSlice = value;
                    _updateMesh = true;

                    NotifyChangeToScene_();
                }
            }
        }

        #endregion IShapeParam

        #region IDrawable メンバ

        /// <summary>
        /// 描画
        /// </summary>
        public void Draw( IRenderer renderer, DrawableOption option )
        {
            if (option.CheckActive(DrawableOptionFlag.IgnorePaneImageDrawing))
            {
                return;
            }

            if (ShapeType == ShapeType.GfxPrimitiveRoundRect &&
                 (_calcRoundRectWidth != OwnerPane.RenderingWidth ||
                  _calcRoundRectHeight != OwnerPane.RenderingHeight))
            {
                _updateMesh = true;
            }

            if (_updateMesh)
            {
                UpdateMeshData();
                _updateMesh = false;
            }

            // HSV補正処理を施して、白黒補間を設定します。
            {
                IMaterial mat = _material;
                renderer.SetMatrialColorBlend(mat.WhiteColor, mat.BlackColor, mat.IsThresholdingAlphaInterpolationEnabled);

            }

            int numMatTex = Material.MaxNumMatTexture;
            TexCoord4[] texCooords = new TexCoord4[Material.MaxNumMatTexture];
            for (int i = 0; i < numMatTex; i++)
            {
                MaterialTexMap texMap = _material.GetMaterialTexMapByIndex(i);
                if (texMap == null || !texMap.IsActive)
                {
                    continue;
                }

                ITexGen texGen = texMap.ITexGen;
                ITextureImage texImg = texMap.TryFindTextureImage(OwnerPane.OwnerSubScene);
                if (texImg == null)
                {
                    continue;
                }

                // テクスチャ行列
                ITexMtx texMtx = texMap.TexGen.ITexMtx;
                FVec2 sca = TextureImageHelper.GetModifiedScale(texImg, texMtx.Scale);
                FVec2 tra = TextureImageHelper.GetModifiedTranslate(texImg, texMtx.Trans);
                renderer.SetTextureMtx((int)texGen.MtxID, tra, sca, texMtx.Rotate);

                // UV座標の計算
                {
                    TexCoord4 texCoord4;
                    if (texGen.IsUVMethod)
                    {
                        texCoord4 = this.GetTexCoord4((int)texGen.Method);
                    }
                    else
                    {
                        // 投影マッピング
                        Matrix34 tempMtx = renderer.CurrentMtx34;
                        renderer.PopMtx();

                        Matrix34 paneCenterMtx = renderer.CurrentMtx34;
                        renderer.PushMtx();
                        renderer.CurrentMtx34 = tempMtx;

                        Matrix34 projMtx34 = TexGenHelper.CalcTextureProjectionMtx(paneCenterMtx, this, texGen, texImg);

                        renderer.SetTextureProjectionState(i, new FVec2(texImg.Size), projMtx34);
                        texCoord4 = new TexCoord4();
                    }
                    texCooords[i] = texCoord4;
                }

                // TextureStage の設定
                if (!_revHWMaterial.LowLevelCombinerSettingsEnabled)
                {
                    int stageIndex = i - 1;
                    if (_revHWMaterial.ITevData.NumStages > stageIndex)
                    {
                        RendererTextrureFormat renderTexFmt = RendererTextrureFormatHelper.GetRendererTextrureFormat(texImg.PixelFmt, texImg.SrcImageHasAlphaBit);
                        ITevStage tevStage = _revHWMaterial.ITevData.GetITevStage(stageIndex);
                        if (tevStage != null)
                        {
                            // テクスチャサンプリング設定を行います。
                            bool isIndirectStage = _revHWMaterial.ITevData.IsIndirectState(stageIndex);
                            renderer.SetTextureSamplingState(i, texMap.WrapS, texMap.WrapT, texMap.MinFilter, texMap.MagFilter, !isIndirectStage);

                            renderer.SetTextureState(
                                texImg.GDIBitmap,
                                i,
                                tevStage.ColorStage.CombineMode,
                                tevStage.AlphaStage.CombineMode,
                                tevStage.ColorStage.IndirectScale,
                                tevStage.ColorStage.IndirectRotate,
                                isIndirectStage ? TextureImageHelper.GetChandelNum(texImg.PixelFmt) : -1,
                                renderTexFmt);
                        }
                        else
                        {
                            renderer.SetTextureSamplingState(i, texMap.WrapS, texMap.WrapT, texMap.MinFilter, texMap.MagFilter, true);
                            renderer.SetTextureState(texImg.GDIBitmap, i, renderTexFmt);
                        }
                    }
                }
                else
                {
                    RendererTextrureFormat renderTexFmt = RendererTextrureFormatHelper.GetRendererTextrureFormat(texImg.PixelFmt, texImg.SrcImageHasAlphaBit);
                    renderer.SetTextureSamplingState(i, texMap.WrapS, texMap.WrapT, texMap.MinFilter, texMap.MagFilter, true);
                    renderer.SetTextureState(texImg.GDIBitmap, i, renderTexFmt);
                }
            }

            // 詳細マテリアル
            if (_revHWMaterial.LowLevelCombinerSettingsEnabled)
            {
                for (int j = 0; j < _revHWMaterial.TevData.NumStages; j++)
                {
                    foreach (var args in _revHWMaterial.TevData.TevStages.GetTevStage(j).ColorStage.TevArgs)
                    {
                        LECore.Structures.Nsrif.Attributes.AttrTevSource attrTevSource = args.Source;
                    }
                }

                // コンスタントカラー
                List<Color> colorList = new List<Color>();

                for (int i = 0; i < _revHWMaterial.TevData.NumTevColors; i++)
                {
                    colorList.Add(_revHWMaterial.TevData.GetTevColor(i).ToSystemColor());
                }

                // 黒カラー
                colorList.Add(_material.BlackColor.ToSystemColor());

                // 白カラー
                colorList.Add(_material.WhiteColor.ToSystemColor());

                renderer.SetDetailedCombinerState(
                    colorList.ToArray(),
                    _revHWMaterial.TevData.TevStages.GetTevStages(),
                    _revHWMaterial.TevData.NumStages);
            }

            // 角丸設定
            if (IProceduralShape != null &&
                IProceduralShape.IsProceduralShapeEnabled)
            {
                // 形状
                renderer.SetProceduralShapeState(OwnerPane.Size.X, OwnerPane.Size.Y, IProceduralShape.Exp, IProceduralShape.Radius);
                // コンスタントバッファの都合により各エフェクトのブレンド設定をまとめて設定
                renderer.SetProceduralShapeEffectBlendState(
                    IProceduralShape.InnerStrokeBlendMode,
                    IProceduralShape.InnerShadowBlendMode,
                    IProceduralShape.ColorOverlayBlendMode,
                    IProceduralShape.GradationOverlayBlendMode);
                // 境界線
                renderer.SetProceduralShapeInnerStrokeState(
                    IProceduralShape.IsInnerStrokeEnabled,
                    IProceduralShape.InnerStrokeSize,
                    IProceduralShape.InnerStrokeColor.AsColor);
                // シャドウ(内側)
                renderer.SetProceduralShapeInnerShadowState(
                    IProceduralShape.IsInnerShadowEnabled,
                    OwnerPane.Size.X, OwnerPane.Size.Y, IProceduralShape.Radius,
                    IProceduralShape.InnerShadowSize,
                    IProceduralShape.InnerShadowAngle,
                    IProceduralShape.InnerShadowDistance,
                    IProceduralShape.InnerShadowColor.AsColor,
                    IProceduralShape.InnerShadowType);
                // カラーオーバーレイ
                renderer.SetProceduralShapeColorOverlayState(
                    IProceduralShape.IsColorOverayEnabled,
                    IProceduralShape.ColorOverlayColor.AsColor);
                // グラデーションオーバーレイ
                float[] controlPoints = new float[IProceduralShape.GradationOverlayControlPointCount];
                Color[] colors = new Color[IProceduralShape.GradationOverlayControlPointCount];
                for (int i = 0; i < IProceduralShape.GradationOverlayControlPointCount; ++i)
                {
                    controlPoints[i] = IProceduralShape.GetGradationOverlayControlPoint(i);
                    colors[i] = IProceduralShape.GetGradationOverlayColor(i).AsColor;

                }
                renderer.SetProceduralShapeGradationOverlayState(
                    IProceduralShape.IsGradationOverayEnabled,
                    controlPoints,
                    colors,
                    IProceduralShape.GradationOverlayAngle);
            }

            // 描画
            {
                Color[] colVtx = new Color[] { this.VtxCol_LT.AsColor, this.VtxCol_RT.AsColor, this.VtxCol_RB.AsColor, this.VtxCol_LB.AsColor };
                FVec2 size = new FVec2(this.OwnerPane.RenderingWidth, this.OwnerPane.RenderingHeight);

                var pe = _revHWMaterial.PEData;

                PEBlend colBlend = pe.UseDefaultBlendSettings ? PEBlend.DefaultBlendColor : pe.Blend;
                PEBlend alpBlend = pe.UseColorBlendSettingsForAlpha ? colBlend : pe.BlendAlpha;

                renderer.SetBlendMode(
                    colBlend.Type,
                    colBlend.SrcFactor, colBlend.DstFactor, colBlend.BlendOp,
                    alpBlend.SrcFactor, alpBlend.DstFactor, alpBlend.BlendOp);

                renderer.SetAlphaCompare(!pe.UseDefaultAlphaTestSettings, pe.ACompare.Comp, pe.ACompare.Ref);

                if (ShapeType == ShapeType.NormalQuad)
                {
                    // 通常の矩形描画
                    renderer.DrawImage(FVec3.Empty, size, texCooords, colVtx);
                }
                else
                {
                    // 特殊形状メッシュ描画
                    renderer.PushMtx();
                    renderer.Scale(OwnerPane.RenderingWidth, -OwnerPane.RenderingHeight, OwnerPane.RenderingWidth);
                    renderer.Trans(0.5f, -0.5f);

                    renderer.DrawMesh(_meshPositions, _meshTexCoords, _meshIndices);

                    renderer.PopMtx();
                }
            }


            // D3DRenderer では何もしていないが、意外と CPU 使用率が高いのでコメントアウト(GetHeightのせい?)
#if false
            // デバック用文字列の表示
            for (int i = 0; i < numMatTex; i++)
            {
                IMaterialTexMap texMap = _material.GetIMaterialTexMapByIndex(i);
                if (texMap != null)
                {
                    string msg = string.Format("Slot:{0} == {1}", i.ToString(), texMap.TexImgName);
                    renderer.DbgDrawString(msg, LayoutEditorCore.GuiFont, OwnerPane.Width, LayoutEditorCore.GuiFont.GetHeight() * i);
                }
            }
#endif
        }

#endregion IDrawable メンバ

#region IPaneExParamater
        /// <summary>
        /// 自身を持っているペインの参照です。
        /// </summary>
        public IPane OwnerPane
        {
            get{ return _ownerPane;}
        }

        /// <summary>
        /// マテリアル名を更新します。
        /// </summary>
        public void UpdateMatarialName()
        {
            this.Material.MaterialName = this.OwnerPane.PaneName;
        }

        /// <summary>
        /// シーン登録時の初期化処理
        /// </summary>
        public void OnJoinSceneInitialize()
        {
            // TODO:
            // テクスチャを検索して、設定しておく。
            // 描画時に毎回検索する方法は廃止する。
        }

        /// <summary>
        /// 編集対象に設定される直前の初期化処理
        /// </summary>
        public void FirstTimeInitialize()
        {
            // 何もしません。
        }

        /// <summary>
        /// UV座標を列挙します。
        /// </summary>
        public IEnumerable<TexCoord4> TexCoords
        {
            get
            {
                int count = 0;
                foreach (var texCoord in this._texCoord4Set)
                {
                    if (count >= NumTexCoord)
                    {
                        break;
                    }

                    yield return texCoord;
                    count++;
                }
            }
        }

        /// <summary>
        /// 警告文字列を取得します。UIの各所で表示します。
        /// </summary>
        public string WarningMsg
        {
            get { return string.Empty; }
        }

        /// <summary>
        /// 非選択状態でも境界を書くかどうか
        /// </summary>
        public bool PaneBoundDrawEnabled
        {
            get { return true; }
        }
#endregion IPaneExParamater

#region IRevHWMaterialHolder
        public IRevHWMaterial[] IRevHWMaterial
        {
            get{ return new IRevHWMaterial[] { _revHWMaterial } ;}
        }

        public RevHWMaterial[] RevHWMaterial
        {
            get{ return new RevHWMaterial[] { _revHWMaterial } ;}
        }
#endregion IRevHWMaterialHolder

#region イベント通知
        /// <summary>
        /// 変更を通知します。
        /// アニメーションアトリビュート以外の更新を通知します。
        /// </summary>
        void NotifyChangeToScene_()
        {
            NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }
#endregion イベント通知

    }
}
