﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace LECore.Structures
{
    using Core;
    using System.Diagnostics;

    class MaterialParamaterPaster
    {
        /// <summary>
        /// マテリアルテクスチャマップをコピーします。
        /// </summary>
        static public void CopyAllMatareialTextureMap(
            MaterialTexMap dstTex,
            IMaterialTexMap srcTex
            )
        {
            dstTex.MagFilter = srcTex.MagFilter;
            dstTex.MinFilter = srcTex.MinFilter;

            dstTex.TextureChanger.TextureSubSet.Clear();
            dstTex.TextureChanger.TextureSubSet.AddTextures(
                srcTex.ITextureChanger.ITextureSubSet.TextureSubSetNameSet );

            dstTex.TexGen.Set( srcTex.ITexGen as TexGen );
            dstTex.WrapS = srcTex.WrapS;
            dstTex.WrapT = srcTex.WrapT;
            dstTex.TexImgName = srcTex.TexImgName;
            dstTex.PaletteName = srcTex.PaletteName;
        }

        static private ISubScene ChoiceOwnerSubScene_(IMaterial srcMat, IMaterial dstMat)
        {
            ISubScene ownerSubScene = srcMat?.OwnerPane?.OwnerSubScene != null ? srcMat?.OwnerPane?.OwnerSubScene : dstMat?.OwnerPane?.OwnerSubScene;
            if (ownerSubScene == null)
            {
                Debug.Assert(false);
                ownerSubScene = LayoutEditorCore.Scene.CurrentISubScene;
            }

            return ownerSubScene;
        }

        /// <summary>
        /// マテリアルのコピー
        /// </summary>
        public static void CopyMaterialParamater(
            PaneParamaterCopyOption option,
            IMaterial dstIMat,
            IMaterial srcMat)
        {
            Material dstMat = dstIMat as Material;

            // マテリアルテクスチャマップ
            if( option.IsParamaterOptionEnable( ParamaterKind.Material_Textures ) )
            {
                if (srcMat.IMaterialTexMapSet.Length > 0)
                {
                    // 一旦すべてリセットし、その後コピーします。
                    dstMat.RemoveMatTextureAllNoEvent();
                    for (int texIdx = 0; texIdx < srcMat.IMaterialTexMapSet.Length; texIdx++)
                    {
                        IMaterialTexMap srcTex = srcMat.IMaterialTexMapSet[texIdx];

                        dstMat.RegisterMatTexture(srcTex.TexImgName, texIdx, srcTex.ResourceType);
                        CopyAllMatareialTextureMap(dstMat.MaterialTexMapSet[texIdx], srcTex);
                    }
                }
                else
                {
                    // すべてリセットします。
                    dstMat.RemoveMatTextureAll();
                }

                // baseValueのコピー
                if (option.CopyBaseValue)
                {
                    foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
                    {
                        if (kind == ParamaterKind.Animation_MaterialColors) continue; // カラーブレンドはここでは処理しない

                        IAnmAttribute[] dstAttrSet = MaterialHelper.GetAnimationTargetAttributeSet(dstIMat, kind);
                        IAnmAttribute[] srcAttrSet = MaterialHelper.GetAnimationTargetAttributeSet(srcMat, kind);
                        PaneParamaterPaster.CopyBaseValue(srcAttrSet, dstAttrSet, false);
                    }
                }
            }

            // カラーブレンド
            if( option.IsParamaterOptionEnable( ParamaterKind.Material_ColorBlend ) )
            {
                if (option.BlackColorAlpha == false)
                {
                    PaneParamaterCopyOption tmpOption = new PaneParamaterCopyOption(option);
                    tmpOption.SetVectorElementOption(VectorElement.A, false);
                    dstMat.BlackColor = ParamaterPasterCommon.CopyRGBA(tmpOption, dstMat.BlackColor, srcMat.BlackColor);
                }
                else
                {
                    dstMat.BlackColor = ParamaterPasterCommon.CopyRGBA(option, dstMat.BlackColor, srcMat.BlackColor);
                }
                dstMat.WhiteColor = ParamaterPasterCommon.CopyRGBA( option, dstMat.WhiteColor, srcMat.WhiteColor );
                dstMat.IsThresholdingAlphaInterpolationEnabled = srcMat.IsThresholdingAlphaInterpolationEnabled;

                dstMat.HOffset = srcMat.HOffset;
                dstMat.SScale = srcMat.SScale;
                dstMat.BScale = srcMat.BScale;

                // baseValueのコピー
                if (option.CopyBaseValue)
                {
                    IAnmAttribute[] dstAttrSet = MaterialHelper.GetAnimationTargetAttributeSet(dstIMat, ParamaterKind.Animation_MaterialColors);
                    IAnmAttribute[] srcAttrSet = MaterialHelper.GetAnimationTargetAttributeSet(srcMat, ParamaterKind.Animation_MaterialColors);
                    PaneParamaterPaster.CopyBaseValue(srcAttrSet, dstAttrSet, false);
                }
            }

            // --------------- アニメーション情報のコピー
            // すべての種類のアニメーションについて...
            foreach( ParamaterKind kind in ParamaterKindHelper.AnimationKindSet )
            {
                // コピー指定されていれば...
                if( option.IsParamaterOptionEnable( kind ) )
                {
                    IAnmAttribute[] dstAttrSet = MaterialHelper.GetAnimationTargetAttributeSet( dstIMat, kind );
                    IAnmAttribute[] srcAttrSet = MaterialHelper.GetAnimationTargetAttributeSet( srcMat, kind );

                    if( dstAttrSet.Length == srcAttrSet.Length )
                    {
                        for( int i = 0 ; i < srcAttrSet.Length ; i++ )
                        {
                            ParamaterPasterCommon.CopyAllAnimAttribute(
                                dstAttrSet[i],
                                srcAttrSet[i],
                                ChoiceOwnerSubScene_(srcMat, dstIMat),
                                option.OverwriteAnimation,
                                option.CopyAnimOnCurrentTagOnly,
                                option.TargetTagName);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 詳細マテリアルのコピー
        /// </summary>
        public static void CopyMaterialDetailParamaters(
            PaneParamaterCopyOption option,
            IRevHWMaterial dstIMat,
            IRevHWMaterial srcMat,
            bool bCopyDetail)
        {
            if (bCopyDetail)
            {
                RevHWMaterial dstMat = dstIMat as RevHWMaterial;

                dstMat.ColorAssignSet.Set(srcMat.IColorAssignSet);

                dstMat.PEData.Set(srcMat.IPEData as PEData);

                dstMat.TevData.Set(srcMat.ITevData);

                dstMat.LowLevelCombinerSettingsEnabled = srcMat.LowLevelCombinerSettingsEnabled;
                dstMat.CombinerUserShaderSettingsEnabled = srcMat.CombinerUserShaderSettingsEnabled;

                dstMat.CombinerUserShader.Set(srcMat.ICombinerUserShader);

                // baseValueのコピー
                if (option.CopyBaseValue)
                {
                    foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
                    {
                        IAnmAttribute[] dstAttrSet = MaterialHelper.GetAnimationTargetAttributeSet(dstIMat, kind);
                        IAnmAttribute[] srcAttrSet = MaterialHelper.GetAnimationTargetAttributeSet(srcMat, kind);
                        PaneParamaterPaster.CopyBaseValue(srcAttrSet, dstAttrSet, false);
                    }
                }
            }

            // --------------- アニメーション情報のコピー
            // すべての種類のアニメーションについて...
            foreach ( ParamaterKind kind in ParamaterKindHelper.AnimationKindSet )
            {
                // コピー指定されていれば...
                if( option.IsParamaterOptionEnable( kind ) )
                {
                    IAnmAttribute[] dstAttrSet = MaterialHelper.GetAnimationTargetAttributeSet( dstIMat, kind );
                    IAnmAttribute[] srcAttrSet = MaterialHelper.GetAnimationTargetAttributeSet( srcMat, kind );

                    if( dstAttrSet.Length == srcAttrSet.Length )
                    {
                        for( int i = 0 ; i < srcAttrSet.Length ; i++ )
                        {
                            ParamaterPasterCommon.CopyAllAnimAttribute(
                                dstAttrSet[i],
                                srcAttrSet[i],
                                ChoiceOwnerSubScene_(srcMat?.IMaterial, dstIMat?.IMaterial),
                                option.OverwriteAnimation,
                                option.CopyAnimOnCurrentTagOnly,
                                option.TargetTagName);
                        }
                    }
                }
            }

        }

    }
}
