﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;

namespace LECore.Structures
{
    using Core;
    using Nsrif.Attributes;

    #region TexMtx


    /// <summary>
    /// テクスチャ
    /// 外部公開インタフェース
    /// </summary>
    public interface ITexMtx
    {
        FVec2            Trans   { get;}
        FVec2            Scale   { get;}
        float            Rotate  { get;}

        IAnmAttribute    TransIAnmAttr { get;}
        IAnmAttribute    ScaleIAnmAttr { get;}
        IAnmAttribute    RotateIAnmAttr { get;}

        AttrTexCoordMtx  AttrTexCoordMtx { get;}
    }

    /// <summary>
    /// マテリアル：テクスチャ
    /// Material クラスが本クラスのインスタンスを生成し、所有します(総数10個)。
    ///
    /// 各 MaterialTexMap( 正確には MaterialTexMap.TexGen ) は、
    /// Material に存在する インスタンス を共有参照します。
    /// 各 MaterialTexMap に  TexMtx インスタンスが生成されない点に注意してください。
    /// </summary>
    internal class TexMtx :
        AnmAttribute,
        ITexMtx
    {
        static readonly AnmAttrDescripter[]  SubAttrDescs =
        {
            new  AnmAttrDescripter( AttributeType.FloatVec2UV, "Trans",  null, new FVec2( 0.0f, 0.0f ) ),
            new  AnmAttrDescripter( AttributeType.FloatVec2UV, "Scale",  null, new FVec2( 1.0f, 1.0f ) ),
            new  AnmAttrDescripter( AttributeType.Float,     "Rotate", null, 0.0f )
        };

        readonly int                    _TexMtxIdx = -1;

        #region ------------ ITexMtxメンバ ------------

        public FVec2     Trans
        {
            get{ return TransAnmAttr.GetAsFVec2();}
            set{ TransAnmAttr.SetValue( value );}
        }
        public FVec2     Scale
        {
            get{ return ScaleAnmAttr.GetAsFVec2();}
            set{ ScaleAnmAttr.SetValue( value );}
        }
        public float     Rotate
        {
            get{ return RotateAnmAttr.GetAsFloat();}
            set{ RotateAnmAttr.SetValue( value );}
        }

        public IAnmAttribute    TransIAnmAttr { get{ return TransAnmAttr;}}
        public IAnmAttribute    ScaleIAnmAttr { get{ return ScaleAnmAttr;}}
        public IAnmAttribute    RotateIAnmAttr { get{ return RotateAnmAttr;}}


        public AttrTexCoordMtx  AttrTexCoordMtx
        {
            get{ return (AttrTexCoordMtx)_TexMtxIdx;}
        }

        #endregion ------------ ITexMtxメンバ ------------

        #region ------------ キーフレームロード時に使用されるメンバ ------------

        public AnmAttribute  TransAnmAttr
        {
            get{ return FindAttributeByIdx( 0 );}
        }

        public AnmAttribute  ScaleAnmAttr
        {
            get{ return FindAttributeByIdx( 1 );}
        }

        public AnmAttribute  RotateAnmAttr
        {
            get{ return FindAttributeByIdx( 2 );}
        }

        #endregion ------------ キーフレームロード時に使用されるメンバ ------------


        static AnmAttrDescripter GetAttrDesc_( int texMtxIdx )
        {
            string name = string.Format( "Texture_Mtx_{0}", texMtxIdx );
            return new AnmAttrDescripter( AttributeType.Combined, name, null, null );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TexMtx( LEDataNode owner, int texMtxIdx )
            : base( owner, SubAttrDescs, GetAttrDesc_( texMtxIdx ) )
        {
            _TexMtxIdx = texMtxIdx;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TexMtx(LEDataNode owner, string descriptionName)
            : base(owner, SubAttrDescs, new AnmAttrDescripter(AttributeType.Combined, descriptionName, null, null))
        {
            _TexMtxIdx = 0;
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        public void Set( ITexMtx src )
        {
            this.Trans = src.Trans;
            this.Scale = src.Scale;
            this.Rotate = src.Rotate;
        }
    }
    #endregion
}
