﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;

namespace LECore.Structures
{
    using Core;


    using Nsrif.Attributes;

    #region チャンネルデータ

    #region ChanMatColors
    /// <summary>
    /// マテリアルカラー管理クラス。
    /// </summary>
    internal sealed class ChanMatColors
        : AnmAttribute
    {

        static readonly AnmAttrDescripter[]  SubAttrDescs =
        {
            new  AnmAttrDescripter( AttributeType.FloatRGBA4, "_register0", null, null ),
            new  AnmAttrDescripter( AttributeType.FloatRGBA4, "_register1", null, null )
        };

        #region ------------ AnmAttrCreateFunc関連 ------------

        static public readonly AnmAttrCreateFunc  CreateFunc = new AnmAttrCreateFunc( CreateChanMatColorsAttr );

        static AnmAttribute CreateChanMatColorsAttr( AnmAttribute owner, AnmAttrDescripter attrDesc )
        {
            AnmAttrDescripter    selfAttrDesc =
                new AnmAttrDescripter( AttributeType.Combined, attrDesc._name, null, null );

            AnmAttribute anmAttr = new ChanMatColors( owner, selfAttrDesc );
            return anmAttr;
        }

        #endregion ------------ AnmAttrCreateFunc関連 ------------

        /// <summary>
        /// RGBAColor インデクサ
        /// </summary>
        public RGBAColor     this[int index]
        {
            get{ return FindAttributeByIdx(index).GetAsRBGA();}
            set{ FindAttributeByIdx(index).SetValue( value );}
        }

        /// <summary>
        /// コンストラクタ(非公開)
        /// </summary>
        ChanMatColors( LEDataNode owner, AnmAttrDescripter selfAttrDesc )
            : base( owner, SubAttrDescs, selfAttrDesc )
        {

        }
    }
    #endregion

    #region ChanAmbColors
    /// <summary>
    /// アンビエントカラー管理クラス。
    /// </summary>
    internal sealed class ChanAmbColors
        : AnmAttribute
    {

        static readonly AnmAttrDescripter[]  SubAttrDescs =
        {
            new  AnmAttrDescripter( AttributeType.FloatRGBA4, "_register0", null, null ),
            new  AnmAttrDescripter( AttributeType.FloatRGBA4, "_register1", null, null )
        };

        #region ------------ AnmAttrCreateFunc関連 ------------

        static public readonly AnmAttrCreateFunc  CreateFunc = new AnmAttrCreateFunc( CreateChanAmbColorsAttr );

        static AnmAttribute CreateChanAmbColorsAttr( AnmAttribute owner, AnmAttrDescripter attrDesc )
        {
            AnmAttrDescripter    selfAttrDesc =
                new AnmAttrDescripter( AttributeType.Combined, attrDesc._name, null, null );

            AnmAttribute anmAttr = new ChanAmbColors( owner, selfAttrDesc );
            return anmAttr;
        }

        #endregion ------------ AnmAttrCreateFunc関連 ------------


        /// <summary>
        /// コンストラクタ(非公開)
        /// </summary>
        ChanAmbColors( AnmAttribute owner, AnmAttrDescripter selfAttrDesc )
            : base( owner, SubAttrDescs, selfAttrDesc )
        {

        }
    }

    #endregion

    #region ChanCtrls
    /// <summary>
    /// チャンネルコントロール管理クラス。
    /// </summary>
    public sealed class ChanCtrls
    {
        private readonly ChanCtrl[] _ctrls = new ChanCtrl[4];

        #region コンストラクタ
        /// <summary>
        /// MEMO:簡略化しました。検証が必要。
        /// </summary>
        public ChanCtrls()
        {
            for (int i = 0; i < 4; i++)
            {
                _ctrls[i] = new ChanCtrl();
            }
        }
        #endregion

        #region アトリビュート
        /// <summary>
        /// インデクサ。
        /// </summary>
        public ChanCtrl this[int ctrlNo]
        {
            get { return _ctrls[ctrlNo]; }
        }
        #endregion


    }
    #endregion

    #region ChanCtrl
    /// <summary>
    /// チャンネルコントロールクラス。
    /// </summary>
    public sealed class ChanCtrl
    {
        private bool            _lighting    = false;
        private AttrColorSource _ambSrc      = AttrColorSource.Reg;
        private AttrColorSource _matSrc      = AttrColorSource.Reg;
        private AttrDiffuseFunc _diffuseFunc = AttrDiffuseFunc.Sign;
        private AttrAttnFunc    _attnFunc    = AttrAttnFunc.Spec;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ChanCtrl() {}

        #endregion

        #region アトリビュート
        /// <summary>
        /// ライト計算フラグ。
        /// </summary>
        public bool Lighting
        {
            get { return _lighting;  }
            set { _lighting = value; }
        }

        /// <summary>
        /// アンビエントソース。
        /// </summary>
        public AttrColorSource AmbSrc
        {
            get { return _ambSrc;  }
            set { _ambSrc = value; }
        }

        /// <summary>
        /// マテリアルソース。
        /// </summary>
        public AttrColorSource MatSrc
        {
            get { return _matSrc;  }
            set { _matSrc = value; }
        }

        /// <summary>
        /// 拡散関数。
        /// </summary>
        public AttrDiffuseFunc DiffuseFunc
        {
            get { return _diffuseFunc;  }
            set { _diffuseFunc = value; }
        }

        /// <summary>
        /// 減衰関数。
        /// </summary>
        public AttrAttnFunc AttnFunc
        {
            get { return _attnFunc;  }
            set { _attnFunc = value; }
        }
        #endregion
    }
    #endregion

    #region ChannelData

    /// <summary>
    /// チャンネルデータクラス。
    /// 外部公開インタフェース
    ///
    /// 現在は 2D レイアウトに使用される、
    /// パラメータだけを限定的に公開しています。
    /// </summary>
    public interface IChannelData
    {
        AttrColorSource  DiffuseCtrlColorSrc   { get;}
        AttrColorSource  AlphaCtrlColorSrc     { get;}

        RGBAColor        MatColReg0            { get;}
    }

    /// <summary>
    /// チャンネルデータクラス。
    /// </summary>
    internal sealed class ChannelData :
        AnmAttribute,
        IChannelData
    {
        static readonly AnmAttrDescripter[]  SubAttrDescs =
        {
            new  AnmAttrDescripter( AttributeType.Combined, "mat_col_regs", ChanMatColors.CreateFunc, null )
        };

        private readonly ChanCtrls     _chanCtrls;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ(非公開)
        /// </summary>
        public ChannelData( AnmAttribute owner, AnmAttrDescripter selfAttrDesc )
            : base( owner, SubAttrDescs, selfAttrDesc )
        {
            _chanCtrls = new ChanCtrls();

            // LayoutEditorで、実際に使用される成分だけ、初期化します。
            DiffuseCtrl.MatSrc   = AttrColorSource.Vtx;
            AlphaCtrl.MatSrc     = AttrColorSource.Vtx;
        }
        #endregion

        #region チャンネルアクセス
        /// <summary>
        /// チャンネルコントロール。
        /// </summary>
        public ChanCtrls ChanCtrls
        {
            get { return _chanCtrls; }
        }

        /// <summary>
        /// ディフューズコントロール。
        /// </summary>
        public ChanCtrl DiffuseCtrl
        {
            get { return _chanCtrls[0]; }
        }

        /// <summary>
        /// アルファコントロール。
        /// </summary>
        public ChanCtrl AlphaCtrl
        {
            get { return _chanCtrls[1]; }
        }

        /// <summary>
        /// スペキュラコントロール。
        /// </summary>
        public ChanCtrl SpecularCtrl
        {
            get { return _chanCtrls[2]; }
        }
        #endregion

        #region IChannelData メンバ

        /// <summary>
        /// マテリアルカラーレジスタ配列を取得します。
        /// </summary>
        ChanMatColors _ChanMatColorRegisters
        {
            get{ return this.FindAttributeByIdx( 0 ) as ChanMatColors; }
        }

        public AttrColorSource  DiffuseCtrlColorSrc
        {
            get{ return DiffuseCtrl.MatSrc;}
            set{ DiffuseCtrl.MatSrc = value;}
        }

        public AttrColorSource  AlphaCtrlColorSrc
        {
            get{ return AlphaCtrl.MatSrc;}
            set{ AlphaCtrl.MatSrc = value;}
        }

        public RGBAColor        MatColReg0
        {
            get{ return _ChanMatColorRegisters[0];}
            set{ _ChanMatColorRegisters[0] = value;}
        }

        /// <summary>
        /// 設定関数
        /// (本当に必要とするパラメータに限ってコピーしています。)
        /// </summary>
        public void Set( IChannelData src )
        {
            this.DiffuseCtrlColorSrc    = src.DiffuseCtrlColorSrc;
            this.AlphaCtrlColorSrc      = src.AlphaCtrlColorSrc;
            this.MatColReg0             = src.MatColReg0;
        }
        #endregion IChannelData メンバ
    }
    #endregion

    #endregion
}
