﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.Text.RegularExpressions;
using System.IO;
using System.Linq;
using System.Collections.Generic;

namespace LECore.Structures
{
    /// <summary>
    /// SceneHelper の概要の説明です。
    /// </summary>
    public static class SceneHelper
    {
        #region 部品

        /// <summary>
        /// レイアウトを列挙します。
        /// </summary>
        public static IEnumerable<string> EnumrateLayoutFiles(this IScene scene)
        {
            if (!Directory.Exists(scene.PartsRootPath))
            {
                yield break;
            }

            foreach (var filePath in Directory.GetFiles(
                scene.PartsRootPath, AppConstants.LayoutFileExtPattern, SearchOption.AllDirectories))
            {
                yield return filePath;
            }
        }

        /// <summary>
        /// 部品レイアウトを列挙します。
        /// </summary>
        public static IEnumerable<string> EnumratePartsLayoutFiles(this IScene scene)
        {
            foreach (var filePath in EnumrateLayoutFiles(scene))
            {
                // FIXME:適当に調査しています！
                string fileContent = File.ReadAllText(filePath);
                if (fileContent.Contains("<control "))
                {
                    yield return filePath;
                }
            }
        }

        /// <summary>
        /// ファイルが存在しなければコピーします。
        /// </summary>
        /// <param name="srcFile"></param>
        /// <param name="dstFile"></param>
        private static void CopyTemporaryFileIfNotExsisted(string srcFile, string dstFile)
        {
            if (!File.Exists(dstFile))
            {
                File.Copy(srcFile, dstFile);

                // 一時コピーは削除することが前提なので、ReadOnly 属性をキャンセルしておきます。
                FileAttributes origAttr = File.GetAttributes(dstFile);
                origAttr = origAttr & ~FileAttributes.ReadOnly;
                File.SetAttributes(dstFile, origAttr);
            }
        }

        /// <summary>
        /// 部品レイアウトファイルをコピーします。
        /// </summary>
        public static void CopyPartsLayoutFile(string layoutFile, string dstFolderPath, List<string> referencedPartsFilePaths)
        {
            referencedPartsFilePaths.Add(layoutFile);

            // lyout
            string copiedPartsLayoutPath = Path.Combine(dstFolderPath, Path.GetFileName(layoutFile));
            CopyTemporaryFileIfNotExsisted(layoutFile, copiedPartsLayoutPath);

            // anim
            string animFile = Path.ChangeExtension(layoutFile, AppConstants.AnimationFileExt);
            if (File.Exists(animFile))
            {
                CopyTemporaryFileIfNotExsisted(animFile, Path.Combine(dstFolderPath, Path.GetFileName(animFile)));
            }

            // 依存ファイルパスを探します。
            string fileContent = File.ReadAllText(copiedPartsLayoutPath);

            // Texture
            foreach (Match m in Regex.Matches(fileContent, "textureFile imagePath=\"(.*?)\""))
            {
                string relativeFilePath = m.Groups[1].Value;
                string imageFilePath = Path.Combine(Path.GetDirectoryName(layoutFile), relativeFilePath);
                if (File.Exists(imageFilePath))
                {
                    fileContent = fileContent.Replace(relativeFilePath, imageFilePath);
                }
            }

            // Font
            foreach (Match m in Regex.Matches(fileContent, "fontFile path=\"(.*?)\""))
            {
                string relativeFilePath = m.Groups[1].Value;
                string fontFilePath = Path.Combine(Path.GetDirectoryName(layoutFile), relativeFilePath);
                if (File.Exists(fontFilePath))
                {
                    fileContent = fileContent.Replace(relativeFilePath, fontFilePath);
                }
            }

            // CombinerUserShader
            const string combinerUserShaderFileNameKey = "CombinerUserShader fileName=";
            foreach (Match m in Regex.Matches(fileContent, combinerUserShaderFileNameKey + "\"(.*?)\""))
            {
                string relativeFilePath = m.Groups[1].Value;
                string combinerFilePath = Path.Combine(Path.GetDirectoryName(layoutFile), relativeFilePath);
                if (File.Exists(combinerFilePath))
                {
                    // コンバイナファイルは複数設定される場合がり relativeFilePath だけだと更新箇所もさらに入れ替えが行われるので項目を含めて一致とします。
                    fileContent = fileContent.Replace(
                                    combinerUserShaderFileNameKey + "\""+ relativeFilePath + "\"",
                                    combinerUserShaderFileNameKey + "\"" + combinerFilePath + "\"");
                }
            }

            File.WriteAllText(copiedPartsLayoutPath, fileContent);

            // 部品レイアウトが他の部品レイアウトを利用している場合は再帰的に処理をおこなう。
            foreach (Match m in Regex.Matches(fileContent, "parts path=\"(.*?)\""))
            {
                string relativeFilePath = m.Groups[1].Value;
                string partsFilePath = Path.Combine(Scene.Instance.PartsRootPath, relativeFilePath);
                if (File.Exists(partsFilePath))
                {
                    CopyPartsLayoutFile(partsFilePath, dstFolderPath, referencedPartsFilePaths);
                }
            }
        }

        /// <summary>
        /// 部品設定テンプレートを取得します。
        /// </summary>
        public static IPartsControlSetting FindPartsControlSetting(this IScene scene, string name)
        {
            return scene.PartsControlSettings.FirstOrDefault((controlSetting) => controlSetting.Name == name);
        }

        /// <summary>
        /// 部品レイアウトを取得します。
        /// </summary>
        public static IPartsSubScene FindPartsSubSceneByFileName(this IScene scene, string fileName)
        {
            var parts = scene.PartsSubScenes.FirstOrDefault((subScene) => Path.GetFileName(subScene.FilePath) == fileName);
            return parts;
        }

        /// <summary>
        /// 部品レイアウトファイルを取得します。
        /// </summary>
        public static string FindPartsSubSceneFilePath(this IScene scene, ISubScene partsSubScene)
        {
            var parts = scene.PartsSubScenes.FirstOrDefault((subScene) => object.ReferenceEquals(subScene.SubScene, partsSubScene));
            return parts != null ? parts.FilePath : string.Empty;
        }

        /// <summary>
        /// 部品サブシーンを読み込み、ツール内部にキャッシュします。
        /// </summary>
        public static void LoadAndCachePartsSubSceneIfNeeded(this IScene scene, string fileName)
        {
            try
            {
                LayoutEditorCore.MsgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSLAYOUTLOADING"));

                var partsSubScene = scene.FindPartsSubSceneByFileName(fileName);
                if (partsSubScene != null && !partsSubScene.IsLoaded)
                {
                    (scene as Scene).LoadPartsSubScene(partsSubScene);
                }
            }
            finally
            {
                LayoutEditorCore.MsgReporter.EndPacking();
            }
        }

        /// <summary>
        /// 部品サブシーンを追加します。
        /// </summary>
        public static IPartsSubScene AddPartsSubScene(this IScene scene, string filePath)
        {
            return (scene as Scene).AddPartsSubScene(filePath);
        }

        /// <summary>
        /// スケーラブルフォントの登録
        /// </summary>
        public static void RegisterScalableFont(this IScene scene,
            string fontName, string fontPathBaseFromPartsRoot, int fontSize, float boldWeight, int borderWidth)
        {
            ScalableFontSettings settings = new ScalableFontSettings();

            Uri root = new Uri(scene.PartsRootPath);

            settings.FontName = fontName;
            settings.FontFilePath = GetAbsolutePathFromPartsRootBasedPath(scene, fontPathBaseFromPartsRoot);
            settings.FontSize = fontSize;
            settings.BoldWeight = boldWeight;
            settings.BorderWidth = borderWidth;

            (scene as Scene).RegisterScalableFont(settings);
        }

        /// <summary>
        /// スケーラブルフォントの削除
        /// </summary>
        public static void ClearScalableFont(this IScene scene)
        {
            (scene as Scene).ClearScalableFont();
        }

        /// <summary>
        /// 部品ルート相対パスを絶対パスに変換します。
        /// </summary>
        public static string GetAbsolutePathFromPartsRootBasedPath(this IScene scene, string partsRootBasedPath)
        {
            if(string.IsNullOrEmpty(partsRootBasedPath))
            {
                return string.Empty;
            }

            // 必要なら環境変数を展開する。
            string newPath;
            if (partsRootBasedPath.Contains("%"))
            {
                newPath = Environment.ExpandEnvironmentVariables(partsRootBasedPath);
            }
            else
            {
                newPath = Path.Combine(scene.PartsRootPath, partsRootBasedPath);
            }

            if (Directory.Exists(newPath))
            {
                return new DirectoryInfo(newPath).FullName;
            }
            else if (File.Exists(newPath))
            {
                return new FileInfo(newPath).FullName;
            }

            return newPath;
        }

        #endregion 部品

        /// <summary>
        /// ルートペインの持ち主サブシーンを持つかどうか
        /// </summary>
        public static bool ContainsOwnerSubSceneOf(this IScene scene, IPane rootPane)
        {
            return scene.ISubSceneSet.FirstOrDefault((subScene) => subScene.RootIPane == rootPane) != null;
        }
    }
}
