﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace LECore.Structures
{
    /// <summary>
    /// 部品ヘルパークラス
    /// </summary>
    public static class PartsSettingsHelper
    {
        /// <summary>
        /// 上書き可能なペイン種類か？
        /// </summary>
        public static bool IsOverridablePaneKind(PaneKind kind)
        {
            return
                kind == PaneKind.Null ||
                kind == PaneKind.Bounding ||
                kind == PaneKind.Textbox ||
                kind == PaneKind.Picture ||
                kind == PaneKind.Window ||
                kind == PaneKind.Parts ||
                kind == PaneKind.Capture;
            // PaneKind.Aligment は非サポート
            // PaneKind.Scissor は非サポート
        }

        //----------------------------------------------------------

        /// <summary>
        /// 部品設定を設定
        /// </summary>
        public static void Set(
            this IPartsSettings target, ISubScene targetScene, IPartsSettings source)
        {
            if (source.IsInitialState)
            {
                (target as PartsSettings).Reset();
                return;
            }

            (target as PartsSettings).Initialize(
                source.Name, source.DescriptionName, source.Description, source.OriginalSize);

            (target as PartsSettings).SetOverrideProperties(
                GetPartsPropaertySettingsSources(targetScene, source.OverridePorperties));
        }

        /// <summary>
        /// コントロール設定を設定
        /// </summary>
        public static void Set(
            this IControlSettingsSet targetCtrlSet, IEnumerable<IControlSettings> sourceCtrlSet)
        {
            targetCtrlSet.Clear();

            foreach(ControlSettings srcCtrl in sourceCtrlSet)
            {
                targetCtrlSet.CloneAdd(srcCtrl);
            }
        }

        /// <summary>
        /// 設定
        /// </summary>
        public static List<PartsPropaertySettingsSource> GetPartsPropaertySettingsSources(
            ISubScene targetScene, IEnumerable<IPartsPropaertySettings> sources)
        {
            var oldPropertySources = new List<PartsPropaertySettingsSource>();
            foreach (var overridePorperty in sources)
            {
                IPane targetPane = targetScene.FindPaneByName(overridePorperty.TargetName);
                if (targetPane != null)
                {
                    oldPropertySources.Add(new PartsPropaertySettingsSource(targetPane, overridePorperty.Description));
                }
                else
                {
                    oldPropertySources.Add(new PartsPropaertySettingsSource(
                        overridePorperty.TargetName,
                        overridePorperty.PaneKind,
                        overridePorperty.Description));
                }
            }

            return oldPropertySources;
        }

        //----------------------------------------------------------

        /// <summary>
        /// 部品プリセット設定がフォントを利用しているかどうか？
        /// </summary>
        public static bool UsesFont(this IDerivativePartsSettings derivativeSettings, ILEFont font)
        {
            return derivativeSettings.PartsPropaerties.FirstOrDefault(
                (prop) =>
                    prop.Paramater != null &&
                    prop.Paramater.OwnerPane != null &&
                    prop.Paramater.OwnerPane.UsesFont(font)) != null;
        }

        //----------------------------------------------------------

        /// <summary>
        /// 未設定のアニメーション区間タグ設定があるかどうか
        /// </summary>
        public static bool ContainsUnsetAnimationTag(this IControlSettings settings)
        {
            return settings.Animations.Any((anim) => anim.IsRequired && string.IsNullOrEmpty(anim.TagName));
        }

        /// <summary>
        /// 設定が見つからないアニメーション区間タグ設定があるかどうか
        /// </summary>
        public static bool ContainsNotFoundAnimationTag(this IControlSettings settings)
        {
            return settings.Animations.Any((anim) => anim.IsRequired && !settings.GetOwnerSubScene().IAnimFrameSectionSet.CheckExistsByName(anim.TagName));
        }

        /// <summary>
        /// アニメーションが出力されないアニメーション区間タグ設定があるかどうか
        /// </summary>
        public static bool ContainsNotExportAnimationTag(this IControlSettings settings)
        {
            if (settings.Owner == null)
            {
                return false;
            }

            return settings.Animations.Any((anim) => CheckNotExportAnimationTag(settings.GetOwnerSubScene(), anim.TagName));
        }

        /// <summary>
        /// アニメーションが出力されないアニメーション区間タグ設定があるかどうか
        /// </summary>
        public static bool CheckNotExportAnimationTag(ISubScene subScene, string animTagName)
        {
            IAnimFrameSection tag = subScene.IAnimFrameSectionSet.FindAnimFrameSectionByName(animTagName);
            if (tag == null)
            {
                return false;
            }

            ParamaterKind paramKind = AnimFrameSectionSetHelper.SearchOutBinayParamaterKindFromKeysInTargetTag(subScene, tag, true);
            return (paramKind == ParamaterKind.None);
        }

        /// <summary>
        /// 不正なアニメーション区間タグ設定があるかどうか
        /// </summary>
        public static bool ContainsInvalidAnimationTag(this IControlSettings settings)
        {
            return ContainsUnsetAnimationTag(settings) || ContainsNotFoundAnimationTag(settings) || ContainsNotExportAnimationTag(settings);
        }

        /// <summary>
        /// 未設定のペイン設定があるかどうか
        /// </summary>
        public static bool ContainsUnsetPane(this IControlSettings settings)
        {
            return settings.Panes.Any((pane) => pane.IsRequired && string.IsNullOrEmpty(pane.PaneName));
        }

        /// <summary>
        /// 設定が見つからないペイン設定があるかどうか
        /// </summary>
        public static bool ContainsNotFoundPane(this IControlSettings settings)
        {
            return settings.Panes.Any((pane) => pane.IsRequired && settings.GetOwnerSubScene().FindPaneByName(pane.PaneName) == null);
        }

        /// <summary>
        /// 不正なアニメーション区間タグ設定があるかどうか
        /// </summary>
        public static bool ContainsInvalidPane(this IControlSettings settings)
        {
            return ContainsUnsetPane(settings) || ContainsNotFoundPane(settings);
        }

        /// <summary>
        /// サイズがゼロになっていないか？
        /// </summary>
        public static ISubScene GetOwnerSubScene(this IControlSettings settings)
        {
            return settings.Owner.Owner;
        }

        /// <summary>
        /// サイズがゼロになっていないか？
        /// </summary>
        public static bool IsPartsSizeZero(this IPartsSettings partsSettings)
        {
            var size = PartsLayoutHelper.CalcPartsPaneSizeFromRectangle(PartsLayoutHelper.CalcPartsPaneRectanbleFromPanes(partsSettings.Owner.IPaneArray));
            return FVec2.Empty == size;
        }
    }

    /// <summary>
    /// ヘルパー
    /// </summary>
    public static class ControlSettingsSetHelper
    {
        /// <summary>
        /// ユニークなコントロール名を取得します。
        /// </summary>
        public static string GetUniqueControlSettingsName(this IControlSettingsSet controlSettingsSet, string origName)
        {
            if (!PaneHelper.CheckPaneNameValid(origName))
            {
                return AppConstants.RegexInvalidCharForObjName.Replace(origName, "");
            }

            return SubSceneHelper.GetValidName(
                (name) => controlSettingsSet.Items.FirstOrDefault((settings) => settings.DescriptionName == name),
                AppConstants.MaxControlNameLength,
                origName);
        }

    }
}
