﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;

namespace LECore.Structures
{
    using LECore.Structures.Core;
    using System.Linq;

    /// <summary>
    ///
    /// </summary>
    public static class MaterialHelper
    {
        //----------------------------------------------------------
        // アニメーションの対象となるアトリビュートのセットの取得
        //----------------------------------------------------------

        /// <summary>
        /// アニメーション要素列の取得
        /// </summary>
        public static IAnmAttribute[] GetAnimationTargetAttributeSet(this IMaterial material, ParamaterKind kind)
        {
            IAnmAttribute[] attributes = new IAnmAttribute[0];
            bool isMaterialAnimationKind = true; // Assertチェック用

            switch (kind)
            {
                case ParamaterKind.Animation_TexturePattern: attributes = GetAnimAttrSet_TexturePattern(material); break;
                case ParamaterKind.Animation_TextureSRT: attributes = GetAnimAttrSet_TextureSRT(material); break;
                case ParamaterKind.Animation_MaterialColors: attributes = GetAnimAttrSet_MaterialColor(material); break;
                default: isMaterialAnimationKind = false; break;
            }

            // 発見されなかった
            Debug.Assert(isMaterialAnimationKind == ParamaterKindHelper.MaterialAnimationKindSet[0].Contains(kind)); // 追加忘れ防止用チェック
            return attributes;
        }

        /// <summary>
        /// アニメーション要素列の取得
        /// </summary>
        public static IAnmAttribute[] GetAnimationTargetAttributeSet(this IRevHWMaterial material, ParamaterKind kind)
        {
            IAnmAttribute[] attributes = new IAnmAttribute[0];
            bool isMaterialAnimationKind = true; // Assertチェック用

            switch (kind)
            {
                case ParamaterKind.Animation_MaterialColors: attributes = GetAnimAttrSet_TevConstantColors(material); break;
                case ParamaterKind.Animation_IndirectTextureSRT: attributes = GetAnimAttrSet_TextureSRT_Indirect(material); break;
                case ParamaterKind.Animation_AlphaTest: attributes = new IAnmAttribute[] { material.IPEData.ACompare.RefAnmAttr }; break;
                default: isMaterialAnimationKind = false; break;
            }

            // 発見されなかった
            Debug.Assert(isMaterialAnimationKind == ParamaterKindHelper.MaterialAnimationKindSet[1].Contains(kind)); // 追加忘れ防止用チェック
            return attributes;
        }

        /// <summary>
        /// テクスチャブレンド比
        /// </summary>
        private static IAnmAttribute[] GetAnimAttrSet_MaterialColor(this IMaterial material)
        {
            List<IAnmAttribute> attrSet = new List<IAnmAttribute>();

            attrSet.Add(material.BlackColorIAnmAttr);
            attrSet.Add(material.WhiteColorIAnmAttr);

            return attrSet.ToArray();
        }

        /// <summary>
        /// テクスチャパターン
        /// </summary>
        private static IAnmAttribute[] GetAnimAttrSet_TexturePattern(this IMaterial material)
        {
            List<IAnmAttribute> attrSet = new List<IAnmAttribute>();

            foreach (IMaterialTexMap texMap in material.IMaterialTexMapSet)
            {
                attrSet.Add(texMap.ITextureChanger.TextureIndexIAnmAttr);
            }

            return attrSet.ToArray();
        }

        /// <summary>
        /// テクスチャSRT
        /// </summary>
        private static IAnmAttribute[] GetAnimAttrSet_TextureSRT(this IMaterial material)
        {
            List<IAnmAttribute> attrSet = new List<IAnmAttribute>();


            foreach (ITexMtx texMtx in material.ITexMtxSet)
            {
                Debug.Assert(texMtx != null);
                attrSet.Add(texMtx.TransIAnmAttr);
                attrSet.Add(texMtx.ScaleIAnmAttr);
                attrSet.Add(texMtx.RotateIAnmAttr);
            }

            return attrSet.ToArray();
        }

        /// <summary>
        /// テクスチャSRT
        /// </summary>
        private static IAnmAttribute[] GetAnimAttrSet_TextureSRT_Indirect(this IRevHWMaterial revMtl)
        {
            List<IAnmAttribute> attrSet = new List<IAnmAttribute>();
            //
            // 詳細情報からインダイレクト行列の出力を試みます。
            //
            for (int i = 0; i < revMtl.IIndirectMaterialTexMtxSet.Length; i++)
            {
                ITexMtx indTexMtx = revMtl.IIndirectMaterialTexMtxSet[i];

                attrSet.Add(indTexMtx.TransIAnmAttr);
                attrSet.Add(indTexMtx.ScaleIAnmAttr);
                attrSet.Add(indTexMtx.RotateIAnmAttr);
            }
            return attrSet.ToArray();
        }

        /// <summary>
        /// コンスタントカラー
        /// </summary>
        private static IAnmAttribute[] GetAnimAttrSet_TevConstantColors(this IRevHWMaterial irevMtl)
        {
            var revMtl = irevMtl as RevHWMaterial;
            if (revMtl == null || revMtl.TevData == null || revMtl.TevData.GetTevColorAttr(0) == null)
            {
                return new IAnmAttribute[0];
            }

            List<IAnmAttribute> attrSet = new List<IAnmAttribute>();

            // tev定数レジスタ0
            attrSet.Add(revMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(0));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(1));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(2));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(3));
            // tev定数レジスタ1
            attrSet.Add(revMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(0));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(1));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(2));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(3));
            // tev定数レジスタ2
            attrSet.Add(revMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(0));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(1));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(2));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(3));
            // tev定数レジスタ3
            attrSet.Add(revMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(0));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(1));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(2));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(3));
            // tev定数レジスタ4
            attrSet.Add(revMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(0));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(1));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(2));
            attrSet.Add(revMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(3));

            return attrSet.ToArray();
        }


        //----------------------------------------------------------
        // デフォルトへのリセット
        //----------------------------------------------------------

        //----------------------------------------------------------
        // 利用調査
        //----------------------------------------------------------

        /// <summary>
        /// 指定テクスチャ名のテクスチャが使用されているか調査します。
        /// </summary>
        public static bool UsesTextureImage(this IMaterial material, string texName)
        {
            string[] nameSet = MaterialHelper.GetAllTextureName(material);
            return Array.IndexOf(nameSet, texName) != -1;
        }

        /// <summary>
        /// マテリアルが関連するテクスチャ名列を取得します。
        /// </summary>
        public static string[] GetAllTextureName(this IMaterial material)
        {
            List<string> textureNameSet = new List<string>();

            foreach (IMaterialTexMap texMap in material.IMaterialTexMapSet)
            {
                // マテリアルテクスチャに登録されていなか？
                if (!textureNameSet.Contains(texMap.TexImgName))
                {
                    textureNameSet.Add(texMap.TexImgName);
                }


                // テクスチャリスト内で使用されていないか？
                string[] subSetTexNameSet = texMap.ITextureChanger.ITextureSubSet.TextureSubSetNameSet;
                foreach (string subSetTexName in subSetTexNameSet)
                {
                    if (!textureNameSet.Contains(subSetTexName))
                    {
                        textureNameSet.Add(subSetTexName);
                    }
                }
            }

            return textureNameSet.ToArray();
        }

        /// <summary>
        /// テクスチャパターンアニメーションを持っているかどうか
        /// </summary>
        static public bool HasAnyAnimationOrTexturePatternTexMap(this IMaterial material)
        {
            for (int i = 0; i < material.IMaterialTexMapSet.Length; i++)
            {
                IMaterialTexMap texMap = material.IMaterialTexMapSet[i];
                if (texMap != null && texMap.ITextureChanger.HasAnyTexturePatternAnimationOrTextureSubSet())
                {
                    return true;
                }
            }

            return false;
        }

        //----------------------------------------------------------
        // その他
        //----------------------------------------------------------

        /// <summary>
        /// テクスチャの取得
        /// </summary>
        public static IMaterialTexMap GetMaterialTexMap(this IMaterial material, int index)
        {
            if (index >= 0 && index < material.IMaterialTexMapSet.Length)
            {
                return material.IMaterialTexMapSet[index];
            }

            return null;
        }
    }
}
