﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.Windows;
using System.Collections.Generic;
using System.Linq;

namespace LECore.Structures
{
    #region LEGroupMgrModifyEventArgs
    /// <summary>
    /// グループマネージャ変更通知イベントパラメータ
    /// </summary>
    internal class LEGroupMgrModifyEventArgs : EventArgs
    {
        public struct GroupMgrParamaters
        {
            readonly public LEGroupMgr GroupMgr;
            readonly public LEGroup Group;
            readonly public IPane MemberPane;
            readonly public int OldIndex;
            readonly public int NewIndex;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public GroupMgrParamaters(
                LEGroupMgr groupMgr,
                LEGroup group,
                IPane member,
                int oldIndex,
                int newIndex )
            {
                GroupMgr = groupMgr;
                Group = group;
                MemberPane = member;
                OldIndex = oldIndex;
                NewIndex = newIndex;
            }

            /// <summary>
            /// グループを名前から取得します。
            /// </summary>
            public LEGroup FindGroupByName()
            {
                Debug.Assert( GroupMgr != null );
                Debug.Assert( Group != null );

                LEGroup group = GroupMgr.FindGroupByName( Group.GrouprName ) as LEGroup;
                Debug.Assert( group != null );

                return group;
            }
        }

        readonly SceneModifyEventArgs.Kind _kind;
        readonly GroupMgrParamaters _param;

        public SceneModifyEventArgs.Kind EventKind { get { return _kind; } }
        public GroupMgrParamaters GroupMgrParams { get { return _param; } }


        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEGroupMgrModifyEventArgs( SceneModifyEventArgs.Kind kind, GroupMgrParamaters param )
        {
            _kind = kind;
            _param = param;
        }

        /// <summary>
        /// グループ追加イベントの生成。
        /// </summary>
        static public LEGroupMgrModifyEventArgs
            MakeAddEvent( LEGroupMgr groupMgr, LEGroup owner )
        {
            GroupMgrParamaters paramaters =
                new GroupMgrParamaters( groupMgr, owner, null, 0, 0 );

            return new LEGroupMgrModifyEventArgs( SceneModifyEventArgs.Kind.GroupMake, paramaters );
        }

        /// <summary>
        /// グループ削除イベントの生成。
        /// </summary>
        static public LEGroupMgrModifyEventArgs
            MakeRemoveEvent( LEGroupMgr groupMgr, LEGroup owner, int index )
        {
            Debug.Assert( index != -1 );

            GroupMgrParamaters paramaters =
                new GroupMgrParamaters( groupMgr, owner, null, index, 0 );

            return new LEGroupMgrModifyEventArgs( SceneModifyEventArgs.Kind.GroupRemove, paramaters );
        }

        /// <summary>
        /// グループ順変イベントの生成。
        /// </summary>
        static public LEGroupMgrModifyEventArgs
            MakeEditGroupOrderEvent( LEGroupMgr groupMgr, LEGroup owner, int oldIndex, int newIndex )
        {
            // 新しい位置が、元の位置より前なら、元の位置は1ずれます。
            if( newIndex < oldIndex )
            {
                oldIndex++;
            }

            GroupMgrParamaters paramaters =
                new GroupMgrParamaters( groupMgr, owner, null, oldIndex, newIndex );

            return new LEGroupMgrModifyEventArgs( SceneModifyEventArgs.Kind.GroupReorder, paramaters );
        }

        /// <summary>
        /// グループへのペイン追加イベントの生成。
        /// </summary>
        static public LEGroupMgrModifyEventArgs
            MakePaneAddEvent( LEGroupMgr groupMgr, LEGroup owner, IPane addedMember )
        {
            GroupMgrParamaters paramaters =
                new GroupMgrParamaters( groupMgr, owner, addedMember, 0, 0 );

            return new LEGroupMgrModifyEventArgs( SceneModifyEventArgs.Kind.GroupAddPane, paramaters );
        }

        /// <summary>
        /// グループからのペイン削除イベントの生成。
        /// </summary>
        static public LEGroupMgrModifyEventArgs
            MakePaneRemoveEvent( LEGroupMgr groupMgr, LEGroup owner, IPane removedMember, int oldIndex )
        {
            GroupMgrParamaters paramaters =
                new GroupMgrParamaters( groupMgr, owner, removedMember, oldIndex, 0 );

            return new LEGroupMgrModifyEventArgs( SceneModifyEventArgs.Kind.GroupRemovePane, paramaters );
        }

        /// <summary>
        /// グループペイン順変更イベントの生成。
        /// </summary>
        static public LEGroupMgrModifyEventArgs
            MakeEditPaneOrderEvent( LEGroupMgr groupMgr, LEGroup owner, IPane pane, int oldIndex, int newIndex )
        {
            // 新しい位置が、元の位置より前なら、元の位置は1ずれます。
            if( newIndex < oldIndex )
            {
                oldIndex++;
            }

            GroupMgrParamaters paramaters =
                new GroupMgrParamaters(
                groupMgr,
                owner,
                pane,
                oldIndex,
                newIndex );

            return new LEGroupMgrModifyEventArgs( SceneModifyEventArgs.Kind.GroupReorderPane, paramaters );
        }

        /// <summary>
        /// グループペイン自身(名前、コメント)変更イベントの生成。
        /// </summary>
        static public LEGroupMgrModifyEventArgs
            MakeEditGroupEvent( LEGroupMgr groupMgr, LEGroup modifiedGroup )
        {
            GroupMgrParamaters paramaters =
                new GroupMgrParamaters( groupMgr, modifiedGroup, null, 0, 0 );

            return new LEGroupMgrModifyEventArgs( SceneModifyEventArgs.Kind.GroupModify, paramaters );
        }
    }
    #endregion LEGroupMgrModifyEventArgs

    /// <summary>
    /// LEGroupMgr の概要の説明です。
    /// サブシーンに1つ実体が存在するクラスです。
    ///
    /// </summary>
    internal class LEGroupMgr :
        ILEGroupMgr
    {
        #region フィールド
        public const string RootGroupName = "RootGroup";

        /// <summary>
        /// GroupModify 配列
        /// </summary>
        readonly List<ILEGroup> _groupSet = new List<ILEGroup>();
        readonly List<ILEGroup> _readOnlyGroupSet = new List<ILEGroup>();

        /// <summary>
        /// グループマネージャ更新イベント
        /// </summary>
        public event System.EventHandler OnGroupMgrModify;
        readonly LEGroup.OnChnageHandler _OnGroupChangeHandler;
        readonly ISubScene _ownerSubScene;
        #endregion フィールド

        #region プロパティ

        /// <summary>
        /// 管理しているグループセット
        /// </summary>
        public ILEGroup[] GroupSet
        {
            get { return _groupSet.ToArray() as ILEGroup[]; }
        }

        /// <summary>
        /// ルートノードの名前を取得します。
        /// </summary>
        public string RootNodeName
        {
            get { return RootGroupName; }
        }
        #endregion プロパティ

        #region イベントメッセージ通知
        /// <summary>
        /// グループ更新ハンドラ
        /// </summary>
        void OnGroupChangeHandler_( EventArgs args )
        {
            if(OnGroupMgrModify != null)
            {
                OnGroupMgrModify(this, args);
            }
        }

        #endregion イベントメッセージ通知

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEGroupMgr(ISubScene owner)
        {
            _OnGroupChangeHandler = OnGroupChangeHandler_;
            _ownerSubScene = owner;
        }

        #region グループの追加/削除
        /// <summary>
        /// グループを新規に登録します。
        /// 同一内容のグループが許可されるように変更になりました。
        ///
        /// 空のグループが許可されるように変更になりました
        /// (Undo/Redo動作の際に、必要となりました)。
        /// </summary>
        public LEGroup MakeNewGroup(string name, IPane[] members)
        {
            // 同名のグループが存在する場合登録を中断します。
            if (FindGroupByName(name) != null)
            {
                Debug.Assert(false);
                return null;
            }

            // 新規作成します。
            LEGroup newGroup = new LEGroup(name, this, _OnGroupChangeHandler);

            // グループ登録
            _groupSet.Add(newGroup);

            // グループにメンバの登録
            if (members != null)
            {
                foreach (IPane member in members)
                {
                    newGroup.AddMember(member, false);
                }
            }

            // 更新イベントを送信します。
            LEGroupMgrModifyEventArgs mgrArgs =
                LEGroupMgrModifyEventArgs.MakeAddEvent(this, newGroup);
            OnGroupMgrModify(this, mgrArgs);

            return newGroup;
        }


        /// <summary>
        /// 空のグループを登録削除します。
        /// </summary>
        public void RemoveEmptyGroup()
        {   // 消去の結果空になったグループを消去します。
            ILEGroup[] groupSet = this.GroupSet;
            foreach( ILEGroup group in groupSet )
            {
                if( !group.HasMember )
                {
                    bool usedByAnyTag = _ownerSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.Any(
                        (tag) => AnimFrameSectionHelper.ExistTargetByName(tag, group.GrouprName));

                    if (!usedByAnyTag)
                    {
                        RemoveGroup(group);
                    }
                }
            }
        }

        /// <summary>
        /// すべてのグループから指定メンバを削除します。
        /// </summary>
        public void RemoveMemberFromAll( IPane member )
        {
            // すべてのグループについて
            // 自身が、消去対象なら消去します。
            foreach (LEGroup group in _groupSet.ToArray())
            {
                group.RemoveMember(member);
            }
        }

        /// <summary>
        /// グループを削除します。
        /// </summary>
        public void RemoveGroup( ILEGroup group )
        {
            Debug.Assert( group != null );

            int idx = _groupSet.IndexOf( group );

            _groupSet.Remove(group);
            SetReadOnlyState(group as LEGroup, false);

            // 更新イベントを送信します。
            LEGroupMgrModifyEventArgs mgrArgs =
                LEGroupMgrModifyEventArgs.MakeRemoveEvent( this, group as LEGroup, idx );
            OnGroupMgrModify( this, mgrArgs );
        }

        /// <summary>
        /// グループの順番を変更します。
        /// </summary>
        public int ChangeGroupOrder( ILEGroup group, int newIndex )
        {
            Debug.Assert( _groupSet.Contains( group ) );
            // int currentIndex = _groupSet.IndexOf( group );
            int currentIndex = _groupSet.IndexOf( FindGroupByName( group.GrouprName ) );


            // 挿入し、現在のインスタンスを削除します。
            if( currentIndex != newIndex )
            {
                _groupSet.Insert( newIndex, group );

                if( currentIndex > newIndex )
                {
                    _groupSet.RemoveAt( currentIndex + 1 );
                }
                else
                {
                    _groupSet.RemoveAt( currentIndex );
                }

                // 更新イベントを送信します。
                LEGroupMgrModifyEventArgs mgrArgs =
                    LEGroupMgrModifyEventArgs.MakeEditGroupOrderEvent(
                    this, group as LEGroup, currentIndex, newIndex );

                OnGroupMgrModify( this, mgrArgs );

            }

            int nextIndex = ( currentIndex > newIndex ) ? newIndex + 1 : newIndex;
            return nextIndex;
        }
        #endregion グループの追加/削除

        #region グループの検索

        /// <summary>
        /// 名前でグループを検索します。
        /// </summary>
        public ILEGroup FindGroupByName( string name )
        {
            foreach( LEGroup group in _groupSet )
            {
                if( group.GrouprName == name )
                {
                    return group;
                }
            }
            return null;
        }

        /// <summary>
        /// 読みとり専用状態を設定します。
        /// <remarks>
        /// ユーザのUI 操作によって実行されないメソッドなの
        /// で、UIへの変更通知をしていません。
        /// </remarks>
        /// </summary>
        public void SetReadOnlyState(LEGroup group, bool isReadonly)
        {
            if (!this._groupSet.Contains(group))
            {
                return;
            }

            if (isReadonly)
            {
                if (!_readOnlyGroupSet.Contains(group))
                {
                    _readOnlyGroupSet.Add(group);
                }
            }
            else
            {
                _readOnlyGroupSet.Remove(group);
            }
        }

        /// <summary>
        /// グループが読み取り専用か調べます。
        /// </summary>
        public bool CheckReadOnlyByName(string name)
        {
            var group = FindGroupByName(name);
            return group != null ? _readOnlyGroupSet.Contains(group) : false;
        }

        /// <summary>
        /// 同一内容のグループがすでに登録されていないか確認します。
        /// </summary>
        public void FindSameGroup( IPane[] paneSet, ArrayList sameGroupSet )
        {
            // 同一内容のグループがすでに登録されていないか確認します。
            foreach( LEGroup group in _groupSet )
            {
                if( group.IsSame( paneSet ) )
                {
                    sameGroupSet.Add( group );
                }
            }
        }

        /// <summary>
        /// グループの番号を取得します。
        /// </summary>
        public int GetGroupIndex( ILEGroup group )
        {
            return _groupSet.IndexOf( group );
        }

        #endregion グループの検索
    }
}
