﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using LECore.Structures.LECoreInterface;

namespace LECore.Structures
{
    using Core;
    using LECore.Structures.Nsrif.Attributes;
    using LECore.Manipulator;

    internal class DropShadow :
        LEDataNode,
        IDropShadow,
        IDisposable
    {
        public const string DropShadowNodeName = "DropShadow";

        readonly DropShadowAttribute _attribute = null;

        readonly Pane _ownerPane = null;
        bool _isKnockoutEnabled = true;
        bool _isOnlyEffectEnabled = false;
        bool _isStaticRenderingEnabled = false;
        bool _isStrokeEnabled = false;
        DropShadowBlendMode _strokeBlendMode = DropShadowBlendMode.Normal;
        bool _isOuterGlowEnabled = false;
        DropShadowBlendMode _outerGlowBlendMode = DropShadowBlendMode.Addition;
        bool _isDropShadowEnabled = false;
        DropShadowBlendMode _dropShadowBlendMode = DropShadowBlendMode.Multiply;

        public DropShadow(Pane ownerPane)
            : base(ownerPane, DropShadowNodeName)
        {
            System.Diagnostics.Debug.Assert(ownerPane != null);

            _ownerPane = ownerPane;
            _ownerPane.BindPaneDropShadow(this);

            _attribute = new Structures.DropShadow.DropShadowAttribute(this);
        }

        public void Dispose()
        {
            _attribute.Dispose();
        }

        /// <summary>
        /// 変更をシーンに通知します。
        /// </summary>
        void NotifyChangeToScene_()
        {
            NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }

        #region IDropShadow


        public IPane OwnerPane
        {
            get
            {
                return _ownerPane;
            }
        }

        public bool IsDropShadowEtcEnabled
        {
            get
            {
                return IsStrokeEnabled || IsOuterGlowEnabled || IsDropShadowEnabled;
            }
        }

        public bool IsKnockoutEnabled
        {
            set
            {
                if (_isKnockoutEnabled != value)
                {
                    _isKnockoutEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isKnockoutEnabled;
            }
        }

        public bool IsOnlyEffectEnabled
        {
            set
            {
                if (_isOnlyEffectEnabled != value)
                {
                    _isOnlyEffectEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isOnlyEffectEnabled;
            }
        }

        public bool IsStaticRenderingEnabled
        {
            set
            {
                if (_isStaticRenderingEnabled != value)
                {
                    _isStaticRenderingEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isStaticRenderingEnabled;
            }
        }

        public bool IsStrokeEnabled
        {
            set
            {
                if (_isStrokeEnabled != value)
                {
                    _isStrokeEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isStrokeEnabled;
            }
        }

        public float strokeSize
        {
            set
            {
                _attribute.StrokeSizeAttr.SetValue(value);
            }

            get
            {
                return _attribute.StrokeSizeAttr.GetAsFloat();
            }
        }

        public DropShadowBlendMode strokeBlendMode
        {
            set
            {
                if (_strokeBlendMode != value)
                {
                    _strokeBlendMode = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _strokeBlendMode;
            }
        }

        public RGBAColor strokeColor
        {
            set
            {
                _attribute.StrokeColorAttr.SetValue(value);
            }

            get
            {
                return _attribute.StrokeColorAttr.GetAsRBGA();
            }
        }

        public bool IsOuterGlowEnabled
        {
            set
            {
                if (_isOuterGlowEnabled != value)
                {
                    _isOuterGlowEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isOuterGlowEnabled;
            }
        }

        public DropShadowBlendMode outerGlowBlendMode
        {
            set
            {
                if (_outerGlowBlendMode != value)
                {
                    _outerGlowBlendMode = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _outerGlowBlendMode;
            }
        }

        public RGBAColor outerGlowColor
        {
            set
            {
                _attribute.OuterGlowColorAttr.SetValue(value);
            }

            get
            {
                return _attribute.OuterGlowColorAttr.GetAsRBGA();
            }
        }

        public float outerGlowSpread
        {
            set
            {
                _attribute.OuterGlowSpreadAttr.SetValue(value);
            }

            get
            {
                return _attribute.OuterGlowSpreadAttr.GetAsFloat();
            }
        }

        public float outerGlowSize
        {
            set
            {
                _attribute.OuterGlowSizeAttr.SetValue(value);
            }

            get
            {
                return _attribute.OuterGlowSizeAttr.GetAsFloat();
            }
        }

        public bool IsDropShadowEnabled
        {
            set
            {
                if (_isDropShadowEnabled != value)
                {
                    _isDropShadowEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isDropShadowEnabled;
            }
        }

        public DropShadowBlendMode dropShadowBlendMode
        {
            set
            {
                if (_dropShadowBlendMode != value)
                {
                    _dropShadowBlendMode = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _dropShadowBlendMode;
            }
        }

        public RGBAColor dropShadowColor
        {
            set
            {
                _attribute.DropShadowColorAttr.SetValue(value);
            }

            get
            {
                return _attribute.DropShadowColorAttr.GetAsRBGA();
            }
        }

        public float dropShadowAngle
        {
            set
            {
                _attribute.DropShadowAngleAttr.SetValue(value);
            }

            get
            {
                return _attribute.DropShadowAngleAttr.GetAsFloat();
            }
        }

        public float dropShadowDistance
        {
            set
            {
                _attribute.DropShadowDistanceAttr.SetValue(value);
            }

            get
            {
                return _attribute.DropShadowDistanceAttr.GetAsFloat();
            }
        }

        public float dropShadowSpread
        {
            set
            {
                _attribute.DropShadowSpreadAttr.SetValue(value);
            }

            get
            {
                return _attribute.DropShadowSpreadAttr.GetAsFloat();
            }
        }

        public float dropShadowSize
        {
            set
            {
                _attribute.DropShadowSizeAttr.SetValue(value);
            }

            get
            {
                return _attribute.DropShadowSizeAttr.GetAsFloat();
            }
        }

        public AnmAttribute StrokeSizeAttr
        {
            get
            {
                return _attribute.StrokeSizeAttr;
            }
        }

        public AnmAttribute StrokeColorAttr
        {
            get
            {
                return _attribute.StrokeColorAttr;
            }
        }

        public AnmAttribute OuterGlowColorAttr
        {
            get
            {
                return _attribute.OuterGlowColorAttr;
            }
        }

        public AnmAttribute OuterGlowSpreadAttr
        {
            get
            {
                return _attribute.OuterGlowSpreadAttr;
            }
        }

        public AnmAttribute OuterGlowSizeAttr
        {
            get
            {
                return _attribute.OuterGlowSizeAttr;
            }
        }

        public AnmAttribute DropShadowColorAttr
        {
            get
            {
                return _attribute.DropShadowColorAttr;
            }
        }

        public AnmAttribute DropShadowAngleAttr
        {
            get
            {
                return _attribute.DropShadowAngleAttr;
            }
        }

        public AnmAttribute DropShadowDistanceAttr
        {
            get
            {
                return _attribute.DropShadowDistanceAttr;
            }
        }

        public AnmAttribute DropShadowSpreadAttr
        {
            get
            {
                return _attribute.DropShadowSpreadAttr;
            }
        }

        public AnmAttribute DropShadowSizeAttr
        {
            get
            {
                return _attribute.DropShadowSizeAttr;
            }
        }

        public IAnmAttribute IStrokeSizeAttr
        {
            get
            {
                return _attribute.StrokeSizeAttr;
            }
        }

        public IAnmAttribute IStrokeColorAttr
        {
            get
            {
                return _attribute.StrokeColorAttr;
            }
        }

        public IAnmAttribute IOuterGlowColorAttr
        {
            get
            {
                return _attribute.OuterGlowColorAttr;
            }
        }

        public IAnmAttribute IOuterGlowSpreadAttr
        {
            get
            {
                return _attribute.OuterGlowSpreadAttr;
            }
        }

        public IAnmAttribute IOuterGlowSizeAttr
        {
            get
            {
                return _attribute.OuterGlowSizeAttr;
            }
        }

        public IAnmAttribute IDropShadowColorAttr
        {
            get
            {
                return _attribute.DropShadowColorAttr;
            }
        }

        public IAnmAttribute IDropShadowAngleAttr
        {
            get
            {
                return _attribute.DropShadowAngleAttr;
            }
        }

        public IAnmAttribute IDropShadowDistanceAttr
        {
            get
            {
                return _attribute.DropShadowDistanceAttr;
            }
        }

        public IAnmAttribute IDropShadowSpreadAttr
        {
            get
            {
                return _attribute.DropShadowSpreadAttr;
            }
        }

        public IAnmAttribute IDropShadowSizeAttr
        {
            get
            {
                return _attribute.DropShadowSizeAttr;
            }
        }
        /// <summary>
        /// ドロップシャドウ用のキャプチャテクスチャを作成します。
        /// </summary>
        public void CreateCaptureTextureForDropShadow(IPane pane, ITextureMgr textureMgr)
        {
            TextureMgrManipulator mnp = new TextureMgrManipulator();
            mnp.BindTarget(textureMgr);
            CaptureTextureManipulator captureTextureMnp = new CaptureTextureManipulator();
            ICaptureTexture captureTexture = mnp.RegisterCaptureTexture(pane, pane.PaneName, true, CaptureTextureUsage.DropShadow).ICaptureTexture;
            captureTextureMnp.BindTarget(captureTexture);
            captureTextureMnp.ClearColor = new FloatColor(0.0f, 0.0f, 0.0f, 0.0f);
            captureTextureMnp.FrameBufferCaptureEnabled = false;
            captureTextureMnp.CaptureOnlyFirstFrame = IsStaticRenderingEnabled;
        }
        #endregion

        internal class DropShadowAttribute :
            AnmAttribute
        {
            static readonly AnmAttrDescripter[] SubAttrDescs =
            {
                new AnmAttrDescripter(AttributeType.Float, "strokeSize", null, 3.0f),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "strokeColor", null, RGBAColor.Black),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "outerGlowColor", null, RGBAColor.White),
                new AnmAttrDescripter(AttributeType.Float, "outerGlowSpread", null, 0.0f),
                new AnmAttrDescripter(AttributeType.Float, "outerGlowSize", null, 5.0f),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "dropShadowColor", null, RGBAColor.Black),
                new AnmAttrDescripter(AttributeType.Float, "dropShadowAngle", null, 120.0f),
                new AnmAttrDescripter(AttributeType.Float, "dropShadowDistance", null, 5.0f),
                new AnmAttrDescripter(AttributeType.Float, "dropShadowSpread", null, 0.0f),
                new AnmAttrDescripter(AttributeType.Float, "dropShadowSize", null, 5.0f)
            };

            public DropShadowAttribute(DropShadow ownerDropShadow)
                : base(ownerDropShadow, SubAttrDescs,
                      new AnmAttrDescripter(AttributeType.Combined, "DropShadow", null, null))
            {
            }

            public AnmAttribute StrokeSizeAttr
            {
                get
                {
                    return FindAttributeByIdx(0);
                }
            }

            public AnmAttribute StrokeColorAttr
            {
                get
                {
                    return FindAttributeByIdx(1);
                }
            }

            public AnmAttribute OuterGlowColorAttr
            {
                get
                {
                    return FindAttributeByIdx(2);
                }
            }

            public AnmAttribute OuterGlowSpreadAttr
            {
                get
                {
                    return FindAttributeByIdx(3);
                }
            }

            public AnmAttribute OuterGlowSizeAttr
            {
                get
                {
                    return FindAttributeByIdx(4);
                }
            }

            public AnmAttribute DropShadowColorAttr
            {
                get
                {
                    return FindAttributeByIdx(5);
                }
            }

            public AnmAttribute DropShadowAngleAttr
            {
                get
                {
                    return FindAttributeByIdx(6);
                }
            }

            public AnmAttribute DropShadowDistanceAttr
            {
                get
                {
                    return FindAttributeByIdx(7);
                }
            }

            public AnmAttribute DropShadowSpreadAttr
            {
                get
                {
                    return FindAttributeByIdx(8);
                }
            }

            public AnmAttribute DropShadowSizeAttr
            {
                get
                {
                    return FindAttributeByIdx(9);
                }
            }
        }
    }
}
