﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing.Drawing2D;
using System.Numerics;

namespace LECore.Structures.Core
{

    /// <summary>
    /// 位置情報を持つオブジェクトが実装するインタフェース
    ///
    /// 結合度を下げるため、可能な限り、本インターフェースに対して
    /// プルグラムをするようにします。
    /// </summary>
    public interface ILocationNode
    {
        /// <summary>
        /// 位置X
        /// </summary>
        float X
        {
            get;
            set;
        }

        /// <summary>
        /// 位置Y
        /// </summary>
        float Y
        {
            get;
            set;
        }

        /// <summary>
        /// 位置Z
        /// </summary>
        float Z
        {
            get;
            set;
        }

        /// <summary>
        /// 位置X
        /// </summary>
        float XInWorld
        {
            get;
        }

        /// <summary>
        /// 位置X
        /// </summary>
        float XInWorld4x4
        {
            get;
        }



        /// <summary>
        /// 位置Y
        /// </summary>
        float YInWorld
        {
            get;
        }

        /// <summary>
        /// 位置Y
        /// </summary>
        float YInWorld4x4
        {
            get;
        }

        /// <summary>
        /// 位置Z
        /// </summary>
        float ZInWorld4x4
        {
            get;
        }

        /// <summary>
        /// Z軸回転
        /// </summary>
        float RotAngleZ
        {
            get;
            set;
        }

        /// <summary>
        /// ローカル変換行列を計算します。
        /// </summary>
        Matrix LocalMtx
        {
            get;
        }

        /// <summary>
        /// ローカル変換行列を計算します。
        /// </summary>
        Matrix34 LocalMtx34
        {
            get;
        }

        /// <summary>
        /// ローカル変換行列を計算します。
        /// </summary>
        Matrix4x4 LocalMtx4x4
        {
            get;
        }
    }

    /// <summary>
    /// 階層構造を表現するインタフェースです。
    ///
    /// 本アプリケーションにおいては、Paneによって実装されます。
    /// 階層構造にアクセスするクラスは、可能な限り、Paneを参照せず、
    ///
    /// IHierarchyNode に対してプログラムをするようにする必要があります。
    ///
    /// </summary>
    public interface IHierarchyNode : ILocationNode
    {




        /// <summary>
        /// 親ノード
        /// </summary>
        IHierarchyNode Parent
        {
            get;
            set;
        }

        /// <summary>
        /// 子ノード
        /// </summary>
        IHierarchyNode[] Children
        {
            get;
        }

        /// <summary>
        /// ワールド変換行列を計算します。
        /// </summary>
        Matrix WorldMtx
        {
            get;
        }

        /// <summary>
        /// スケール値が描画に妥当か？
        /// </summary>
        bool ValidScaleForDraw { get;}

        /// <summary>
        /// 子供を持つか判定します。
        /// </summary>
        bool HasChildren{ get; }

        /// <summary>
        /// 子供ノードを追加する
        /// </summary>
        /// <param name="childNode"></param>
        void AddChildNodeByLoaclCoordinate( IHierarchyNode childNode );

        /// <summary>
        /// 子供ノードを削除する
        /// </summary>
        /// <param name="childNode"></param>
        void RemoveChildNode( IHierarchyNode childNode );

        /// <summary>
        /// ノードの番号を取得します。
        /// </summary>
        int IndexOf( IHierarchyNode childNode );

        /// <summary>
        /// 親子階層構造を破壊します。ペインを消去する際などに使用してください。
        /// </summary>
        void DestroyHierarchyConnection();

        /// <summary>
        /// 子ノードとして正しいか判定します。
        /// </summary>
        /// <param name="candidate"></param>
        /// <returns></returns>
        bool IsValidForChildOf( IHierarchyNode candidate );

        /// <summary>
        /// 自分が指定ノードの子供か判定します。
        /// </summary>
        bool IsChildrenOf( IHierarchyNode candidate );
    }
}
