﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Collections.Generic;


namespace LECore.Structures.Core
{
    using LECore.Structures;

    /// <summary>
    /// アトリビュート記述子
    ///
    /// アトリビュートインスタンスの構築の時に、
    /// コンストラクタの引数として使用されます。
    /// </summary>
    internal class AnmAttrDescripter
    {
        /// <summary>
        /// カーブの補間タイプを変更できるかどうか
        /// </summary>
        public enum AnimationTypeLock
        {
            True,
            False
        };

        public readonly AttributeType _type;
        public readonly string _name;
        public readonly AnmAttribute.AnmAttrCreateFunc _createFunc;
        public readonly object _initialValue;
        public readonly bool _lockAnimationInterpolateType;
        public readonly bool _useCustomTimeSource;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmAttrDescripter
        (
            AttributeType type,
            string name,
            AnmAttribute.AnmAttrCreateFunc createFunc,
            object initialValue
        )
            : this(type, name, createFunc, initialValue, AnimationTypeLock.False, false)
        {

        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmAttrDescripter
        (
            AttributeType type,
            string name,
            AnmAttribute.AnmAttrCreateFunc createFunc,
            object initialValue,
            AnimationTypeLock animationTypeLock,
            bool useCustomTimeSource
        )
        {
            _type = type;
            _name = name;
            _createFunc = createFunc;
            _initialValue = initialValue;
            _lockAnimationInterpolateType = animationTypeLock == AnimationTypeLock.True;
            _useCustomTimeSource = useCustomTimeSource;
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        public AnmAttrDescripter
        (
            AnmAttrDescripter src
        )
        {
            _type = src._type;
            _name = src._name;
            _createFunc = src._createFunc;
            _initialValue = src._initialValue;
            _lockAnimationInterpolateType = src._lockAnimationInterpolateType;
            _useCustomTimeSource = src._useCustomTimeSource;
        }
    }

    /// <summary>
    /// メッセージ伝播を行う基本クラス。
    /// </summary>
    internal class LEDataNode
        : IAnmAttribute
    {
        #region イベント
        public enum EventKind
        {
            Modify,
            AnimationModify,
            AnimationAddRemove,
            AnimationModeModify,
            AttributeModify, // アトリビュート変更。現状は、拡張ユーザー情報の追加時にしか利用されていません。
        }

        public delegate void LEDataNodeModifyHandler(LEDataNode sender, int eventKind);
        public event LEDataNodeModifyHandler OnModify;

        #endregion イベント

        #region フィールド

        string _nodeName = null;
        LEDataNode _ownerNode = null;
        string _description = string.Empty;
        bool _isActiveAttribute = true;
        bool _enableLocalize = true;
        /// <summary>
        /// サブデータノード配列
        /// </summary>
        readonly protected List<IAnmAttribute> _subAttrArray = new List<IAnmAttribute>();

        #endregion フィールド

        #region プロパティ

        /// <summary>
        /// 説明文章です。
        /// </summary>
        public string Description
        {
            get { return _description; }
            set { _description = value; }
        }

        /// <summary>
        /// 有効なアトリビュートか？
        /// </summary>
        public bool IsActiveAttribute
        {
            get { return _isActiveAttribute; }
            set
            {
                if (_isActiveAttribute != value)
                {
                    _isActiveAttribute = value;
                    NotifyModifyEvent(this, (int)EventKind.AnimationAddRemove);
                }
            }
        }

        /// <summary>
        /// ローカライズが有効か？
        /// </summary>
        public bool EnableLocalize
        {
            get { return _enableLocalize; }
            set { _enableLocalize = value; }
        }

        /// <summary>
        /// サブデータノード配列を取得します。
        /// </summary>
        public List<IAnmAttribute> SubAttrArray
        {
            get { return _subAttrArray; }
        }

        /// <summary>
        /// カレントのタグ名です。
        /// </summary>
        protected string CurrentTagName
        {
            get
            {
                return CurrentISubScene?.CurrentTagName ?? String.Empty;
            }
        }

        /// <summary>
        /// カレントのサブシーンです。
        /// </summary>
        protected ISubScene CurrentISubScene
        {
            get
            {
                return LECore.LayoutEditorCore.Scene.CurrentISubScene;
            }
        }

        #endregion プロパティ


        /// <summary>
        /// サブデータノードとして登録
        /// </summary>
        /// <param name="subNode"></param>
        protected void RegisterSubAttribute_(LEDataNode subNode)
        {
            Debug.Assert(subNode != null);
            // ArrayListに登録
            _subAttrArray.Add(subNode);
        }

        /// <summary>
        /// 変更イベントを通知します。
        ///
        /// 親ノードが存在する場合、
        /// 親ノードにも変更があったことを伝播します。
        /// </summary>
        public virtual void NotifyModifyEvent(LEDataNode sender, int kind)
        {
            // イベントを呼び出します。TODO:
            if (OnModify != null)
            {
                OnModify(sender, kind);
            }

            // 親ノードが存在する場合、
            // 親ノードにも変更があったことを伝播します。
            if (_ownerNode != null)
            {
                _ownerNode.NotifyModifyEvent(sender, kind);
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEDataNode(LEDataNode owner, string nodeName)
        {
            _ownerNode = owner;
            _nodeName = nodeName;

            // オーナーが指定されていた場合、
            // オーナーのサブアトリビュートとして自身を登録します。
            if (_ownerNode != null)
            {
                _ownerNode.RegisterSubAttribute_(this);
            }
        }

        #region IAnmAttribute メンバ
        public string Name
        {
            get { return _nodeName; }
            internal set { _nodeName = value; }
        }

        public string OwnerName
        {
            get { return (_ownerNode != null) ? _ownerNode.Name : string.Empty; }
        }

        public string DescriptionName
        {
            get
            {
                if (_description != string.Empty)
                {
                    return string.Format("{0} [{1}]", _nodeName, _description);
                }
                else
                {
                    return _nodeName;
                }
            }
        }

        public IAnmAttribute OwnerNode
        {
            get { return _ownerNode; }
            set { _ownerNode = value as LEDataNode; }
        }

        public virtual IAnmAttribute FindSubAttributeByIdx(int idx)
        {
            if (idx < this.NumSubAttribute)
            {
                if (_subAttrArray.Count != this.NumSubAttribute)
                {
                    return null;
                }

                return _subAttrArray[idx] as IAnmAttribute;
            }
            else
            {
                return null;
            }
        }

        #region 仮想関数実装（派生クラスでオーバーライドされます）
        public virtual bool HasAnimationCurve { get { return false; } }
        public virtual bool HasAnyKey { get { return false; } }
        public virtual int CurveNum { get { return 0; } }
        public virtual int NumSubAttribute { get { return _subAttrArray.Count; } }
        public virtual bool HasSubAttribute { get { return _subAttrArray.Count != 0; } }
        public virtual IAnmCurve ICurrentAnimationCurve { get { return null; } }
        public virtual IAnmCurve[] Curves { get { return null; } }
        public virtual IAnmCurve IDefaultCurve { get { return null; } }
        public virtual object Value { get { return null; } }
        #endregion

        #endregion
    }

    /// <summary>
    /// アトリビュート
    /// </summary>
    internal class AnmAttribute :
        LEDataNode,
        IDisposable,
        ITimeChageEventListener // 時間変更通知を受け取る
    {
        #region ------------- 定数 -------------
        static readonly public string[] PerCharacterParamName =
        {
            "CharTransformScale",
            "CharTransformTrans",
            "CharTransformRotate",
            "CharTransformTopColor",
            "CharTransformBottomColor"
        };

        static readonly AnmAttrDescripter[] IntVecDesc =
        {
            new AnmAttrDescripter( AttributeType.Int, "X", null, 0 ),
            new AnmAttrDescripter( AttributeType.Int, "Y", null, 0 )
        };

        static readonly AnmAttrDescripter[] FloatVecDesc =
        {
            new AnmAttrDescripter( AttributeType.Float, "X", null, 0.0f ),
            new AnmAttrDescripter( AttributeType.Float, "Y", null, 0.0f )
        };

        static readonly AnmAttrDescripter[] FloatVecUVDesc =
        {
            new AnmAttrDescripter( AttributeType.Float, "U", null, 0.0f ),
            new AnmAttrDescripter( AttributeType.Float, "V", null, 0.0f )
        };

        static readonly AnmAttrDescripter[] FloatVec3Desc =
        {
            new AnmAttrDescripter( AttributeType.Float, "X", null, 0.0f ),
            new AnmAttrDescripter( AttributeType.Float, "Y", null, 0.0f ),
            new AnmAttrDescripter( AttributeType.Float, "Z", null, 0.0f )
        };

        static readonly AnmAttrDescripter[] RGBA4Desc =
        {
            new AnmAttrDescripter( AttributeType.Byte, "R", null, 255 ),
            new AnmAttrDescripter( AttributeType.Byte, "G", null, 255 ),
            new AnmAttrDescripter( AttributeType.Byte, "B", null, 255 ),
            new AnmAttrDescripter( AttributeType.Byte, "A", null, 255 )
        };

        static readonly AnmAttrDescripter[] FloatRGBA4Desc =
        {
            new AnmAttrDescripter( AttributeType.Float, "R", null, 1.0f ),
            new AnmAttrDescripter( AttributeType.Float, "G", null, 1.0f ),
            new AnmAttrDescripter( AttributeType.Float, "B", null, 1.0f ),
            new AnmAttrDescripter( AttributeType.Float, "A", null, 1.0f )
        };

        #endregion ------------- 定数 -------------

        #region ----------------- フィールド -----------------
        // アニメーションカーブ
        private AnimationCurves _anmCurves = new AnimationCurves();

        // 持ち主のアトリビュート(サブアトリビュート以外でnullとなります。)
        readonly LEDataNode _ownerAttribute = null;

        // 型種類
        AttributeType _valueType;
        // アトリビュート値
        object _value;
        object _baseValue;

        bool _useCustomTimeSource;

        // 最大値
        float _maxValue;
        // 最小値
        float _minValue;


        #endregion // ----------------- フィールド -----------------

        #region ----------------- プロパティ -----------------
        /// <summary>
        /// アニメーションしているかを調べます。
        /// </summary>
        public override bool HasAnimationCurve
        {
            get { return (CurrentAnimationCurve != null); }
        }

        /// <summary>
        /// アニメーションカーブを取得します。
        /// null の場合もありえます。
        /// </summary>
        public override IAnmCurve ICurrentAnimationCurve
        {
            get
            {
                IAnmCurve curve = this._anmCurves.GetAnmCurve(this.CurrentTagName);
                return (curve != null) ? curve : this._anmCurves.DefaultCurve;
            }
        }

        /// <summary>
        /// アニメーションカーブを取得します。
        /// </summary>
        protected AnmCurve CurrentAnimationCurve
        {
            get
            {
                AnmCurve curve = this._anmCurves.GetAnmCurve(this.CurrentTagName);
                return (curve != null) ? curve : this._anmCurves.DefaultCurve;
            }
        }

        /// <summary>
        /// 所持しているカーブにキーが1つ以上存在するか。
        /// </summary>
        public override bool HasAnyKey
        {
            get { return _anmCurves.HasAnyKey; }
        }

        /// <summary>
        /// 参照カーブか。
        /// </summary>
        public bool IsReferenceCurves
        {
            get { return this.OwnerName != null && PerCharacterParamName.Contains(this.OwnerName); }
        }

        /// <summary>
        /// 所有しているアニメーションの数
        /// </summary>
        public override int CurveNum
        {
            get { return _anmCurves.CurveNum; }
        }

        /// <summary>
        /// カーブの配列を取得します。
        /// </summary>
        public override IAnmCurve[] Curves
        {
            get
            {
                return _anmCurves.Curves;
            }
        }

        /// <summary>
        /// デフォルトのアニメーションカーブを取得します。
        /// </summary>
        public override IAnmCurve IDefaultCurve
        {
            get
            {
                return _anmCurves.GetAnmCurve(String.Empty);
            }
        }

        /// <summary>
        /// デフォルトのアニメーションカーブを取得します。
        /// </summary>
        protected AnmCurve DefaultCurve
        {
            get
            {
                return _anmCurves.GetAnmCurve(String.Empty);
            }

            set
            {
                _anmCurves.RegistAnmCurve(AnimCurvesParam.__NoSelected__, value);
            }
        }

        /// <summary>
        /// アトリビュートの型
        /// </summary>
        public AttributeType Type
        {
            get { return _This._valueType; }
        }

        /// <summary>
        /// 持ち主のアトリビュート
        /// (自身がサブアトリビュート以外の場合は null になります)
        /// </summary>
        public LEDataNode OwnerAttribute
        {
            get { return _This._ownerAttribute; }
        }

        /// <summary>
        /// サブアトリビュートの数を取得します。
        /// </summary>
        public override int NumSubAttribute
        {
            get { return _This._subAttrArray.Count; }
        }

        /// <summary>
        /// サブアトリビュートを持つか？
        /// </summary>
        public override bool HasSubAttribute
        {
            get { return _This.NumSubAttribute != 0; }
        }

        /// <summary>
        /// 最大値です。
        /// </summary>
        public float MaxValue
        {
            get { return _This._maxValue; }
            set
            {
                if (_This.Type == AttributeType.Byte)
                {
                    value = Math.Max(Math.Min(value, 255), 0);
                }

                _This._maxValue = value;
            }
        }

        /// <summary>
        /// 最小値です。
        /// </summary>
        public float MinValue
        {
            get { return _This._minValue; }
            set
            {
                if (_This.Type == AttributeType.Byte)
                {
                    value = Math.Max(Math.Min(value, 255), 0);
                }

                _This._minValue = value;
            }
        }

        /// <summary>
        /// 見かけ上のスケール
        /// </summary>
        public float ViewScale { get; set; } = 1.0f;

        /// <summary>
        ///
        /// </summary>
        public override IAnmAttribute FindSubAttributeByIdx(int idx)
        {
            if (_valueType == AttributeType.RefAttribute)
            {
                return _This.FindAttributeByIdx(idx);
            }
            else
            {
                return base.FindSubAttributeByIdx(idx);
            }
        }

        /// <summary>
        /// アトリビュート実体の参照を取得します。
        /// 参照型のアトリビュートの場合は、自分自身ではなく、
        /// 初期化時に設定された、参照アトリビュートを返します。
        /// </summary>
        AnmAttribute _This
        {
            get
            {
                if (_valueType == AttributeType.RefAttribute)
                {
                    return _value as AnmAttribute;
                }
                else
                {
                    return this;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public InterporationType DefaultInterpType { get; set; } = InterporationType.None;

        /// <summary>
        ///
        /// </summary>
        public bool LockInterpType { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool IsReadOnlyLocked { get; set; }

        /// <summary>
        /// 時間変更ハンドラ
        /// </summary>
        public OnTimeChangedHandler TimeChangedHandler { get; set; }

        /// <summary>
        /// カスタム時間ソースからのイベントを中継する
        /// </summary>
        public TimeReleyProvider TimeReleyProvider { get; set; }

        /// <summary>
        /// カスタム時間ソース
        /// </summary>
        public ITimeChangeEventProvider TimeEventProvider { get; set; }

        /// <summary>
        /// baseValueを設定可能かどうか
        /// </summary>
        public bool CanSetBaseValue
        {
            get
            {
                return CurrentTagName == AnimCurvesParam.__NoSelected__;
            }
        }

        public bool UseCustomTimeSource
        {
            get { return _This._useCustomTimeSource; }
        }

        /// <summary>
        /// 現在値
        /// </summary>
        public override object Value
        {
            get
            {
                object val;
                GetValue(out val);
                return val;
            }
        }

        #endregion // ----------------- プロパティ -----------------

        #region ----------------- イベント -----------------


        #endregion ----------------- イベント -----------------

        #region ----------------- デリゲート -----------------
        // 複合型アトリビュートの初期化に使用されれる
        // インスタンス生成メソッドのデリゲート
        public delegate AnmAttribute AnmAttrCreateFunc(AnmAttribute owner, AnmAttrDescripter attrDesc);
        #endregion ----------------- デリゲート -----------------

        #region ----------------- アクセサ -----------------

        #region ----------------- 静的アクセサ：アニメーションカーブから使用されるもの -----------------
        /// <summary>
        /// int型で Enum 型に値を設定します。
        ///
        /// Enum 型で定義されていない、
        /// 不正なint型が指定された場合は、操作は無視されます。
        ///
        /// FIXME !（仮実装）
        /// </summary>
        /// <param name="newValue"></param>
        static bool SetIntToEnumValue_(ref object enumVal, int newValue)
        {
            Debug.Assert(enumVal.GetType().IsEnum);

            // enum定義をすべて巡回し、適合するものがあれば、値を設定します。
            FieldInfo[] enumFields = enumVal.GetType().GetFields();
            for (int i = 1; i < enumFields.Length; i++)
            {
                FieldInfo fieldInfo = enumFields[i];

                // enum 定義に含まれる値であるなら...
                object enumObj = fieldInfo.GetValue(enumVal);
                int intval = (int)enumObj;
                if (intval == newValue)
                {
                    // 値の更新を行います。
                    enumVal = enumObj;
                    return true;
                }
            }

            // 更新されなかった
            return false;
        }

        /// <summary>
        /// 最小・最大値でクランプします。
        /// </summary>
        private static float ClampMinMax_(float val, float min, float max)
        {
            return Math.Min(Math.Max(val, min), max);
        }

        /// <summary>
        /// アニメーションカーブ内で使用されます。
        /// </summary>
        public static float GetValueAsFloat(AttributeType type, object val)
        {
            switch (type)
            {
                case AttributeType.Float: return Convert.ToSingle(val);
                case AttributeType.Bool: return Convert.ToSingle(val);
                case AttributeType.Int: return Convert.ToSingle(val);
                case AttributeType.Enum: return Convert.ToSingle(val);
                case AttributeType.Byte: return Convert.ToSingle(val);
                default:
                    // その他の型は利用禁止。アニメーションからは、起こりえないはず。
                    Debug.Assert(false);
                    return 0.0f;
            }
        }

        /// <summary>
        /// float 型を byte 値域でクランプされた、byte 値に変換します。
        /// </summary>
        /// <param name="fval"></param>
        /// <returns></returns>
        static byte GetClampedByte_(object fval)
        {
            int ival = Convert.ToInt32(fval);
            if (ival > byte.MaxValue)
            {
                ival = byte.MaxValue;
            }
            else if (ival < byte.MinValue)
            {
                ival = byte.MinValue;
            }
            return Convert.ToByte(ival);
        }



        /// <summary>
        /// アニメーションカーブ内で使用されます。
        /// </summary>
        public static void SetValueAsFloat(AttributeType type, ref object val, float fval)
        {
            try
            {
                switch (type)
                {
                    case AttributeType.Float: val = fval; break;
                    case AttributeType.Bool: val = (fval > 0.0f); break;
                    case AttributeType.Int: val = (int)fval; break;
                    case AttributeType.Enum: SetIntToEnumValue_(ref val, (int)fval); break;
                    case AttributeType.Byte: val = GetClampedByte_(fval); break;
                    default:
                        // その他の型は利用禁止。アニメーションからは、起こりえないはず。
                        Debug.Assert(false);
                        break;
                }
            }
            catch (Exception e)
            {
                DbgConsole.WriteLine(e.ToString());
            }
        }

        #region ITimeChageEventListener メンバ

        #region -------------- ITimeChageEventListener 関連非公開メソッド

        #endregion -------------- ITimeChageEventListener 関連非公開メソッド

        /// <summary>
        /// アトリビュート値を更新します。
        /// </summary>
        /// <param name="newTime"></param>
        public void UpdateAttributeValue(int newTime)
        {
            if (!IsActiveAttribute || ICurrentAnimationCurve == null)
            {
                return;
            }

            float result = ICurrentAnimationCurve.Evaluate(newTime);
            SetAsFloat(result);
        }

        /// <summary>
        /// 時間変更ハンドラ
        /// </summary>
        /// <param name="newTime"></param>
        public void OnTimeChangedHandler(int time, TimeChageEventType type)
        {
            UpdateAttributeValue(time);
        }

        #endregion


        #endregion ----------------- アニメーションカーブから使用されるもの -----------------

        #region ----------------- Enum型関連 -----------------
        /// <summary>
        /// プリミティブ型の値を更新する。
        /// </summary>
        /// <param name="val"></param>
        void SetPrimitiveEnumValue_(object enumVal)
        {
            // 初期化時には代入します。
            if (_This._value == null)
            {
                Debug.Assert(enumVal.GetType().IsEnum);
                _This._value = enumVal;
            }
            // 初期化時以外の場合は、初期化時に設定された、enum情報をもとに
            // 整数ベースで設定値が適切か判定します。
            else
            {
                if (SetIntToEnumValue_(ref _This._value, (int)enumVal))
                {
                    NotifyModifyEvent(_This);
                }
            }
        }
        #endregion ----------------- Enum型関連 -----------------

        #region ----------------- object 型による Set Get -----------------
        /// <summary>
        /// プリミティブ型の値を更新する。
        /// </summary>
        /// <param name="val"></param>
        void SetPrimitiveValue_(object val)
        {
            if (_This._value != val)
            {
                _This._value = val;

                // 重複する通知を避けるため、 NotifyModifyEvent()
                // は プリミティブ型だけ実行する
                NotifyModifyEvent(_This);
            }
        }

        /// <summary>
        /// baseValueを設定します。
        /// </summary>
        private void SetBaseValue_(object val)
        {
            _This._baseValue = val;
        }

        /// <summary>
        /// baseValueを設定します。
        /// </summary>
        public void SetBaseValue(object val)
        {
            try
            {
                switch (_This._valueType)
                {
                    case AttributeType.Int: SetBaseValue_(val); break;
                    case AttributeType.Float: SetBaseValue_(val); break;
                    case AttributeType.Bool: SetBaseValue_(val); break;
                    case AttributeType.Enum: SetBaseValue_(val); break;
                    case AttributeType.Byte: SetBaseValue_(GetClampedByte_(val)); break;
                    case AttributeType.IntVec2:
                        {
                            IVec2 p = (IVec2)val;
                            _This.FindAttributeByIdx(0).SetBaseValue_(p.X);
                            _This.FindAttributeByIdx(1).SetBaseValue_(p.Y);
                            break;
                        }
                    case AttributeType.FloatVec2:
                    case AttributeType.FloatVec2UV:
                        {
                            FVec2 p = (FVec2)val;
                            _This.FindAttributeByIdx(0).SetBaseValue_(p.X);
                            _This.FindAttributeByIdx(1).SetBaseValue_(p.Y);
                            break;
                        }
                    case AttributeType.FloatVec3:
                        {
                            FVec3 p = (FVec3)val;
                            _This.FindAttributeByIdx(0).SetBaseValue_(p.X);
                            _This.FindAttributeByIdx(1).SetBaseValue_(p.Y);
                            _This.FindAttributeByIdx(2).SetBaseValue_(p.Z);
                            break;
                        }
                    case AttributeType.ByteRGBA4:
                        {
                            RGBAColor col = (RGBAColor)val;
                            _This.FindAttributeByIdx(0).SetBaseValue_(col.R);
                            _This.FindAttributeByIdx(1).SetBaseValue_(col.G);
                            _This.FindAttributeByIdx(2).SetBaseValue_(col.B);
                            _This.FindAttributeByIdx(3).SetBaseValue_(col.A);
                            break;
                        }
                    case AttributeType.FloatRGBA4:
                        {
                            FVec4 p = (FVec4)val;
                            _This.FindAttributeByIdx(0).SetBaseValue_(p.X);
                            _This.FindAttributeByIdx(1).SetBaseValue_(p.Y);
                            _This.FindAttributeByIdx(2).SetBaseValue_(p.Z);
                            _This.FindAttributeByIdx(3).SetBaseValue_(p.W);
                            break;
                        }
                    default:
                        Trace.Assert(false);
                        break;
                }
            }
            catch
            {
                Trace.Assert(false);
            }
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        public void SetValue(object val)
        {
            try
            {
                switch (_This._valueType)
                {
                    case AttributeType.Int: SetPrimitiveValue_(val); break;
                    case AttributeType.Float: SetPrimitiveValue_(val); break;
                    case AttributeType.Bool: SetPrimitiveValue_(val); break;
                    case AttributeType.Enum: SetPrimitiveEnumValue_(val); break;
                    case AttributeType.Byte: SetPrimitiveValue_(GetClampedByte_(val)); break;
                    case AttributeType.IntVec2:
                        {
                            IVec2 p = (IVec2)val;
                            _This.FindAttributeByIdx(0).SetPrimitiveValue_(p.X);
                            _This.FindAttributeByIdx(1).SetPrimitiveValue_(p.Y);
                            break;
                        }
                    case AttributeType.FloatVec2:
                    case AttributeType.FloatVec2UV:
                        {
                            FVec2 p = (FVec2)val;
                            _This.FindAttributeByIdx(0).SetPrimitiveValue_(p.X);
                            _This.FindAttributeByIdx(1).SetPrimitiveValue_(p.Y);
                            break;
                        }
                    case AttributeType.FloatVec3:
                        {
                            FVec3 p = (FVec3)val;
                            _This.FindAttributeByIdx(0).SetPrimitiveValue_(p.X);
                            _This.FindAttributeByIdx(1).SetPrimitiveValue_(p.Y);
                            _This.FindAttributeByIdx(2).SetPrimitiveValue_(p.Z);
                            break;
                        }
                    case AttributeType.ByteRGBA4:
                        {
                            RGBAColor col = (RGBAColor)val;
                            _This.FindAttributeByIdx(0).SetPrimitiveValue_(col.R);
                            _This.FindAttributeByIdx(1).SetPrimitiveValue_(col.G);
                            _This.FindAttributeByIdx(2).SetPrimitiveValue_(col.B);
                            _This.FindAttributeByIdx(3).SetPrimitiveValue_(col.A);
                            break;
                        }
                    case AttributeType.FloatRGBA4:
                        {
                            FVec4 p = (FVec4)val;
                            _This.FindAttributeByIdx(0).SetPrimitiveValue_(p.X);
                            _This.FindAttributeByIdx(1).SetPrimitiveValue_(p.Y);
                            _This.FindAttributeByIdx(2).SetPrimitiveValue_(p.Z);
                            _This.FindAttributeByIdx(3).SetPrimitiveValue_(p.W);
                            break;
                        }
                    default:
                        Trace.Assert(false);
                        break;
                }
            }
            catch
            {
                Trace.Assert(false);
            }
        }

        /// <summary>
        /// 2次元ベクトル複数アトリビュートの値を取得します。
        /// </summary>
        void Get2DVectorBaseValue_(out object val0, out object val1)
        {
            _This.FindAttributeByIdx(0).GetBaseValue(out val0);
            _This.FindAttributeByIdx(1).GetBaseValue(out val1);
        }
        void Get2DVectorValue_(out object val0, out object val1)
        {
            _This.FindAttributeByIdx(0).GetValue(out val0);
            _This.FindAttributeByIdx(1).GetValue(out val1);
        }

        /// <summary>
        /// 3次元ベクトル複数アトリビュートの値を取得します。
        /// </summary>
        void Get3DVectorBaseValue_(out object val0, out object val1, out object val2)
        {
            _This.FindAttributeByIdx(0).GetBaseValue(out val0);
            _This.FindAttributeByIdx(1).GetBaseValue(out val1);
            _This.FindAttributeByIdx(2).GetBaseValue(out val2);
        }
        void Get3DVectorValue_(out object val0, out object val1, out object val2)
        {
            _This.FindAttributeByIdx(0).GetValue(out val0);
            _This.FindAttributeByIdx(1).GetValue(out val1);
            _This.FindAttributeByIdx(2).GetValue(out val2);
        }

        /// <summary>
        /// 4次元ベクトル複数アトリビュートの値を取得します。
        /// </summary>
        void Get4DVectorBaseValue_(out object val0, out object val1, out object val2, out object val3)
        {
            _This.FindAttributeByIdx(0).GetBaseValue(out val0);
            _This.FindAttributeByIdx(1).GetBaseValue(out val1);
            _This.FindAttributeByIdx(2).GetBaseValue(out val2);
            _This.FindAttributeByIdx(3).GetBaseValue(out val3);
        }
        void Get4DVectorValue_(out object val0, out object val1, out object val2, out object val3)
        {
            _This.FindAttributeByIdx(0).GetValue(out val0);
            _This.FindAttributeByIdx(1).GetValue(out val1);
            _This.FindAttributeByIdx(2).GetValue(out val2);
            _This.FindAttributeByIdx(3).GetValue(out val3);
        }

        /// <summary>
        /// 設定、取得が可能なタイプか調べます。
        /// </summary>
        public bool CanGetOrSetValue()
        {
            switch (_This._valueType)
            {
                case AttributeType.Int:
                case AttributeType.Float:
                case AttributeType.Bool:
                case AttributeType.Enum:
                case AttributeType.Byte:
                case AttributeType.IntVec2:
                case AttributeType.FloatVec2:
                case AttributeType.FloatVec2UV:
                case AttributeType.FloatVec3:
                case AttributeType.ByteRGBA4:
                case AttributeType.FloatRGBA4:
                    return true;
            }

            return false;
        }

        /// <summary>
        /// baseValueを取得します。
        /// </summary>
        public void GetBaseValue(out object val)
        {
            try
            {
                object x;
                object y;
                object z;
                object w;

                switch (_This._valueType)
                {
                    // 以下のタイプはただ代入するだけです。
                    case AttributeType.Int: val = _This._baseValue; break;
                    case AttributeType.Float: val = _This._baseValue; break;
                    case AttributeType.Bool: val = _This._baseValue; break;
                    case AttributeType.Enum: val = _This._baseValue; break;
                    case AttributeType.Byte: val = _This._baseValue; break;

                    // 以下のタイプは複数アトリビュートを扱うタイプ
                    case AttributeType.IntVec2:
                        Get2DVectorBaseValue_(out x, out y);
                        val = new IVec2((int)x, (int)y);
                        break;
                    case AttributeType.FloatVec2:
                    case AttributeType.FloatVec2UV:
                        Get2DVectorBaseValue_(out x, out y);
                        val = new FVec2((float)x, (float)y);
                        break;
                    case AttributeType.FloatVec3:
                        Get3DVectorBaseValue_(out x, out y, out z);
                        val = new FVec3((float)x, (float)y, (float)z);
                        break;
                    case AttributeType.ByteRGBA4:
                        {
                            object r, g, b, a;
                            _This.FindAttributeByIdx(0).GetBaseValue(out r);
                            _This.FindAttributeByIdx(1).GetBaseValue(out g);
                            _This.FindAttributeByIdx(2).GetBaseValue(out b);
                            _This.FindAttributeByIdx(3).GetBaseValue(out a);

                            val = new RGBAColor((byte)r, (byte)g, (byte)b, (byte)a);
                            break;
                        }
                    case AttributeType.FloatRGBA4:
                        Get4DVectorBaseValue_(out x, out y, out z, out w);
                        val = new FVec4((float)x, (float)y, (float)z, (float)w);
                        break;
                    default:
                        Trace.Assert(false);
                        val = null;
                        break;
                }
            }
            catch
            {
                // 値の取得に失敗しました。
                Trace.Assert(false);
                val = null;
            }
        }

        /// <summary>
        /// 値を取得します。
        /// </summary>
        public void GetValue(out object val)
        {
            try
            {
                object x;
                object y;
                object z;
                object w;

                switch (_This._valueType)
                {
                    // 以下のタイプはただ代入するだけです。
                    case AttributeType.Int: val = _This._value; break;
                    case AttributeType.Float: val = _This._value; break;
                    case AttributeType.Bool: val = _This._value; break;
                    case AttributeType.Enum: val = _This._value; break;
                    case AttributeType.Byte: val = _This._value; break;

                    // 以下のタイプは複数アトリビュートを扱うタイプ
                    case AttributeType.IntVec2:
                        Get2DVectorValue_(out x, out y);
                        val = new IVec2((int)x, (int)y);
                        break;
                    case AttributeType.FloatVec2:
                    case AttributeType.FloatVec2UV:
                        Get2DVectorValue_(out x, out y);
                        val = new FVec2((float)x, (float)y);
                        break;
                    case AttributeType.FloatVec3:
                        Get3DVectorValue_(out x, out y, out z);
                        val = new FVec3((float)x, (float)y, (float)z);
                        break;
                    case AttributeType.ByteRGBA4:
                        {
                            object r, g, b, a;
                            _This.FindAttributeByIdx(0).GetValue(out r);
                            _This.FindAttributeByIdx(1).GetValue(out g);
                            _This.FindAttributeByIdx(2).GetValue(out b);
                            _This.FindAttributeByIdx(3).GetValue(out a);

                            val = new RGBAColor((byte)r, (byte)g, (byte)b, (byte)a);
                            break;
                        }
                    case AttributeType.FloatRGBA4:
                        Get4DVectorValue_(out x, out y, out z, out w);
                        val = new FVec4((float)x, (float)y, (float)z, (float)w);
                        break;
                    default:
                        Trace.Assert(false);
                        val = null;
                        break;
                }
            }
            catch
            {
                // 値の取得に失敗しました。
                Trace.Assert(false);
                val = null;
            }
        }
        #endregion ----------------- object 型による Set Get -----------------

        #region ----------------- float 型による Set Get -----------------

        /// <summary>
        /// フロート値に変換します。
        /// アニメーション関連のクラスから利用されます。
        /// </summary>
        /// <param name="type"></param>
        /// <param name="val"></param>
        /// <returns></returns>
        public float GetAsFloat()
        {
            return ClampMinMax_(GetValueAsFloat(_This._valueType, _This._value), this.MinValue, this.MaxValue) * ViewScale;
        }

        /// <summary>
        /// フロート値に変換します。
        /// アニメーション関連のクラスから利用されます。
        /// </summary>
        /// <param name="type"></param>
        /// <param name="val"></param>
        /// <returns></returns>
        public float GetBaseValueAsFloat()
        {
            return ClampMinMax_(GetValueAsFloat(_This._valueType, _This._baseValue), this.MinValue, this.MaxValue) * ViewScale;
        }

        /// <summary>
        /// フロート値で設定します。
        /// </summary>
        /// <param name="inVal"></param>
        public void SetBaseValueAsFloat(float inVal)
        {
            try
            {
                inVal = ClampMinMax_(inVal / ViewScale, this.MinValue, this.MaxValue);

                switch (_This._valueType)
                {
                    case AttributeType.Float:
                        if (inVal != (float)_This._baseValue)
                        {
                            _This._baseValue = inVal;
                            NotifyModifyEvent(_This);
                        }
                        break;
                    case AttributeType.Bool:
                        {
                            bool newVal = Convert.ToBoolean(inVal);
                            if (newVal != (bool)_This._baseValue)
                            {
                                _This._baseValue = newVal;
                                NotifyModifyEvent(_This);
                            }
                            break;
                        }
                    case AttributeType.Int:
                        {
                            int newVal = (int)inVal;
                            if (newVal != (int)_This._baseValue)
                            {
                                _This._baseValue = newVal;
                                NotifyModifyEvent(_This);
                            }
                            break;
                        }
                    case AttributeType.Enum:
                        {
                            int newVal = (int)inVal;
                            if (newVal != (int)_This._baseValue)
                            {
                                SetPrimitiveEnumValue_(newVal);
                            }
                            break;
                        }
                    case AttributeType.Byte:
                        {
                            byte newVal = GetClampedByte_(inVal);
                            byte currentVal = GetClampedByte_(_This._baseValue);
                            if (newVal != currentVal)
                            {
                                _This._baseValue = newVal;
                                NotifyModifyEvent(_This);
                            }
                            break;
                        }
                    default:
                        // case AttributeType.Enum:
                        // 利用禁止。アニメーションからは、起こりえないはず。
                        Debug.Assert(false);
                        break;
                }
            }
            catch
            {
                // キャスト失敗などによって、設定が行えなかった
                Debug.Assert(false);
            }

        }

        /// <summary>
        /// フロート値で設定します。
        /// </summary>
        /// <param name="inVal"></param>
        public void SetAsFloat(float inVal)
        {
            try
            {
                inVal = ClampMinMax_(inVal / ViewScale, this.MinValue, this.MaxValue);

                switch (_This._valueType)
                {
                    case AttributeType.Float:
                        if (inVal != (float)_This._value)
                        {
                            _This._value = inVal;
                            NotifyModifyEvent(_This);
                        }
                        break;
                    case AttributeType.Bool:
                        {
                            bool newVal = Convert.ToBoolean(inVal);
                            if (newVal != (bool)_This._value)
                            {
                                _This._value = newVal;
                                NotifyModifyEvent(_This);
                            }
                            break;
                        }
                    case AttributeType.Int:
                        {
                            int newVal = (int)inVal;
                            if (newVal != (int)_This._value)
                            {
                                _This._value = newVal;
                                NotifyModifyEvent(_This);
                            }
                            break;
                        }
                    case AttributeType.Enum:
                        {
                            int newVal = (int)inVal;
                            if (newVal != (int)_This._value)
                            {
                                SetPrimitiveEnumValue_(newVal);
                            }
                            break;
                        }
                    case AttributeType.Byte:
                        {
                            byte newVal = GetClampedByte_(inVal);
                            byte currentVal = GetClampedByte_(_This._value);
                            if (newVal != currentVal)
                            {
                                _This._value = newVal;
                                NotifyModifyEvent(_This);
                            }
                            break;
                        }
                    default:
                        // case AttributeType.Enum:
                        // 利用禁止。アニメーションからは、起こりえないはず。
                        Debug.Assert(false);
                        break;
                }
            }
            catch (Exception e)
            {
                // キャスト失敗などによって、設定が行えなかった
                Debug.Assert(false, e.ToString());
            }

        }

        /// <summary>
        /// IVec2のアトリビュートを取得します。
        /// </summary>>
        public IVec2 GetBaseValueAsIVec2()
        {
            Debug.Assert(_This._valueType == AttributeType.IntVec2);

            object ivec2 = null;
            GetBaseValue(out ivec2);

            return (IVec2)ivec2;
        }
        public IVec2 GetAsIVec2()
        {
            Debug.Assert(_This._valueType == AttributeType.IntVec2);

            object ivec2 = null;
            GetValue(out ivec2);

            return (IVec2)ivec2;
        }

        /// <summary>
        /// Fvec2のアトリビュートを取得します。
        /// </summary>>
        public FVec2 GetBaseValueAsFVec2()
        {
            Debug.Assert(_This._valueType == AttributeType.FloatVec2 || _This._valueType == AttributeType.FloatVec2UV);

            object fvec2 = null;
            GetBaseValue(out fvec2);

            return (FVec2)fvec2;
        }
        public FVec2 GetAsFVec2()
        {
            Debug.Assert(_This._valueType == AttributeType.FloatVec2 || _This._valueType == AttributeType.FloatVec2UV);

            object fvec2 = null;
            GetValue(out fvec2);

            return (FVec2)fvec2;
        }

        /// <summary>
        /// Fvec3のアトリビュートを取得します。
        /// </summary>>
        public FVec3 GetBaseValueAsFVec3()
        {
            Debug.Assert(_This._valueType == AttributeType.FloatVec3);

            object fvec3 = null;
            GetBaseValue(out fvec3);

            return (FVec3)fvec3;
        }
        public FVec3 GetAsFVec3()
        {
            Debug.Assert(_This._valueType == AttributeType.FloatVec3);

            object fvec3 = null;
            GetValue(out fvec3);

            return (FVec3)fvec3;
        }

        /// <summary>
        /// ByteRGBA4 アトリビュートをRGBAColor型で取得します。
        /// </summary>
        /// <returns></returns>
        public RGBAColor GetBaseValueAsRBGA()
        {
            Debug.Assert(_This._valueType == AttributeType.ByteRGBA4);

            object ret;
            GetBaseValue(out ret);
            return (RGBAColor)ret;
        }
        public RGBAColor GetAsRBGA()
        {
            Debug.Assert(_This._valueType == AttributeType.ByteRGBA4);

            object ret;
            GetValue(out ret);
            return (RGBAColor)ret;
        }

        /// <summary>
        /// Fvec4 アトリビュートをFloatColor型で取得します。
        /// </summary>
        /// <returns></returns>
        public FloatColor GetBaseValueAsFloatColor()
        {
            object fvec4 = null;
            GetBaseValue(out fvec4);

            FVec4 color = (FVec4)fvec4;

            return new FloatColor(color.X, color.Y, color.Z, color.W);
        }
        public FloatColor GetAsFloatColor()
        {
            object fvec4 = null;
            GetValue(out fvec4);

            FVec4 color = (FVec4)fvec4;

            return new FloatColor(color.X, color.Y, color.Z, color.W);
        }

        /// <summary>
        /// Byte アトリビュートをbyte型で取得します。
        /// </summary>
        /// <returns></returns>
        public byte GetBaseValueAsByte()
        {
            Debug.Assert(_This._valueType == AttributeType.Byte);

            object ret;
            GetBaseValue(out ret);
            return (byte)ret;
        }
        public byte GetAsByte()
        {
            Debug.Assert(_This._valueType == AttributeType.Byte);

            object ret;
            GetValue(out ret);
            return (byte)ret;
        }

        #endregion ----------------- object 型による Set Get -----------------

        #endregion // ----------------- アクセサ -----------------

        #region ------------------ 初期化 ------------------
        /// <summary>
        /// サブアトリビュートの初期化を行います。
        /// コンストラクタから呼ばれます。
        /// </summary>
        /// <param name="owner"></param>
        /// <param name="subAttrDescs"></param>
        /// <param name="selfAttrDesc"></param>
        void InitializeSubAttributes_
        (
            AnmAttrDescripter[] subAttrDescs,
            AnmAttrDescripter selfAttrDesc
        )
        {
            // サブアトリビュートが存在する場合は...
            if (subAttrDescs != null)
            {
                // 指定された、サブアトリビュートの実体を生成。
                foreach (AnmAttrDescripter subAttrDesc in subAttrDescs)
                {
                    // 実体を生成します。
                    if (subAttrDesc._createFunc != null)
                    {
                        // 生成メソッドを使用する場合
                        subAttrDesc._createFunc(this, subAttrDesc);
                    }
                    else
                    {
                        // 既定のインスタンス生成が行われます。
                        new AnmAttribute(this, null, subAttrDesc);
                    }
                }
            }
        }

        /// <summary>
        /// ITimeChageEventListenerのメンバを初期化します。
        /// </summary>
        public void InitializeITimeChageEventListener(bool useCustomTimeSource)
        {
            // すでに設定済みだが、後から異なる設定で再度初期化が呼ばれている
            if (TimeReleyProvider != null && useCustomTimeSource == false)
            {
                Debug.Assert(false, "Un Expected type change at useCustomTimeSource !");
            }

            TimeReleyProvider = useCustomTimeSource ? new TimeReleyProvider() : null;

            if(TimeChangedHandler == null)
            {
                TimeChangedHandler = this.OnTimeChangedHandler;
            }
        }

        /// <summary>
        /// カスタム時間ソースを初期化します。
        /// </summary>
        public void InitializeCustomTimeSource(ITimeChangeEventProvider timeEventProvider)
        {
            Debug.Assert(TimeEventProvider == null);
            Debug.Assert(TimeReleyProvider != null);

            TimeEventProvider = timeEventProvider;
            TimeEventProvider.OnTimeChanged += TimeReleyProvider.Reley;
        }


        #region -------------- MakeInstance_ 初期値設定 --------------

        /// <summary>
        /// 型に応じて、組み込み複合型アトリビュートの初期値を設定します。
        /// </summary>
        static void SetInitialValueIfNeeded_(AnmAttribute attr, AnmAttrDescripter attrDesc)
        {
            switch (attrDesc._type)
            {
                case AttributeType.IntVec2: SetDefaultRgbaAttr_(attr, attrDesc); break;
                case AttributeType.FloatVec2:
                case AttributeType.FloatVec2UV: SetDefaultFVec2Attr_(attr, attrDesc); break;
                case AttributeType.FloatVec3: SetDefaultFVec3Attr_(attr, attrDesc); break;
                case AttributeType.ByteRGBA4: SetDefaultRgbaAttr_(attr, attrDesc); break;
                case AttributeType.FloatRGBA4: SetDefaultFloatRgba4Attr_(attr, attrDesc); break;
                default: return;
            }
        }

        /// <summary>
        /// AttributeType.IntVec2 の初期値を設定します。
        /// </summary>
        static void SetDefaultIVec2Attr_(AnmAttribute attrIVec2, AnmAttrDescripter attrDesc)
        {
            if (attrDesc._initialValue != null)
            {
                IVec2 defaultVal = (IVec2)attrDesc._initialValue;
                attrIVec2.FindAttributeByIdx(0).SetValue(defaultVal.X);
                attrIVec2.FindAttributeByIdx(1).SetValue(defaultVal.Y);
            }
        }

        /// <summary>
        /// AttributeType.FloatVec2 の初期値を設定します。
        /// </summary>
        static void SetDefaultFVec2Attr_(AnmAttribute attrFVec2, AnmAttrDescripter attrDesc)
        {
            if (attrDesc._initialValue != null)
            {
                FVec2 defaultVal = (FVec2)attrDesc._initialValue;
                attrFVec2.FindAttributeByIdx(0).SetValue(defaultVal.X);
                attrFVec2.FindAttributeByIdx(1).SetValue(defaultVal.Y);
            }
        }

        /// <summary>
        /// AttributeType.FloatVec3 の初期値を設定します。
        /// </summary>
        static void SetDefaultFVec3Attr_(AnmAttribute attrFVec3, AnmAttrDescripter attrDesc)
        {
            if (attrDesc._initialValue != null)
            {
                FVec3 defaultVal = (FVec3)attrDesc._initialValue;
                attrFVec3.FindAttributeByIdx(0).SetValue(defaultVal.X);
                attrFVec3.FindAttributeByIdx(1).SetValue(defaultVal.Y);
                attrFVec3.FindAttributeByIdx(2).SetValue(defaultVal.Z);
            }
        }


        /// <summary>
        /// AttributeType.ByteRGBA4 の初期値を設定します。
        /// </summary>
        static void SetDefaultRgbaAttr_(AnmAttribute attrRgba, AnmAttrDescripter attrDesc)
        {
            if (attrDesc._initialValue != null)
            {
                RGBAColor defaultVal = (RGBAColor)attrDesc._initialValue;
                attrRgba.FindAttributeByIdx(0).SetValue(defaultVal.R);
                attrRgba.FindAttributeByIdx(1).SetValue(defaultVal.G);
                attrRgba.FindAttributeByIdx(2).SetValue(defaultVal.B);
                attrRgba.FindAttributeByIdx(3).SetValue(defaultVal.A);
            }
        }

        /// <summary>
        /// AttributeType.FloatRGBA4 の初期値を設定します。
        /// </summary>
        static void SetDefaultFloatRgba4Attr_(AnmAttribute attrFloatRgba, AnmAttrDescripter attrDesc)
        {
            if (attrDesc._initialValue != null)
            {
                FVec4 defaultVal = (FVec4)attrDesc._initialValue;
                attrFloatRgba.FindAttributeByIdx(0).SetValue(defaultVal.X);
                attrFloatRgba.FindAttributeByIdx(1).SetValue(defaultVal.Y);
                attrFloatRgba.FindAttributeByIdx(2).SetValue(defaultVal.Z);
                attrFloatRgba.FindAttributeByIdx(3).SetValue(defaultVal.W);
            }
        }


        #endregion -------------- 初期値設定 --------------

        /// <summary>
        /// アトリビュート型から、規定の補間タイプを取得します。
        /// </summary>
        static public InterporationType GetDefaultInterpType(AttributeType attrType)
        {
            switch (attrType)
            {
                // ステップ
                case AttributeType.Bool: return InterporationType.Step;
                case AttributeType.Enum: return InterporationType.Step;
                // エルミート
                case AttributeType.Byte: return InterporationType.Spline;
                case AttributeType.Float: return InterporationType.Spline;
                case AttributeType.Int: return InterporationType.Spline;

                // 上記のタイプ以外では、アニメーションカーブの初期化は行われません。
                default: Debug.Assert(false); return InterporationType.None;
            }
        }

        /// <summary>
        /// InterporationType が AttributeType にとって適切か判定します。
        /// </summary>
        public static bool CheckInterpTypeValid(AttributeType attrType, InterporationType type)
        {
            switch (attrType)
            {
                // ステップ
                case AttributeType.Bool:
                case AttributeType.Enum:
                    // Step種類が有効
                    return type == InterporationType.Step;
                // エルミート
                case AttributeType.Byte:
                case AttributeType.Float:
                case AttributeType.Int:
                    // 全ての種類が有効
                    return true;
                default:
                    Debug.Assert(false);
                    return false;
            }
        }

        /// <summary>
        /// 組み込み複合型アトリビュートのサブアトリビュート記述子を取得します。
        /// </summary>
        /// <param name="type"></param>
        /// <returns></returns>
        static AnmAttrDescripter[] GetSubAnmAttrDescripter_(AttributeType type)
        {
            switch (type)
            {
                case AttributeType.IntVec2: return IntVecDesc;
                case AttributeType.FloatVec2: return FloatVecDesc;
                case AttributeType.FloatVec2UV: return FloatVecUVDesc;
                case AttributeType.FloatVec3: return FloatVec3Desc;
                case AttributeType.ByteRGBA4: return RGBA4Desc;
                case AttributeType.FloatRGBA4: return FloatRGBA4Desc;
                default: Debug.Assert(false); return null;
            }
        }

        static AnmAttrDescripter[] SetTimeSourceFlagToDescripter_(AnmAttrDescripter[] srcDescs, bool useCustomTimeSource)
        {
            List<AnmAttrDescripter> dstDescs = new List<AnmAttrDescripter>();

            foreach (AnmAttrDescripter srcDesc in srcDescs)
            {
                dstDescs.Add(new AnmAttrDescripter(srcDesc._type, srcDesc._name, srcDesc._createFunc, srcDesc._initialValue, srcDesc._lockAnimationInterpolateType ? AnmAttrDescripter.AnimationTypeLock.True : AnmAttrDescripter.AnimationTypeLock.False, useCustomTimeSource));
            }

            return dstDescs.ToArray();
        }

        /// <summary>
        /// サブアトリビュートを持つアトリビュートのコンストラクタ(複合型)
        /// </summary>
        /// <param name="types"></param>
        /// <param name="names"></param>
        /// <param name="name"></param>
        public AnmAttribute
        (
            LEDataNode owner,
            AnmAttrDescripter[] subAttrDescs,
            AnmAttrDescripter selfAttrDesc
        )
            : base(owner, selfAttrDesc._name)
        {
            // ---------------- パラメータを初期化します。
            _valueType = selfAttrDesc._type;
            _ownerAttribute = owner;
            _value = null;
            _baseValue = null;
            _useCustomTimeSource = selfAttrDesc._useCustomTimeSource;

            switch (selfAttrDesc._type)
            {
                case AttributeType.Int:
                case AttributeType.Float:
                case AttributeType.Bool:
                case AttributeType.Enum:
                case AttributeType.Byte:
                    // 値の設定
                    Debug.Assert(selfAttrDesc._initialValue != null);
                    SetValue(selfAttrDesc._initialValue);
                    SetBaseValue(selfAttrDesc._initialValue);

                    // アニメーションカーブの初期化
                    AnmCurve curve = new AnmCurve();
                    curve.BindTarget(this, selfAttrDesc._lockAnimationInterpolateType, this._useCustomTimeSource);
                    this.DefaultCurve = curve;
                    break;
                case AttributeType.IntVec2:
                case AttributeType.FloatVec2:
                case AttributeType.FloatVec2UV:
                case AttributeType.FloatVec3:
                case AttributeType.ByteRGBA4:
                case AttributeType.FloatRGBA4:
                    // デフォルトの設定
                    if (subAttrDescs == null)
                    {
                        subAttrDescs = GetSubAnmAttrDescripter_(selfAttrDesc._type);
                    }

                    // 設定を子に引き継ぐ
                    if (selfAttrDesc._useCustomTimeSource)
                    {
                        subAttrDescs = SetTimeSourceFlagToDescripter_(subAttrDescs, selfAttrDesc._useCustomTimeSource);
                    }

                    // 値を子に引き継ぐ
                    if (selfAttrDesc._initialValue != null)
                    {
                        var values = ValueToEnumerable(selfAttrDesc._type, selfAttrDesc._initialValue).Cast<object>().ToArray();
                        Debug.Assert(values.Length == subAttrDescs.Length);
                        if (values.Length == subAttrDescs.Length)
                        {
                            subAttrDescs = subAttrDescs.Zip(values, (x, y) =>
                            new AnmAttrDescripter(
                                x._type,
                                x._name,
                                x._createFunc,
                                y,
                                x._lockAnimationInterpolateType ? AnmAttrDescripter.AnimationTypeLock.True: AnmAttrDescripter.AnimationTypeLock.False,
                                x._useCustomTimeSource)).ToArray();
                        }
                    }
                    break;
                case AttributeType.Combined:
                    // 何もしない
                    break;
                case AttributeType.RefAttribute:
                    // type が参照型
                    // _valueに参照対象である AnmAttribute を設定します。
                    _value = selfAttrDesc._initialValue;
                    break;
                default:
                    Debug.Assert(false);
                    break;
            }

            InitializeSubAttributes_(subAttrDescs, selfAttrDesc);

            // 最大、最小値
            this.MaxValue = float.MaxValue;
            this.MinValue = float.MinValue;
        }

        private IEnumerable ValueToEnumerable(AttributeType type, object value)
        {
            try
            {
                switch (type)
                {
                    case AttributeType.IntVec2:
                        return ((IVec2)value).AsIEnumerabe();
                    case AttributeType.FloatVec2:
                        return ((FVec2)value).AsIEnumerable();
                    case AttributeType.FloatVec2UV:
                        return ((FVec2)value).AsIEnumerable();
                    case AttributeType.FloatVec3:
                        return ((FVec3)value).AsIEnumerable();
                    case AttributeType.ByteRGBA4:
                        return ((RGBAColor)value).AsIEnumerable();
                    case AttributeType.FloatRGBA4:
                        return ((FVec4)value).AsIEnumerable();
                }
            }
            catch
            {
            }

            Debug.Assert(false);
            return Enumerable.Empty<object>();
        }

        #region IDisposable メンバ

        /// <summary>
        /// リソースの開放をおこないます。
        /// 本クラスでは、メッセージハンドラのリセットをおこなっています。
        /// </summary>
        public virtual void Dispose()
        {
            _anmCurves.Dispose();

            // サブアトリビュートに関しても同様に、開放します。
            foreach (LEDataNode subNode in _subAttrArray)
            {
                if (subNode is AnmAttribute)
                {
                    AnmAttribute anmAttrNode = subNode as AnmAttribute;
                    anmAttrNode.Dispose();
                }
            }

            // TimeEventHandlerを削除します
            GetTimeSource().TimeChangingHandlers.Remove(this);
        }

        #endregion



        #endregion ------------------ 初期化 ------------------

        /// <summary>
        ///
        /// </summary>
        public ITimeChangeEventProvider GetTimeSource()
        {
            return (TimeReleyProvider != null) ? TimeReleyProvider as ITimeChangeEventProvider : GlobalTime.Inst;
        }

        /// <summary>
        /// TimeEventHandlerを追加/削除します
        /// </summary>
        public void SetTimeEventHandler(bool add)
        {
            // ひとつもキーを持っていない場合に追加/削除を行う
            if ((!this.HasAnyKey))
            {
                ITimeChangeEventProvider timeSource = GetTimeSource();
                if (add)
                {
                    timeSource.TimeChangingHandlers.Add(this);
                }
                else
                {
                    timeSource.TimeChangingHandlers.Remove(this);
                }
            }
        }

        /// <summary>
        /// タグ名とアニメーションカーブを紐付けて登録します。
        /// </summary>
        public virtual void RegistAnmCurve(string tag, AnmCurve curve)
        {
            _anmCurves.RegistAnmCurve(tag, curve);
        }

        /// <summary>
        /// 指定したタグに、アニメーションカーブの参照を紐付けます。
        /// 参照ソースは、分割モード用のアニメーションカーブが使われます。
        /// </summary>
        public virtual void ReferenceAnmCurve(string tag)
        {
            _anmCurves.ReferenceAnmCurve(tag);
        }

        /// <summary>
        /// アニメーションカーブを削除します。
        /// </summary>
        public virtual void RemoveAnmCurve(string tag)
        {
            _anmCurves.RemoveAnmCurve(tag);
        }

        /// <summary>
        /// アニメーションカーブを全削除します。
        /// </summary>
        public virtual void ClearAnmCurve()
        {
            _anmCurves.ClearAnmCurves();
        }

        /// <summary>
        /// 分割モード用のアニメーションカーブを削除します。
        /// </summary>
        public virtual void RemoveMultiCurves()
        {
            _anmCurves.RemoveMultiCurves();
        }

        /// <summary>
        /// 指定したタグ名に紐付いたアニメーションカーブを取得します。
        /// </summary>
        public virtual AnmCurve GetAnmCurve(string tag)
        {
            return _anmCurves.GetAnmCurve(tag);
        }

        /// <summary>
        /// アニメーションカーブを複製します。
        /// </summary>
        public virtual void CopyAnmCurves(AnmAttribute src)
        {
            _anmCurves.ClearAnmCurves();
            foreach (string tag in SubSceneHelper.GetTags(CurrentISubScene))
            {
                AnmCurve curve = src.GetAnmCurve(tag);
                if (curve != null)
                {
                    _anmCurves.RegistAnmCurve(tag, curve);
                }
            }
        }

        /// <summary>
        /// アニメーションタグ名を変更します。
        /// この機能はアニメーションの分割管理モードの場合に使用されます。
        /// </summary>
        public virtual void ModifyAnmTag(string srcTag, string dstTag)
        {
            _anmCurves.ModifyAnmTag(srcTag, dstTag);
        }

        /// <summary>
        /// 通常のカーブをタグに対応するカーブとして登録します。
        /// （AddAnmTag から呼ばれます）
        /// </summary>
        void RegisterNormalAnmCurveForTag_(string tagName)
        {
            // すでにある場合は何もしない。
            if(GetAnmCurve(tagName) != null)
            {
                return;
            }

            AnmCurve newCurve = new AnmCurve();
            newCurve.BindTargetKeepInterpType(this, this.LockInterpType, this._useCustomTimeSource);

            this.RegistAnmCurve(tagName, newCurve);
        }

        /// <summary>
        /// アニメーションタグを追加します。
        /// この機能はアニメーションの分割管理モードの場合に使用されます。
        /// 内部では、タグ毎のアニメーションカーブを登録しています。
        /// </summary>
        public void AddAnmTag(string tagName)
        {
            // 注意：
            // 分割モードの仕様：
            // 0番はタグ未選択に対応。1番は参照ソースを格納。
            // 0番から番号順に AddAnmTag が呼ばれるという前提
            // ...で、CurveNum が 2 以上なら参照として処理する。。。としています。
            if (this.IsReferenceCurves)
            {
                bool isNotReferenceSource = this.CurveNum > 1;
                if (isNotReferenceSource)
                {
                    // 参照カーブ（実体を共有する形）として登録する。
                    this.ReferenceAnmCurve(tagName);
                }else
                {
                    // 通常のカーブとして登録する。
                    RegisterNormalAnmCurveForTag_(tagName);
                }
            }
            else
            {
                // 通常のカーブとして登録する。
                RegisterNormalAnmCurveForTag_(tagName);
            }
        }

        /// <summary>
        /// アニメーションタグを削除します。
        /// この機能はアニメーションの分割管理モードの場合に使用されます。
        /// </summary>
        public void RemoveAnmTag(string tagName)
        {
            this.RemoveAnmCurve(tagName);
        }

        /// <summary>
        /// 番号からサブアトリビュートを取得
        /// </summary>
        /// <param name="idx"></param>
        /// <returns></returns>
        public AnmAttribute FindAttributeByIdx(int idx)
        {
            return _This.FindSubAttributeByIdx(idx) as AnmAttribute;
        }

        /// <summary>
        /// 変更を通知します。
        /// _This.OnAnmAttrModify イベントには伝播しないため、
        /// 参照アトリビュートには影響しない点に注意が必要です。
        /// </summary>
        public void NotifyModifyEvent(AnmAttribute sender)
        {
            NotifyModifyEvent(sender, (int)EventKind.Modify);
        }

        /// <summary>
        /// オーバーライド：更新メッセージの伝播
        /// </summary>
        public override void NotifyModifyEvent(LEDataNode sender, int kind)
        {
            base.NotifyModifyEvent(sender, kind);
        }

        /// <summary>
        /// アニメーションカーブに変更があったときに呼ばれます。
        /// </summary>
        public void OnAnmCurveModified()
        {
            NotifyModifyEvent(this, (int)EventKind.AnimationModify);
        }

        /// <summary>
        /// アニメーションカーブに変更があったときに呼ばれます。
        /// </summary>
        public void OnAnmCurveKeyAddRemove()
        {
            NotifyModifyEvent(this, (int)EventKind.AnimationAddRemove);
        }

        /// <summary>
        /// baseValueをサブアトリビュートを含め設定します。
        /// 現在のvalueがbaseValueに設定されます。
        /// </summary>
        public void SetBaseValueRecursive()
        {
            if (CanGetOrSetValue() &&
                NumSubAttribute == 0)
            {
                object val;
                GetValue(out val);
                SetBaseValue(val);
            }

            foreach (AnmAttribute subAttr in SubAttrArray)
            {
                subAttr.SetBaseValueRecursive();
            }
        }

        /// <summary>
        /// 現在値をbaseValueに書き換えます。
        /// </summary>
        public void ResetValueAsBaseValueRecursive()
        {
            if (CanGetOrSetValue() && NumSubAttribute == 0)
            {
                object val;
                GetBaseValue(out val);
                SetValue(val);
            }

            foreach (AnmAttribute subAttr in SubAttrArray)
            {
                subAttr.ResetValueAsBaseValueRecursive();
            }
        }

        /// <summary>
        /// アニメーションカーブを管理するクラス。
        /// </summary>
        internal class AnimationCurves
        {
            #region ------------- フィールド -------------

            private Dictionary<string, AnmCurve> _anmCurves = new Dictionary<string, AnmCurve>();

            #endregion ------------- フィールド -------------

            #region ------------- プロパティ -------------

            /// <summary>
            /// 所有しているアニメーションの数
            /// </summary>
            public int CurveNum
            {
                get { return _anmCurves.Count(); }
            }

            /// <summary>
            /// 所持しているカーブにキーが1つ以上存在するか。
            /// </summary>
            public bool HasAnyKey
            {
                get
                {
                    foreach (AnmCurve curve in _anmCurves.Values)
                    {
                        if (curve.NumKeyFrame > 0)
                        {
                            return true;
                        }
                    }

                    return false;
                }
            }

            /// <summary>
            /// カーブの配列を取得します。
            /// </summary>
            public AnmCurve[] Curves
            {
                get
                {
                    return _anmCurves.Values.ToArray();
                }
            }

            /// <summary>
            /// デフォルトのアニメーションカーブを取得します。
            /// </summary>
            public AnmCurve DefaultCurve
            {
                get
                {
                    return GetAnmCurve(String.Empty);
                }

                set
                {
                    RegistAnmCurve(AnimCurvesParam.__NoSelected__, value);
                }
            }

            #endregion ------------- プロパティ -------------

            /// <summary>
            /// リソースを破棄します。
            /// </summary>
            public void Dispose()
            {
                foreach (AnmCurve curve in _anmCurves.Values)
                {
                    curve.Dispose();
                }
                // _anmCurvesをクリアしてしまうとアプリ終了時のDisposeでnull参照が発生するため、クリアしない
                //_anmCurves.Clear();
            }

            /// <summary>
            /// タグ名とアニメーションカーブを紐付けて登録します。
            /// </summary>
            public void RegistAnmCurve(string tag, AnmCurve curve)
            {
                if (!_anmCurves.ContainsKey(tag))
                {
                    _anmCurves[tag] = curve;
                }
            }

            /// <summary>
            /// 指定したタグに、アニメーションカーブの参照を紐付けます。
            /// 参照ソースは、分割モード用のアニメーションカーブが使われます。
            /// </summary>
            public void ReferenceAnmCurve(string tag)
            {
                var combCurve = _anmCurves[AnimCurvesParam.__NoSelected__];
                var referencePair = _anmCurves.FirstOrDefault(item => item.Value != combCurve);
                Debug.Assert(!referencePair.Equals(default(KeyValuePair<string, AnmCurve>)));

                RegistAnmCurve(tag, referencePair.Value);
            }

            /// <summary>
            /// アニメーションカーブを削除します。
            /// </summary>
            public void RemoveAnmCurve(string tag)
            {
                AnmCurve curve;
                if (_anmCurves.TryGetValue(tag, out curve))
                {
                    _anmCurves.Remove(tag);
                }
            }

            /// <summary>
            /// 分割モード用のアニメーションカーブを削除します。
            /// </summary>
            public void RemoveMultiCurves()
            {
                AnmCurve defaultCurve = DefaultCurve;
                ClearAnmCurves();
                DefaultCurve = defaultCurve;
            }

            /// <summary>
            /// すべてのアニメーションカーブを削除します。
            /// </summary>
            public void ClearAnmCurves()
            {
                _anmCurves.Clear();
            }

            /// <summary>
            /// タグ名を変更します。
            /// </summary>
            public void ModifyAnmTag(string srcTag, string dstTag)
            {
                AnmCurve curve;
                if (_anmCurves.TryGetValue(srcTag, out curve))
                {
                    RemoveAnmCurve(srcTag);
                    RegistAnmCurve(dstTag, curve);
                }
            }

            /// <summary>
            /// 指定したタグ名に紐付いたアニメーションカーブを取得します。
            /// </summary>
            public AnmCurve GetAnmCurve(string tag)
            {
                if (tag != null)
                {
                    AnmCurve curve;
                    if (_anmCurves.TryGetValue(tag, out curve))
                    {
                        return curve;
                    }
                }

                return null;
            }
        }
    }

    /// <summary>
    /// アニメーションカーブ管理の公開パラメータ
    /// </summary>
    public class AnimCurvesParam
    {
        // タグ未選択状態のタグ名
        static readonly public string __NoSelected__ = String.Empty;
    }
}
