﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Windows.Forms;

namespace LECore.Structures
{
    using Save_Load;
    using SerializableFmt = LECore.Structures.SerializableObject.Lyt;
    using SerializableAnmFmt = LECore.Structures.SerializableObject.Lan;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;


    /// <summary>
    /// 外部公開用インタフェース
    /// </summary>
    public interface ILEClipboard
    {
        bool   Copy   ( object srcInstance );
        object Paste  ( System.Type  dstInstType );
        bool RegisterConvertionFuncSet( LEClipboardConvertFunc func );

        bool CanPaste( System.Type dstInstType );
    }

    #region クリップボード保存のために使用されるクラス

    /// <summary>
    /// クリップボード用保存データクラスへの変換関数セット
    /// 本クラスを派生左折形で、各データ構造に固有の変換処理を行うクラスを実装します。
    ///
    /// 変換処理を行うクラスをマネージャに登録することで、
    /// 各データ構造にコピー＆ペースト機能が提供されます。
    /// </summary>
    public class LEClipboardConvertFunc
    {
        /// <summary>
        /// データを変換するデリゲータ
        /// </summary>
        public delegate object DataConvertFunc ( object srcInstance );

        /// <summary>
        /// アプリケーション内部データ => クリップボード用保存データ
        /// </summary>
        public readonly DataConvertFunc     IntrnalToSavable;

        /// <summary>
        /// クリップボード用保存データ => アプリケーション内部データ
        /// </summary>
        public readonly DataConvertFunc     SavableToInternal;

        /// <summary>
        /// アプリケーション内部データ型
        /// </summary>
        public readonly System.Type         InternalObjType;

        /// <summary>
        /// クリップボード用保存データ型
        /// </summary>
        public readonly System.Type         SavableObjType;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEClipboardConvertFunc(
            DataConvertFunc   toSavable,
            DataConvertFunc   toAppInstace,
            System.Type       internalObjType,
            System.Type       savableObjType )
        {
            IntrnalToSavable      = toSavable;
            SavableToInternal     = toAppInstace;

            InternalObjType  = internalObjType;
            SavableObjType   = savableObjType;
        }
    }



    /// <summary>
    /// PaneModify[] をコピーペーストする際に使用される変換関数セットです。
    /// </summary>
    internal class LEClipboardPaneSetConvertFunc : LEClipboardConvertFunc
    {
        /// <summary>
        /// 内部で型変換に使用するクラス。
        /// (ファイル出力の際に使用するシリアライズクラスを流用しています。)
        /// </summary>
        ///
        static readonly RlytConverter   _serializableRlytCvtr  =
            new RlytConverter( string.Empty, LayoutEditorCore.MsgReporter );

        /// <summary>
        ///
        /// </summary>
        class InternalPane
        {
            PaneData _Pane;

            // ペイン
            public PaneData Pane { get { return _Pane; } }

            // ペイン名
            public string Name { get { return Pane.Name; } }

            // ペインアトリビュート
            public Dictionary<ParamaterKind, AttributeData[]> Attribute { get { return Pane.AttributeSet; } }

            public InternalPane(IPane srcPane)
            {
                Set(srcPane);
            }

            /// <summary>
            /// ペインの持つパラメータを複製します
            /// </summary>
            public void Set(IPane srcPane)
            {
                // アニメーションを持つアトリビュートを抽出します
                Dictionary<ParamaterKind, AttributeData[]> attrSet = new Dictionary<ParamaterKind, AttributeData[]>();

                foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
                {
                    List<AttributeData> subAttrSet = new List<AttributeData>();
                    IAnmAttribute[] srcAttrSet = srcPane.GetAnimationTargetAttributeSet(kind);

                    foreach (IAnmAttribute srcAttr in srcAttrSet)
                    {
                        // アトリビュートの持つパラメータを複製
                        AttributeData attrData = SetAttributeDataRecursive_(srcAttr);
                        subAttrSet.Add(attrData);
                    }

                    attrSet.Add(kind, subAttrSet.ToArray());
                }

                // 新しいインスタンスに更新
                _Pane = new PaneData(
                    srcPane.PaneName,
                    attrSet);
            }

            /// <summary>
            /// アトリビュートの持つパラメータを複製します
            /// </summary>
            private AttributeData SetAttributeDataRecursive_(IAnmAttribute srcAttr)
            {
                // サブアトリビュートを処理する
                List<AttributeData> subAttrSet = new List<AttributeData>();
                for (int i = 0; i < srcAttr.NumSubAttribute; i++)
                {
                    AttributeData subAttrData = SetAttributeDataRecursive_(srcAttr.FindSubAttributeByIdx(i));
                    subAttrSet.Add(subAttrData);
                }

                // アトリビュートの持つパラメータを複製
                AttributeData attrData = SetAttributeData_(srcAttr, subAttrSet.ToArray());

                return attrData;
            }

            /// <summary>
            /// アトリビュートの持つパラメータを複製します
            /// </summary>
            private AttributeData SetAttributeData_(IAnmAttribute srcIAttr, AttributeData[] subAttrSet)
            {
                AnmAttribute srcAttr = srcIAttr as AnmAttribute;

                // baseValue
                object baseValue = null;
                if (srcAttr.CanGetOrSetValue())
                {
                    srcAttr.GetBaseValue(out baseValue);
                }

                return new AttributeData(
                    srcAttr.Name,
                    baseValue,
                    subAttrSet);
            }
        }

        /// <summary>
        ///
        /// </summary>
        class PaneData
        {
            readonly string _Name;
            readonly Dictionary<ParamaterKind, AttributeData[]> _AttributeSet;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public PaneData(
                string Name,
                Dictionary<ParamaterKind, AttributeData[]> AttributeSet)
            {
                _Name = Name;
                _AttributeSet = AttributeSet;
            }

            // ペイン名
            public string Name { get { return _Name; } }

            // アトリビュート
            public Dictionary<ParamaterKind, AttributeData[]> AttributeSet { get { return _AttributeSet; } }
        }

        /// <summary>
        ///
        /// </summary>
        class AttributeData
        {
            readonly string _Name;
            readonly object _BaseValue;
            readonly AttributeData[] _SubAttribute;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public AttributeData(
                string Name,
                object BaseValue,
                AttributeData[] SubAttribute)
            {
                _Name = Name;
                _BaseValue = BaseValue;
                _SubAttribute = SubAttribute;
            }

            // アトリビュート名
            public string Name { get { return _Name; } }

            // baseValue
            public object BaseValue { get { return _BaseValue; } }

            // サブアトリビュート
            public AttributeData[] SubAttribute { get { return _SubAttribute; } }
        }

        /// <summary>
        ///
        /// </summary>
        class InternalClipBoardPane
        {
            readonly PaneSetDupulicator.IntermediateData _paneData;
            readonly string[] _texturePathSet;
            readonly ClipBoardPane.FontResource[] _fontPathSet;
            readonly ClipBoardPane.CaptureTextureResource[] _captureTextureResourceSet;
            readonly InternalPane[] _paneSet;

            public Dictionary<string, TexImagePixelFmt> TextureFormatMap { get; private set; }

            /// <summary>
            ///
            /// </summary>
            public InternalClipBoardPane(
                PaneSetDupulicator.IntermediateData paneData,
                InternalPane[] paneSet,
                string[] texturePathSet,
                ClipBoardPane.FontResource[] fontPathSet,
                Dictionary<string, TexImagePixelFmt> textureFormatMap,
                ClipBoardPane.CaptureTextureResource[] captureTextureResourceSet)
            {
                _paneData = paneData;
                _paneSet = paneSet;
                _texturePathSet = texturePathSet;
                _fontPathSet = fontPathSet;
                _captureTextureResourceSet = captureTextureResourceSet;

                TextureFormatMap = new Dictionary<string, TexImagePixelFmt>(textureFormatMap);
            }

            public PaneSetDupulicator.IntermediateData PaneData { get { return _paneData; } }
            public InternalPane[] PaneSet { get { return _paneSet; } }
            public string[] TexturePathSet { get { return _texturePathSet; } }
            public ClipBoardPane.FontResource[] FontPathSet { get { return _fontPathSet; } }
            public ClipBoardPane.CaptureTextureResource[] CaptureTextureResourceSet { get { return _captureTextureResourceSet; } }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEClipboardPaneSetConvertFunc()
            :base( new DataConvertFunc( ToSavableFunc_ ),
            new DataConvertFunc( ToInternalFunc_ ),
            typeof( ClipBoardPane ),
            typeof( InternalClipBoardPane ) )
        {
        }

        /// <summary>
        /// 親ペイン情報を変換します。
        /// </summary>
        /// <param name="srcPaneSet"></param>
        /// <returns></returns>
        static Hashtable
            ConvertToSaveParentPaneSet_( IPane[] srcPaneSet )
        {
            Hashtable          parentPaneTable = new Hashtable();
            foreach( Pane pane in srcPaneSet )
            {
                IPane    parentPane = pane.Parent as IPane;
                if( Array.IndexOf( srcPaneSet, parentPane ) == -1 )
                {
                    parentPaneTable.Add( pane.PaneName, pane.ParentMtx );
                }
            }

            return parentPaneTable;
        }

        /// <summary>
        /// シリアライズ可能なオブジェクトへ変換します。
        /// </summary>
        static object ToSavableFunc_( object internalPanes )
        {
            Debug.Assert( internalPanes is ClipBoardPane );

            ClipBoardPane    srcClipBoardPane = internalPanes as ClipBoardPane;

            PaneSetDupulicator.IntermediateData interData
                = PaneSetDupulicator.ToIntermediate(
                srcClipBoardPane.PaneSet,
                srcClipBoardPane.Option );

            List<InternalPane> interPaneSet = new List<InternalPane>();
            foreach (IPane pane in srcClipBoardPane.PaneSet)
            {
                InternalPane interPane = new InternalPane(pane);
                interPaneSet.Add(interPane);
            }

            return new InternalClipBoardPane(
                interData,
                interPaneSet.ToArray(),
                srcClipBoardPane.TextureResourcePathSet,
                srcClipBoardPane.FontResourcePathSet,
                srcClipBoardPane.TextureFormatMap,
                srcClipBoardPane.CaptureTextureResourceSet);
        }

        /// <summary>
        /// アプリケーション内部オブジェクトへ変換します。
        /// </summary>
        static object ToInternalFunc_( object serializedPanes )
        {
            InternalClipBoardPane internalClipBoardPane
                = serializedPanes as InternalClipBoardPane;

            IPane[] dstPanes = PaneSetDupulicator.ToPaneSet( internalClipBoardPane.PaneData );

            SetBaseValueInternal_(dstPanes, internalClipBoardPane);

            return new ClipBoardPane(
                dstPanes,
                PaneSetDupulicator.Option.None,
                internalClipBoardPane.TexturePathSet,
                internalClipBoardPane.FontPathSet,
                internalClipBoardPane.TextureFormatMap,
                internalClipBoardPane.CaptureTextureResourceSet);
        }

        /// <summary>
        /// baseValueを設定します。
        /// </summary>
        static void SetBaseValueInternal_(IPane[] dstPanes, InternalClipBoardPane internalClipBoardPane)
        {
            // internalClipBoardPane.PaneSetに保持されている情報を参照します
            foreach (InternalPane pane in internalClipBoardPane.PaneSet)
            {
                // ペイン名で貼り付け対象を検索
                var targetPanes = dstPanes.Where(dst => dst.PaneName == pane.Name);
                if (targetPanes == null || targetPanes.Count() == 0) continue;

                Debug.Assert(targetPanes.Count() == 1);
                var targetPane = targetPanes.First(); // targetPanesはCount=1になる想定

                // アニメーションを持つアトリビュートを抽出します
                List<AttributeData> subAttrSet = new List<AttributeData>();
                foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
                {
                    IAnmAttribute[] dstAttrSet = targetPane.GetAnimationTargetAttributeSet(kind);
                    AttributeData[] attrData = pane.Attribute[kind];

                    for (int i = 0; i < attrData.Count(); i++)
                    {
                        SetBaseValueInternalRecursive_(attrData[i], dstAttrSet[i]);
                    }
                }
            }
        }

        static void SetBaseValueInternalRecursive_(AttributeData srcData, IAnmAttribute dstIAttr)
        {
            Debug.Assert(srcData.Name == dstIAttr.Name);

            AnmAttribute dstAttr = dstIAttr as AnmAttribute;

            // baseValueを更新
            if (dstAttr.CanGetOrSetValue())
            {
                dstAttr.SetBaseValue(srcData.BaseValue);
            }

            // サブアトリビュートを処理する
            for (int i = 0; i < srcData.SubAttribute.Count(); i++)
            {
                SetBaseValueInternalRecursive_(srcData.SubAttribute[i], dstAttr.SubAttrArray[i]);
            }
        }
    }





    /// <summary>
    /// IAnmCurve[] をコピーペーストする際に使用される変換関数セットです。
    /// </summary>
    internal class LEClipboardAnmAttributeSetConvertFunc : LEClipboardConvertFunc
    {

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEClipboardAnmAttributeSetConvertFunc()
            : base( new DataConvertFunc( ToSavableFunc_ ),
            new DataConvertFunc( ToInternalFunc_ ),
            typeof( LEClipboardAnmAttribute[] ),
            typeof( LEClipboardAnmAttribute[] ) )
        {

        }

        /// <summary>
        /// シリアライズ可能なオブジェクトへ変換します。
        /// </summary>
        static object ToSavableFunc_( object obj )
        {

            return obj;
        }

        /// <summary>
        /// アプリケーション内部オブジェクトへ変換します。
        /// </summary>
        static object ToInternalFunc_( object obj )
        {
            LEClipboardAnmAttribute[] keyFrameSet = obj as LEClipboardAnmAttribute[];
            Debug.Assert( keyFrameSet != null );

            return keyFrameSet;
        }

    }



    #endregion

    /// <summary>
    /// レイアウトエディタで使用される、クリップボードオブジェクトです。
    /// コピー・ペースト機能の実現に使用されます。
    ///
    /// コピー動作では、オブジェクトがシリアル化可能なオブジェクトに変換されます。
    /// シリアル化されたオブジェクトは、Clipboard に 保存されます。
    ///
    /// ユーザは事前に（オブジェクト => シリアル化オブジェクト）変換を実装する、
    /// デリゲータを登録する必要があります。
    ///
    /// デリゲータの登録はRegisterConvertionFuncSet()関数を使用します。
    /// グローバルなアプリケーション初期化のタイミングで行ってください。
    /// </summary>
    internal class LEClipboard :
        ILEClipboard
    {
        #region ------------ フィールド ------------

        // LEClipboardConvertFunc と System.Type の マップ
        // LEClipboard は入力のあったクラスの型をキーとして本テーブルを引き
        // 変換に使用すべき関数セットを取得します。
        Hashtable                       _ConvertFuncTbl = new Hashtable();

        /// <summary>
        /// シングルトン実体
        /// </summary>
        static LEClipboard              _theInstance = null;

        /// <summary>
        /// クリップボードの内容
        /// </summary>
        object                          _contents    = null;

        #endregion ------------ フィールド ------------


        #region ------------ プロパティ ------------
        /// <summary>
        /// シングルトンインスタンスを取得します。
        /// </summary>
        static public LEClipboard  Instance
        {
            get
            {
                if( _theInstance == null )
                {
                    _theInstance = new LEClipboard();
                }
                return _theInstance;
            }
        }
        #endregion ------------ プロパティ ------------



        /// <summary>
        /// 変換関数セットを探します。
        /// </summary>
        /// <param name="type"></param>
        /// <returns></returns>
        LEClipboardConvertFunc
            FindConvertionFuncSetFromLEInstaceType_( System.Type type )
        {
            if( _ConvertFuncTbl.ContainsKey( type ) )
            {
                return _ConvertFuncTbl[type] as LEClipboardConvertFunc;
            }
            return null;
        }




        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEClipboard()
        {
        }

        /// <summary>
        /// 変換関数を登録します。
        /// </summary>
        /// <param name="func">登録関数セット</param>
        /// <returns>登録の成否</returns>
        public bool RegisterConvertionFuncSet( LEClipboardConvertFunc func )
        {
            Debug.Assert( func != null );

            if( !_ConvertFuncTbl.ContainsKey( func.InternalObjType ) )
            {
                _ConvertFuncTbl[func.InternalObjType] = func;
                return true;
            }
            return false;
        }

        /// <summary>
        /// クリップボードにコピーします。
        /// </summary>
        /// <param name="srcInstance">コピー元オブジェクト</param>
        /// <returns>成否</returns>
        public bool Copy( object srcInstance )
        {
            // 自分の型に適する、変換関数をさがします。
            // 変換関数が発見されたら、変換関数を使用して変換し、クリップボードに登録します。
            LEClipboardConvertFunc clipboardConvertFunc =
               FindConvertionFuncSetFromLEInstaceType_( srcInstance.GetType() );
            if( clipboardConvertFunc != null )
            {
                _contents = clipboardConvertFunc.IntrnalToSavable( srcInstance );

                // クリップボードの内容が更新されたことをシーンに通知します。
                SceneModifyEventArgs ea =
                    SceneModifyEventArgs.CreateClipBoardModifyEventArgs( srcInstance );
                Scene.Instance.OnSceneResourceChanged( ea );

                return true;
            }
            else
            {
                // 未知のオブジェクトがコピーされました。
                // RegisterConvertionFuncSet()関数を利用して、
                // 適切な　LEClipboardConvertFunc　を登録してください。
                Debug.Assert( false, "Unknown Object is Copied. Register the proper LEClipboardConvertFunc for the object by using RegisterConvertionFuncSet()" );
            }

            return false;
        }

        /// <summary>
        /// クリップボードの内容を実体化して取得します。
        /// </summary>
        /// <param name="dstInstType">実体化するオブジェクトの型</param>
        /// <returns>失敗した場合はnullが返ります。</returns>
        public object Paste( System.Type  dstInstType )
        {
            // 貼り付けたい型に適する変換関数を探します。
            LEClipboardConvertFunc clipboardConvertFunc =
                FindConvertionFuncSetFromLEInstaceType_( dstInstType );
            if( clipboardConvertFunc != null )
            {
                // 変換関数の入力型に相当する型が
                // クリップボードに保存されているか探します。
                // 発見されたら、変換関数の変換結果を返します。
                if( _contents != null )
                {
                    if( clipboardConvertFunc.SavableObjType == _contents.GetType() )
                    {
                        return clipboardConvertFunc.SavableToInternal( _contents );
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// 貼り付けが可能な状態か判定します。
        /// </summary>
        public bool CanPaste( System.Type dstInstType )
        {
            // 貼り付けたい型に適する変換関数を探します。
            LEClipboardConvertFunc clipboardConvertFunc =
                FindConvertionFuncSetFromLEInstaceType_( dstInstType );

            if( clipboardConvertFunc != null && _contents != null )
            {
                return clipboardConvertFunc.SavableObjType == _contents.GetType();
            }
            return false;
        }

        /// <summary>
        /// クリップボードの内容を破棄します。
        /// </summary>
        public void Reset()
        {
            // TODO:Dispose も必要かもしれない。
            _contents = null;
        }

    }
}
