﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;

namespace LECore.Structures
{
    using Core;
    /// <summary>
    /// クリップボード用アトリビュート
    /// </summary>
    public class LEClipboardAnmAttribute
    {
        readonly List<LEClipboardAnmAttribute> _subAttributeSet = new List<LEClipboardAnmAttribute>();
        readonly List<IAnmKeyFrame> _keyFrameSet = new List<IAnmKeyFrame>();
        readonly AttributeType _type;
        readonly string _ownerName;

        public int NumSubAttribute { get { return _subAttributeSet.Count; } }
        public IAnmKeyFrame[] IKeyFrameSet { get { return _keyFrameSet.ToArray(); } }
        public AttributeType TargetAttrType { get { return _type; } }
        public string OwnerName { get { return _ownerName; } }
        public LEClipboardAnmAttribute[] SubAttribute { get { return _subAttributeSet.ToArray(); } }



        /// <summary>
        /// コンストラクタ
        /// keySetに指定したキーだけ、コピーされます。
        /// keySetにゼロを指定した場合は、すべてのキーがコピーされます。
        /// </summary>
        public LEClipboardAnmAttribute( IAnmAttribute srcAttr, IAnmKeyFrame[] keySet )
        {
            if( srcAttr.ICurrentAnimationCurve != null )
            {
                _type = srcAttr.ICurrentAnimationCurve.TargetAttrType;
                _ownerName = srcAttr.OwnerName;
                foreach( AnmKeyFrame srcKey in srcAttr.ICurrentAnimationCurve.IKeyFrameSet )
                {
                    bool bExist = ( keySet.Length > 0 ) ?
                    Array.Exists( keySet, delegate( IAnmKeyFrame key )
                    {
                        return key == srcKey;
                    } ) : true;

                    if( bExist )
                    {

                        AnmKeyFrame dstKey = new AnmKeyFrame(
                            srcKey.OwnerAnmCurve,
                            srcKey.Time,
                            srcKey.ValueAsFloat,
                            srcKey.UnifyTangents,
                            srcKey.InTangent,
                            srcKey.OutTangent,
                            srcKey.InterporationType );

                        _keyFrameSet.Add( dstKey );
                    }
                }
            }
            else
            {
                _type = AttributeType.Combined;
            }

            // 子アトリビュートについて、再起的に処理します。
            for( int i = 0 ; i < srcAttr.NumSubAttribute ; i++ )
            {
                _subAttributeSet.Add( new LEClipboardAnmAttribute( srcAttr.FindSubAttributeByIdx( i ), keySet ) );
            }
        }

        /// <summary>
        /// 有効か判定します。
        /// 再帰的に判定します。
        /// </summary>
        static public bool CheckPasteValid( IAnmAttribute dstAttr, LEClipboardAnmAttribute clipBoardAttr )
        {
            int numSub = clipBoardAttr._subAttributeSet.Count;

            // サブアトリビュートが無い場合は、
            // アトリビュート種類が同一かで判断します。
            if( numSub <= 0 )
            {
                // コピー元が FloatColor ならば、コピー先も FloatColor でなければNG
                // FloatColor は TargetAttrType が float となり、位置などのカーブと判別できないためアトリビュートの名前で判断する。
                if (IsFloatColorAttr(clipBoardAttr.OwnerName) != IsFloatColorAttr(dstAttr.OwnerName))
                {
                    return false;
                }

                if( dstAttr.ICurrentAnimationCurve != null )
                {
                    return clipBoardAttr.TargetAttrType == dstAttr.ICurrentAnimationCurve.TargetAttrType;
                }
            }

            // サブアトリビュート数が一致しなければNG
            if( numSub != dstAttr.NumSubAttribute )
            {
                return false;
            }

            // サブアトリビュート以下がすべて有効でなければNG
            for( int i = 0 ; i < numSub ; i++ )
            {
                if( !CheckPasteValid(
                    dstAttr.FindSubAttributeByIdx( i ),
                    clipBoardAttr._subAttributeSet[i] ) )
                {
                    return false;
                }
            }

            // 有効
            return true;
        }

        /// <summary>
        /// FloatColor のアトリビュートかどうかを文字列(アトリビュートの名前)から判別します。
        /// </summary>
        static public bool IsFloatColorAttr(string ownerName)
        {
            // AnmAttrDescripter で FloatRGBA4 が設定されているアトリビュート。
            return ownerName.Equals("BlackColor") ||
                   ownerName.Equals("WhiteColor") ||
                   ownerName.Equals("_register0") ||
                   ownerName.Equals("_register1") ||
                   ownerName.StartsWith("TEV_color");
        }
    }
}
