﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Generic;
using System.Linq;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;

    public partial class RlytConverter
    {
        #region File => App

        private bool GetPaneRequiredFlag_(IPartsControlSetting srcControlSetting, string panePramaterName)
        {
            if (srcControlSetting == null) { return false; }

            var paneSetting = srcControlSetting.PaneNames.FirstOrDefault((setting) => setting.Name == panePramaterName);
            if (paneSetting == null) { return false; }

            return paneSetting.IsRequired;
        }

        private bool GetAnimRequiredFlag_(IPartsControlSetting srcControlSetting, string animPramaterName)
        {
            if (srcControlSetting == null) { return false; }

            var paneSetting = srcControlSetting.AnimationNames.FirstOrDefault((setting) => setting.Name == animPramaterName);
            if (paneSetting == null) { return false; }

            return paneSetting.IsRequired;
        }

        /// <summary>
        ///
        /// </summary>
        private void Convert_Parts_(
            FileFmt.LYT rlyt,
            AppData.ISubScene subScene)
        {
            // 設定済みなら何もしません
            // (例外としてはじいていましたが、
            // 追加読み込みなど別部品からペインだけを取り込みたい場合があるのでスルーするように変更しました)
            AppData.IPartsSettings dstPartsSettings = subScene.IPartsSettings;
            if (!dstPartsSettings.IsInitialState)
            {
                return;
            }

            if (string.IsNullOrEmpty(rlyt.partsName))
            {
                return;
            }

            // 部品設定を初期化
            IPartsControlSetting srcControlSetting = Scene.Instance.FindPartsControlSetting(rlyt.partsName);
            string partsName = rlyt.control != null ? rlyt.control[0].name : rlyt.partsName;

            (dstPartsSettings as PartsSettings).Initialize(
                partsName,
                rlyt.partsName,
                SerializableConverterUtil.ReplaceLineFeedCode(rlyt.partsDescription),
                new FVec3(Convert_FVec2_(rlyt.partsSize)));

            // IPartsPropaertySettings
            FileFmt.PropertyForm[] propertyForm = rlyt.propertyForm;
            var propSettings = new List<PartsPropaertySettingsSource>();
            if (propertyForm != null)
            {
                foreach (var property in propertyForm)
                {
                    // 正確な原因は把握できていないが、ターゲット名が無い不正データが
                    // 生成されることがあるので、読み込み時に破棄するようにする。
                    if (string.IsNullOrEmpty(property.target)) { continue; }

                    PartsPropaertySettingsSource prop = new PartsPropaertySettingsSource(
                        property.target, Convert_SOPaneKind_(property.kind), property.description);
                    propSettings.Add(prop);
                }
            }

            (dstPartsSettings as PartsSettings).SetOverrideProperties(propSettings);
        }

        /// <summary>
        /// Control(IPartsSettings)のファイル側からアプリケーション側への変換を行います。
        /// </summary>
        private void Convert_Control_(
            FileFmt.LYT rlyt,
            AppData.ISubScene subScene)
        {
            AppData.IPartsSettings dstPartsSettings = subScene.IPartsSettings;

            if (rlyt.control == null)
            {
                return;
            }

            var controlSettingsSet = (subScene as SubScene).ControlSettingsSet;
            controlSettingsSet.Clear();

            foreach (FileFmt.Control srcControl in rlyt.control)
            {
                IPartsControlSetting srcControlSetting = Scene.Instance.FindPartsControlSetting(srcControl.name);
                string descriptions = srcControlSetting != null ? srcControlSetting.Description : rlyt.partsDescription;// 両者は必ず同じ内容なはず。

                // IParamaterPane
                List<AppData.IParamaterPane> dstPanes = new List<AppData.IParamaterPane>();
                if (srcControl.parameterPane != null)
                {
                    // コントロール定義に従って、パラメータを登録する。
                    if (srcControlSetting != null)
                    {
                        foreach (PartsParamaterTemplate srcParamPane in srcControlSetting.PaneNames)
                        {
                            // ファイルに情報が記録されている情報があれば優先的に使用して初期化
                            FileFmt.ControlParameterPane savedSControlParmPane = srcControl.parameterPane.FirstOrDefault((savedCtrlPane) => savedCtrlPane.name == srcParamPane.Name);
                            string paramPaneName = (savedSControlParmPane != null) ? savedSControlParmPane.name : srcParamPane.Name;
                            string paneName = (savedSControlParmPane != null) ? savedSControlParmPane.paneName : string.Empty;

                            dstPanes.Add(new AppData.ParamaterPane(paramPaneName, paneName, GetPaneRequiredFlag_(srcControlSetting, paramPaneName)));
                        }
                    }

                    // 廃止されたパラメータ(ファイルに保存されているが、コントロール定義から消えているもの)
                    // はリスト末尾に追加しておく。
                    foreach (var unfoundParam in srcControl.parameterPane.Where(
                        (srcParam) => dstPanes.All((dstPane) => dstPane.Name != srcParam.name)))
                    {
                        dstPanes.Add(new AppData.ParamaterPane(
                            unfoundParam.name, unfoundParam.paneName, GetPaneRequiredFlag_(srcControlSetting, unfoundParam.name)));
                    }
                }

                // IParamaterAnimaiton
                List<AppData.IParamaterAnimaiton> dstAnims = new List<AppData.IParamaterAnimaiton>();
                if (srcControl.parameterAnimation != null)
                {
                    // コントロール定義に従って、パラメータを登録する。
                    if (srcControlSetting != null)
                    {
                        foreach (PartsParamaterTemplate srcParamAnim in srcControlSetting.AnimationNames)
                        {
                            // ファイルに情報が記録されている情報があれば優先的に使用して初期化
                            FileFmt.ControlParameterAnimation savedSControlParmAnim = srcControl.parameterAnimation.FirstOrDefault((savedCtrlAnim) => savedCtrlAnim.name == srcParamAnim.Name);
                            string paramAnimName = (savedSControlParmAnim != null) ? savedSControlParmAnim.name : srcParamAnim.Name;
                            string tagName = (savedSControlParmAnim != null) ? savedSControlParmAnim.tagName : string.Empty;

                            dstAnims.Add(new AppData.ParamaterAnimaiton(paramAnimName, tagName, GetAnimRequiredFlag_(srcControlSetting, paramAnimName)));
                        }
                    }

                    // 廃止されたパラメータ(ファイルに保存されているが、コントロール定義から消えているもの)
                    // はリスト末尾に追加しておく。
                    foreach (var unfoundParam in srcControl.parameterAnimation.Where(
                        (srcParam) => dstAnims.All((dstAnim) => dstAnim.Name != srcParam.name)))
                    {
                        dstAnims.Add(new AppData.ParamaterAnimaiton(
                            unfoundParam.name, unfoundParam.tagName, GetAnimRequiredFlag_(srcControlSetting, unfoundParam.name)));
                    }
                }

                // コントロール定義を初期化
                ControlSettings newCtrl = new ControlSettings(controlSettingsSet);
                newCtrl.InitializeNoEvent(
                    srcControl.name,
                    srcControlSetting != null ? srcControlSetting.UIName : srcControl.name,
                    !string.IsNullOrEmpty(srcControl.userName) ? srcControl.userName : srcControl.name,
                    SerializableConverterUtil.ReplaceLineFeedCode(descriptions),
                    dstPanes,
                    dstAnims);

                // 拡張ユーザデータをアプリ形式に変換
                foreach (UserDataElement ud in GetUserDataElementSetImpl_(srcControl.userData))
                {
                    (newCtrl.IUserDataHolder as UserDataHolder).AddUserDataElement(ud.Name, ud.UserDataKind, ud.Overwrite, ud.Value);
                }

                controlSettingsSet.CloneAdd(newCtrl);
            }
        }

        private void Convert_LayoutUserData_(FileFmt.LYT rlyt, ISubScene subScene)
        {
            // 拡張ユーザデータをアプリ形式に変換
            foreach (UserDataElement ud in GetUserDataElementSetImpl_(rlyt.userData))
            {
                (subScene.IUserDataHolder as UserDataHolder).AddUserDataElement(ud.Name, ud.UserDataKind, ud.Overwrite, ud.Value);
            }
        }

        #endregion // File => App

        #region App => File

        /// <summary>
        /// IPartsSettingsのアプリケーション側からファイル側への変換を行います。
        /// </summary>
        /// <param name="scene">アプリケーション側データです。</param>
        /// <returns>ファイル側データです。</returns>
        private void Convert_ToFileParts_(AppData.ISubScene scene, FileFmt.LYT dstLyt)
        {
            AppData.IPartsSettings srcPartsSettings = scene.IPartsSettings;
            if (srcPartsSettings == null || srcPartsSettings.IsInitialState)
            {
                return;
            }

            dstLyt.partsDerivative = null; // 部品プリセットはもう使われなくなりました。
            dstLyt.propertyForm = Convert_ToFilePropertyForm_(scene);

            dstLyt.partsName = srcPartsSettings.DescriptionName;
            dstLyt.partsDescription = srcPartsSettings.Description;
            dstLyt.partsSize = Convert_Vec2_(PartsLayoutHelper.CalcPartsPaneSizeFromRectangle(PartsLayoutHelper.CalcPartsPaneRectanbleFromPanes(scene.IPaneArray)));
            dstLyt.partsDerivativeBaseName = srcPartsSettings.BasePartsName;
            dstLyt.partsDerivativeBaseLastModifyDate = srcPartsSettings.BasePartsLastModify.ToString();
            dstLyt.partsDerivativeBaseHashValue = srcPartsSettings.BasePartsHashValue;
        }

        /// <summary>
        /// Controlのアプリケーション側からファイル側への変換を行います。
        /// </summary>
        private FileFmt.Control[] Convert_ToFileControlSet_(AppData.ISubScene scene)
        {
            return scene.IControlSettings.Select<AppData.IControlSettings, FileFmt.Control>((srcCtrl) => Convert_ToFileControl_(srcCtrl)).ToArray();
        }

        /// <summary>
        /// Control(IPartsSettings)のアプリケーション側からファイル側への変換を行います。
        /// </summary>
        /// <param name="scene">アプリケーション側データです。</param>
        /// <returns>ファイル側データです。</returns>
        private FileFmt.Control Convert_ToFileControl_(AppData.IControlSettings srcControlSettings)
        {
            if (srcControlSettings == null)
            {
                return null;
            }

            var controlSettingTemplate = LayoutEditorCore.Scene.PartsControlSettings.FirstOrDefault((setting) => setting.Name == srcControlSettings.Name);
            FileFmt.Control dstControl = new FileFmt.Control();

            // Name
            dstControl.name = srcControlSettings.Name;
            dstControl.userName = srcControlSettings.DescriptionName;

            // IParamaterPane
            dstControl.parameterPane = srcControlSettings.Panes.Select((srcPane) =>
            {
                // 未設定でかつ、コントロール定義からも見つからない場合は保存しません。
                if (string.IsNullOrEmpty(srcPane.PaneName))
                {
                    if (controlSettingTemplate != null &&
                        !controlSettingTemplate.PaneNames.Any((paneParamTemplate) => paneParamTemplate.Name == srcPane.Name))
                    {
                        return null;
                    }
                }

                var dstPane = new FileFmt.ControlParameterPane();
                dstPane.name = srcPane.Name;
                dstPane.paneName = srcPane.PaneName;
                return dstPane;
            }).ToArray();

            // IParamaterAnimaiton
            dstControl.parameterAnimation = srcControlSettings.Animations.Select((srcAnim) =>
            {
                // 未設定でかつ、コントロール定義からも見つからない場合は保存しません。
                if (string.IsNullOrEmpty(srcAnim.TagName))
                {
                    if (controlSettingTemplate != null &&
                        !controlSettingTemplate.AnimationNames.Any((animParamTemplate) => animParamTemplate.Name == srcAnim.Name))
                    {
                        return null;
                    }
                }

                var dstAnim = new FileFmt.ControlParameterAnimation();
                dstAnim.name = srcAnim.Name;
                dstAnim.tagName = srcAnim.TagName;
                return dstAnim;
            }).ToArray();

            // 拡張ユーザデータをファイル形式に変換
            dstControl.userData = Convert_UserDataEx_(srcControlSettings.IUserDataHolder.UserDataElementSet);

            return dstControl;
        }

        /// <summary>
        /// Control(IPartsSettings)のアプリケーション側からファイル側への変換を行います。
        /// </summary>
        /// <param name="scene">アプリケーション側データです。</param>
        /// <returns>ファイル側データです。</returns>
        private FileFmt.PropertyForm[] Convert_ToFilePropertyForm_(AppData.ISubScene scene)
        {
            AppData.IPartsSettings srcPartsSettings = scene.IPartsSettings;
            if (srcPartsSettings == null)
            {
                return null;
            }

            return Convert_ToFilePropertyForm_(srcPartsSettings.OverridePorperties);
        }

        /// <summary>
        ///
        /// </summary>
        private FileFmt.PropertyForm[] Convert_ToFilePropertyForm_(
            IEnumerable<IPartsPropaertySettings> srcPorperties)
        {
            List<FileFmt.PropertyForm> dstPropertyForms = new List<FileFmt.PropertyForm>();
            foreach (var srcProp in srcPorperties)
            {
                FileFmt.PropertyForm newProp = new FileFmt.PropertyForm();
                newProp.kind = Convert_SOPaneKind_(srcProp.PaneKind);
                newProp.target = srcProp.TargetName;
                newProp.description = srcProp.Description;
                dstPropertyForms.Add(newProp);
            }

            return dstPropertyForms.ToArray();
        }

        #endregion // App => File
    }
}
