﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Diagnostics;
using System.Linq;
using System.Xml.Serialization;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.SerializableObject.Lan;


    /// <summary>
    /// rlan(レイアウト情報)を出力するクラスです。
    /// </summary>
    internal class RlanExporter : BaseSingleExporter
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RlanExporter()
        {
            Initialize( null );
        }

        #region IExporter メンバ
        /// <summary>
        /// 出力ファイルの拡張子を取得します(短縮)。
        /// </summary>
        public override string[] GetFileExtensionShort()
        {
            return new string[] { AppConstants.AnimationFileExtNoDot };
        }

        /// <summary>
        /// 出力ファイルの拡張子を取得します(長い)。
        /// </summary>
        public override string[] GetFileExtensionLong()
        {
            return new string[] { AppConstants.AnimationFileDesc };
        }

        /// <summary>
        /// 出力ファイルの拡張子を取得します(長い)。
        /// </summary>
        public override Type GetSelializableObjType()
        {
            return typeof( Document );
        }

        /// <summary>
        /// 出力するものが何もない場合は false を返すと、一切の処理がスキップされます。
        /// </summary>
        public override bool CheckNothingToExport(ISubScene scene)
        {
            return ConvertToSerializableImpl_(scene as SubScene, null) == null;
        }

        /// <summary>
        /// XML シリアライズ可能な、クラスに変換します。
        /// </summary>
        public override object ConvertToSerializable_( SubScene targetScene, string outPath, LEMsgReporter msgReporter )
        {
            return ConvertToSerializableImpl_(targetScene, msgReporter);
        }

        /// <summary>
        /// XML シリアライズ可能な、クラスに変換します。
        /// </summary>
        private object ConvertToSerializableImpl_(SubScene targetScene, LEMsgReporter msgReporter)
        {
            // XML シリアライズ可能な、クラスに変換します。
            RlanConverter converter = new RlanConverter(msgReporter);
            converter.SaveOptions = this.ExportOption;

            // コンバートクラスを使用して、シーンからアニメーション情報を格納する
            // シリアル化可能クラスへと変換します。
            Document anmDoc = converter.Convert_SODocument(targetScene, ParamaterKind.Animation_All);
            Debug.Assert(anmDoc != null);

            // 空の場合は何も出力しません。
            return (
                anmDoc.body.lan.Length > 0 ||
                anmDoc.body.animTag.Length > 0 ||
                anmDoc.body.animShare.Length > 0 ||
                anmDoc.body.separateMode) ? anmDoc : null;
        }
        #endregion IExporter メンバ
    }
}

