﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;

namespace LECore.Save_Load
{
    using Structures;
    using Structures.Core;
    using Structures.LECoreInterface;
    using Structures.SerializableObject.Rlan;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Rlan;

    /// <summary>
    /// RlanConverter の概要の説明です。
    /// </summary>
    public class RlanConverter
    {
        #region        フィールド
        /// <summary>
        /// メッセージ報告クラス
        /// </summary>
        readonly LEMsgReporter _msgReporter = null;
        ParamaterKind _animationFileType = ParamaterKind.Animation_PaneSRT;
        static readonly string _VersionString = "1.0.0";
        bool _fixInfinityCurve = false;

        #endregion     フィールド

        #region プロパティ
        /// <summary>
        /// キー領域外のカーブを焼き付けるか？
        /// </summary>
        public bool FixInfinityCurve
        {
            get { return _fixInfinityCurve; }
            set { _fixInfinityCurve = value; }
        }
        #endregion プロパティ

        /// <summary>
        /// 警告メッセージを表示します。
        /// </summary>
        /// <param name="msg"></param>
        void ReporterWarningMsg_( string msg )
        {
            string title = StringResMgr.Get( "LECORE_SYS_ERROR_FILE_EXPORT", _animationFileType.ToString() );
            _msgReporter.ReportWarning( title, msg );
        }

        /// <summary>
        /// ログ出力します。
        /// </summary>
        void OutLog_( string msgID )
        {
            if( _msgReporter != null )
            {
                _msgReporter.OutLogByID( msgID );
            }
        }

        #region FileFmt => AppData

        /// <summary>
        /// グローバル時間関連のパラメータを設定します。
        /// </summary>
        void SetGrobalTime_( FileFmt.RLAN srcRlan )
        {
            GrobalTime gTime = GrobalTime.Inst;

            bool bChange =
                gTime.AnimPlayStartTime != srcRlan.convertStartFrame ||
                gTime.AnimPlayEndTime != srcRlan.convertEndFrame ||
                gTime.AnimStartTime != srcRlan.startFrame ||
                gTime.AnimEndTime != srcRlan.endFrame;

            if( bChange )
            {
                bool bResult = gTime.Set(
                    srcRlan.startFrame,
                    srcRlan.endFrame,
                    srcRlan.convertStartFrame,
                    srcRlan.convertEndFrame );

                if( bResult == false )
                {
                    string title = StringResMgr.Get( "LECORE_DLG_TITLE_LOADANIM" );
                    string msg = StringResMgr.Get( "LECORE_SYS_ERROR_INVALID_ANIMFRAME" );
                    _msgReporter.ReportInfomation( title, msg );
                }
                // "LECORE_SYS_MSG_CHANGE_FRAME_RANGE"
                // フレーム更新通知メッセージは廃止しました。
            }
        }

        /// <summary>
        /// FileFmt.Document から ParamaterKind に対応する RLAN を取得します。
        /// </summary>
        FileFmt.RLAN
            GetRlan_( FileFmt.Document srcDocument, ParamaterKind animationFileType )
        {
            foreach( FileFmt.RLAN srcRlan in srcDocument.body.rlan )
            {
                if( srcRlan.animType == ParamaterKindToAnimationType_( animationFileType ) )
                {
                    return srcRlan;
                }
            }
            // 発見できませんでした。
            return null;
        }

        /// <summary>
        /// AnimTag のフラグ情報を ParamaterKind へ変換します。
        /// </summary>
        /// <param name="animTag"></param>
        /// <returns></returns>
        ParamaterKind ConvertAnimTagFlagsToParamaterKind_( AnimTag animTag )
        {
            ParamaterKind result = ParamaterKind.None;

            if( animTag.outputPaneSRT ) { result |= ParamaterKind.Animation_PaneSRT; }
            if( animTag.outputVertexColor ) { result |= ParamaterKind.Animation_VertexColors; }
            if( animTag.outputVisibility ) { result |= ParamaterKind.Animation_Visivility; }
            if( animTag.outputMaterialColor ) { result |= ParamaterKind.Animation_MaterialColors; }
            if( animTag.outputTexturePattern ) { result |= ParamaterKind.Animation_TexturePattern; }
            if( animTag.outputTextureSRT ) { result |= ParamaterKind.Animation_TextureSRT; }
            if( animTag.outputIndTextureSRT ) { result |= ParamaterKind.Animation_IndirectTextureSRT; }

            return result;
        }


        #endregion FileFmt => AppData

        #region AppData => FileFmt


        /// <summary>
        /// AppData.SubScene => FileFmt.RLVI
        /// </summary>
        FileFmt.RLAN
            Convert_AppDataToRLAN_( AppData.ISubScene srcSubScene )
        {
            FileFmt.RLAN rlan = new RLAN();

            rlan.animLoop = FileFmt.AnimLoopType.Loop; // TODO:

            rlan.startFrame = GrobalTime.Inst.AnimStartTime;
            rlan.endFrame = GrobalTime.Inst.AnimEndTime;
            rlan.convertStartFrame = GrobalTime.Inst.AnimPlayStartTime;
            rlan.convertEndFrame = GrobalTime.Inst.AnimPlayEndTime;
            rlan.animContent = null;

            return rlan;
        }

        /// <summary>
        /// アニメーションタグ情報を変換します。
        /// </summary>
        FileFmt.AnimTag[] ConvertAppFrameSectionDataToTagData_(
            AnimFrameSectionSet animFrameSectionSet )
        {
            List<AnimTag> animTagSet = new List<AnimTag>();
            IAnimFrameSection[] afsSet = animFrameSectionSet.IAnimFrameSectionSet;

            foreach( IAnimFrameSection srcAfs in afsSet )
            {
                AnimTag dstTag = new AnimTag();

                dstTag.name = srcAfs.Name;
                dstTag.comment = srcAfs.Comment;
                dstTag.startFrame = srcAfs.StartFrame;
                dstTag.endFrame = srcAfs.EndFrame;

                dstTag.outputEnabled = srcAfs.BinarySettings.Active;
                dstTag.fileName = srcAfs.BinarySettings.FileName;
                dstTag.animLoop = (AnimLoopType)srcAfs.BinarySettings.LoopType;

                // 出力アニメーション種類フラグを設定します。
                ParamaterKind animKind = srcAfs.BinarySettings.TargetAnimKind;
                dstTag.outputPaneSRT = ( animKind & ParamaterKind.Animation_PaneSRT ) != 0;
                dstTag.outputVertexColor = ( animKind & ParamaterKind.Animation_VertexColors ) != 0;
                dstTag.outputVisibility = ( animKind & ParamaterKind.Animation_Visivility ) != 0;
                dstTag.outputMaterialColor = ( animKind & ParamaterKind.Animation_MaterialColors ) != 0;
                dstTag.outputTexturePattern = ( animKind & ParamaterKind.Animation_TexturePattern ) != 0;
                dstTag.outputTextureSRT = ( animKind & ParamaterKind.Animation_TextureSRT ) != 0;
                // インダイレクトテクスチャSRTに関しては、フラグを共有することにします。
                dstTag.outputIndTextureSRT = dstTag.outputTexturePattern;

                animTagSet.Add( dstTag );
            }
            return animTagSet.ToArray();
        }

        /// <summary>
        /// AppData.SubScene => FileFmt.AnimMaterial[]
        /// </summary>
        FileFmt.AnimContent[]
            Convert_AppDataToAnimContentSet_( ParamaterKind kind, AppData.ISubScene srcSubScene )
        {
            // TextureSRT なら...
            switch( kind )
            {
                case ParamaterKind.Animation_TextureSRT: return RltsConverter.Convert_AppDataToAnimContentSet_Rlts_( srcSubScene );
                case ParamaterKind.Animation_IndirectTextureSRT: return RltsConverter.Convert_AppDataToAnimContentSet_Rlts_Indirect_( srcSubScene );
                case ParamaterKind.Animation_PaneSRT: return RlpaConverter.Convert_AppDataToAnimContentSet_Rlpa_( srcSubScene );
                case ParamaterKind.Animation_TexturePattern: return RltpConverter.Convert_AppDataToAnimContentSet_Rltp_( srcSubScene );
                case ParamaterKind.Animation_VertexColors: return RlvcConverter.Convert_AppDataToAnimContentSet_Rlvc_( srcSubScene );
                case ParamaterKind.Animation_MaterialColors: return RlmcConverter.Convert_AppDataToAnimContentSet_Rlmc_( srcSubScene );
                case ParamaterKind.Animation_Visivility: return RlviConverter.Convert_AppDataToAnimContentSet_Rlvi_( srcSubScene );

                default: Debug.Assert( false ); return null;
            }
        }

        #endregion AppData => FileFmt

        #region 列挙子変換
        FileFmt.AnimationType
            ParamaterKindToAnimationType_( ParamaterKind animationFileType )
        {
            switch( animationFileType )
            {
                case ParamaterKind.Animation_PaneSRT: return AnimationType.PainSRT;
                case ParamaterKind.Animation_VertexColors: return AnimationType.VertexColor;
                case ParamaterKind.Animation_Visivility: return AnimationType.Visibility;
                case ParamaterKind.Animation_TextureSRT: return AnimationType.TextureSRT;
                case ParamaterKind.Animation_TexturePattern: return AnimationType.TexturePattern;
                case ParamaterKind.Animation_MaterialColors: return AnimationType.MaterialColor;
                case ParamaterKind.Animation_IndirectTextureSRT: return AnimationType.IndTextureSRT;
                default: Debug.Assert( false ); return AnimationType.PainSRT;
            }
        }

        ParamaterKind
            AnimationTypeToParamaterKind_( FileFmt.AnimationType animationFileType )
        {
            switch( animationFileType )
            {
                case AnimationType.PainSRT: return ParamaterKind.Animation_PaneSRT;
                case AnimationType.VertexColor: return ParamaterKind.Animation_VertexColors;
                case AnimationType.Visibility: return ParamaterKind.Animation_Visivility;
                case AnimationType.TextureSRT: return ParamaterKind.Animation_TextureSRT;
                case AnimationType.TexturePattern: return ParamaterKind.Animation_TexturePattern;
                case AnimationType.MaterialColor: return ParamaterKind.Animation_MaterialColors;
                case AnimationType.IndTextureSRT: return ParamaterKind.Animation_IndirectTextureSRT;
                default: Debug.Assert( false ); return ParamaterKind.Animation_PaneSRT;
            }
        }
        #endregion 列挙子変換

        /// <summary>
        /// アニメーションをシーンに読み込みます。
        /// </summary>
        public void SetDataToScene(
            AppData.ISubScene dstScene,
            FileFmt.Document srcDocument,
            ParamaterKind animationFileType )
        {
            // アニメーション区間タグの読み込み
            dstScene.BeginMassiveModify();
            if( srcDocument != null && srcDocument.body != null )
            {
                SetTagDataToScene( dstScene, srcDocument.body.animTag );
            }
            dstScene.EndMassiveModify();


            // アニメーション区間フレームを読み込みます。
            foreach( ParamaterKind animKind in ParamaterKindHelper.AnimationKindSet )
            {
                // 指定されていれば...
                if( ( animationFileType & animKind ) != 0 )
                {
                    // 指定された animationFileType にふさわしい、rlan を 取得します。
                    FileFmt.RLAN srcRlan = GetRlan_( srcDocument, animKind );
                    if( srcRlan != null && srcRlan.animContent.Length != 0 )
                    {
                        // 一度でも、読み込み試行をおこなえば終了します。
                        SetGrobalTime_( srcRlan );
                        break;
                    }
                }
            }

            dstScene.BeginMassiveModify();
            // 全てのアニメーションについて...
            foreach( ParamaterKind animKind in ParamaterKindHelper.AnimationKindSet )
            {
                // 指定されていなければ中断
                if( ( animationFileType & animKind ) == 0 )
                {
                    continue;
                }

                // 指定された animationFileType にふさわしい、rlan を 取得します。
                FileFmt.RLAN srcRlan = GetRlan_( srcDocument, animKind );
                if( srcRlan == null || srcRlan.animContent.Length == 0 )
                {
                    continue;
                }

                OutLog_( "LECORE_RLANLOAD_LOAD_ANIM" );

                // 読み込み関数を実行
                foreach( RlanHelper.AnimLoadFunction loader in
                    RlanHelper.GetAnimLoadFunctionSet( animKind ) )
                {
                    RlanHelper.DoLoad( loader, dstScene.IPaneArray, srcRlan.animContent );
                }
            }
            dstScene.EndMassiveModify();
        }

        /// <summary>
        /// タグ情報をファイルフォーマットに書き出します。
        /// </summary>
        public FileFmt.AnimTag[] SetTagDataToFileFmt(
            AnimFrameSectionSet animFrameSectionSet )
        {
            return ConvertAppFrameSectionDataToTagData_( animFrameSectionSet );
        }

        /// <summary>
        /// タグ情報をシーンに読み込みます。
        /// </summary>
        public void SetTagDataToScene(
            AppData.ISubScene dstScene,
            AnimTag[]			animTagSet )
        {
            if( animTagSet != null )
            {
                OutLog_( "LECORE_RLANLOAD_LOAD_ANIMTAG" );

                foreach( AnimTag animTag in animTagSet )
                {
                    string comment = SerializableConverterUtil.ReplaceLineFeedCode( animTag.comment );
                    IAnimFrameSection frameSection =
                        dstScene.AnimFrameSectionSet.AddNewFrameSection(
                        animTag.name,
                        comment,
                        animTag.startFrame,
                        animTag.endFrame );

                    bool result = true;
                    if( frameSection != null )
                    {
                        result = dstScene.AnimFrameSectionSet.SetOutBinayActive( frameSection, animTag.outputEnabled );
                        result = dstScene.AnimFrameSectionSet.SetOutBinayFileName( frameSection, animTag.fileName );
                        result = dstScene.AnimFrameSectionSet.SetOutBinayLoopType( frameSection, (AnimationLoopType)animTag.animLoop );

                        ParamaterKind animParamKind = ConvertAnimTagFlagsToParamaterKind_( animTag );
                        result = dstScene.AnimFrameSectionSet.SetOutBinayParamaterKind( frameSection, animParamKind );
                    }
                }
            }
        }

        /// <summary>
        /// アニメーションを書き出します。
        /// </summary>
        public FileFmt.Document Convert_SODocument(
            ISubScene scene,
            ParamaterKind kind )
        {
            // ------------ RLAN を書き出します。
            // 全ての種類のアニメから...
            List<RLAN> rlanSet = new List<RLAN>();
            foreach( ParamaterKind paramKind in
                ParamaterKindHelper.AnimationKindSet )
            {
                _animationFileType = paramKind;

                // 指定された種類のアニメのみを出力します。
                if( ( paramKind & kind ) != 0 )
                {
                    // AnimContent を抽出します。
                    FileFmt.AnimContent[] contentSet =
                        Convert_AppDataToAnimContentSet_( paramKind, scene );

                    // 抽出されれば、RLANを生成して登録します。
                    if( contentSet.Length != 0 )
                    {
                        RLAN rlan = Convert_AppDataToRLAN_( scene );

                        rlan.animType = ParamaterKindToAnimationType_( paramKind );
                        rlan.animContent = contentSet;

                        // Infinity領域のキーフレーム焼付け処理
                        if( FixInfinityCurve )
                        {
                            Rlan.BakedInfinityKeyBuilder.BuildBakedKey(
                                rlan.animContent,
                                rlan.convertStartFrame,
                                rlan.convertEndFrame,
                                paramKind );
                        }

                        rlanSet.Add( rlan );
                    }
                }
            }

            // ------------ FileFmt.Document を書き出します。
            FileFmt.Document result = new Document();
            result.head = Convert_head_( null );
            result.body = new DocumentBody();
            result.version = _VersionString;

            // DocumentBody について初期化します。
            result.body.rlan = rlanSet.ToArray();
            result.body.animTag = ConvertAppFrameSectionDataToTagData_( scene.AnimFrameSectionSet );

            return result;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RlanConverter( LEMsgReporter msgReporter )
        {
            _msgReporter = msgReporter;
        }


        /// <summary>
        /// TODO:
        /// 名前空間が異なるだけで、全く同じデータ構造に対して
        /// 全く同じコードを複製している。
        ///
        /// 名前空間が異なるので、文法上は完全に別の型であるので、
        /// 同一のコードで処理が行えない。
        ///
        /// 何とか、共通化できないものか...
        ///
        /// </summary>
        #region FileFmt.Head ヘッダ情報関連


        /// <summary>
        /// FileFmt.HeadCreate へと変換します。
        /// </summary>
        FileFmt.HeadCreate Convert_headCreate_( object srcData )
        {
            FileFmt.HeadCreate result = new FileFmt.HeadCreate();

            result.date = DateTime.Now;

            result.host = Environment.MachineName;
            result.user = Environment.UserName;

            // TODO:
            result.source = "";

            return result;
        }

        /// <summary>
        /// FileFmt.HeadGenerator へと変換します。
        /// </summary>
        FileFmt.HeadGenerator Convert_headGenerator_( object srcData )
        {
            FileFmt.HeadGenerator rusult = new FileFmt.HeadGenerator();

            rusult.name = "";
            rusult.version = _VersionString;

            return rusult;
        }

        /// <summary>
        /// FileFmt.Head へと変換します。
        /// </summary>
        FileFmt.Head Convert_head_( object srcData )
        {
            FileFmt.Head result = new FileFmt.Head();

            result.comment = "";
            result.title = "";
            result.create = Convert_headCreate_( null );
            result.generator = Convert_headGenerator_( null );

            return result;
        }

        #endregion FileFmt.Head ヘッダ情報関連
    }
}
