﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Collections;
using System.Diagnostics;

namespace LECore.Save_Load
{
    using Structures;
    using Structures.Core;
    using Structures.SerializableObject.Lan;
    using System.Linq;
    using static RlanHelper;
    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lan;

    class RltsConverter
    {
        #region ------------------ Rlts(保存)
        /// <summary>
        /// ペインの持つテクスチャSRTアニメーションをシリアライズ可能なクラスに変換します。
        /// </summary>
        public static FileFmt.AnimContent[] SaveTextureSRTAnim(AppData.IPane srcPane, string tag)
        {
            return ConvertToAnimContents<IMaterial, AnimTexSRTTarget>(
                tag,
                ParamaterKind.Animation_TextureSRT,
                srcPane,
                x => x.UsedMaterials.Select(y => new AnimContentSource<IMaterial>(y.MaterialName, y)),
                x => x.ITexMtxSet.Select((y, i) => new { y, i }).Where(z => z.y != null).SelectMany(z => GetAnimTargetSourcesFromSubAttributes(z.y.TransIAnmAttr, new[] { AnimTargetType.TranslateS, AnimTargetType.TranslateT }, (byte)z.i)
                                                                            .Concat(GetAnimTargetSourcesFromSubAttributes(z.y.ScaleIAnmAttr, new[] { AnimTargetType.ScaleS, AnimTargetType.ScaleT }, (byte)z.i))
                                                                            .Concat(new[] { new AnimTargetSource(z.y.RotateIAnmAttr, AnimTargetType.Rotate, (byte)z.i) }))
                ).ToArray();
        }


        /// <summary>
        /// ペインの持つテクスチャSRTアニメーションをシリアライズ可能なクラスに変換します。
        /// </summary>
        public static FileFmt.AnimContent[] SaveIndirectTextureSRTAnim(AppData.IPane srcPane, string tag)
        {
            return ConvertToAnimContents<IRevHWMaterial, AnimIndTexSRTTarget>(
                tag,
                ParamaterKind.Animation_TextureSRT,
                srcPane,
                x => (x.IPaneExParamater as IRevHWMaterialHolder)?.IRevHWMaterial?.Select(y => new AnimContentSource<IRevHWMaterial>(y.MaterialName, y)) ?? Enumerable.Empty<AnimContentSource<IRevHWMaterial>>(),
                x => x.IIndirectMaterialTexMtxSet.Select((y, i) => new { y, i }).Where(z => z.y != null).SelectMany(z => GetAnimTargetSourcesFromSubAttributes(z.y.TransIAnmAttr, new[] { AnimTargetType.TranslateS, AnimTargetType.TranslateT }, (byte)z.i)
                                                                                            .Concat(GetAnimTargetSourcesFromSubAttributes(z.y.ScaleIAnmAttr, new[] { AnimTargetType.ScaleS, AnimTargetType.ScaleT }, (byte)z.i))
                                                                                            .Concat(new[] { new AnimTargetSource(z.y.RotateIAnmAttr, AnimTargetType.Rotate, (byte)z.i) }))
                ).ToArray();
        }

        #endregion ------------------ Rlts


        #region ---------- Rlts(読み込み)


        /// <summary>
        /// アニメーションアトリビュート
        /// </summary>
        static AnmAttribute
            GetTargetSubAnmAttr_( AppData.TexMtx dstMtx, AnimTargetType type )
        {
            switch( type )
            {
                case AnimTargetType.TranslateS: return dstMtx.TransAnmAttr.FindAttributeByIdx( 0 );
                case AnimTargetType.TranslateT: return dstMtx.TransAnmAttr.FindAttributeByIdx( 1 );
                case AnimTargetType.ScaleS: return dstMtx.ScaleAnmAttr.FindAttributeByIdx( 0 );
                case AnimTargetType.ScaleT: return dstMtx.ScaleAnmAttr.FindAttributeByIdx( 1 );
                case AnimTargetType.Rotate: return dstMtx.RotateAnmAttr;
                default: Debug.Assert( false ); return null;
            }
        }

        /// <summary>
        /// キーフレームの読み込みを行います
        /// </summary>
        static void SetKeyFrame_( AppData.TexMtx dstMtx, AnimTarget animTarget, string tag)
        {
            AnmAttribute dstSubAttr = GetTargetSubAnmAttr_( dstMtx, animTarget.target );
            RlanHelper.SetHermiteKeyFrame_( dstSubAttr, animTarget, tag);
        }

        /// <summary>
        /// テクスチャSRT
        /// </summary>
        static public void LoadRltsToMaterial( object dstObject, FileFmt.AnimContent srcAnimContent, string tag)
        {
            // アニメーション対象を探します。
            // 失敗した時点で中断します。
            Pane dstPane = dstObject as Pane;
            if( dstPane == null )
            {
                return;
            }

            Material dstMaterial = dstPane.FindMaterialByName( srcAnimContent.name );
            if( dstMaterial == null )
            {
                return;
            }

            foreach( FileFmt.AnimTarget srcAnimTarget in srcAnimContent.Items )
            {
                AppData.TexMtx dstMtx = dstMaterial.GetTextureMtx( srcAnimTarget.id );
                if( dstMtx == null )
                {
                    continue;
                }

                SetKeyFrame_( dstMtx, srcAnimTarget, tag );
            }
        }

        /// <summary>
        /// インダイレクトテクスチャSRT
        /// </summary>
        static public void LoadRltsToIndirectMaterial( object dstObject, FileFmt.AnimContent srcAnimContent, string tag)
        {
            // アニメーション対象を探します。
            // 失敗した時点で中断します。
            Pane dstPane = dstObject as Pane;
            if( dstPane == null )
            {
                return;
            }

            // マテリアルを検索します。
            RevHWMaterial dstMaterial = Array.Find(
                PaneHelper.GetRevHWMatFromPane( dstPane ),
                delegate( IRevHWMaterial material )
                {
                    return material.MaterialName == srcAnimContent.name;
                } ) as RevHWMaterial;
            if( dstMaterial == null )
            {
                return;
            }

            foreach (FileFmt.AnimTarget srcAnimTarget in srcAnimContent.Items)
            {
                var dstMtx = dstMaterial.IIndirectMaterialTexMtxSet[srcAnimTarget.id] as AppData.TexMtx;
                if (dstMtx == null)
                {
                    continue;
                }

                SetKeyFrame_(dstMtx, srcAnimTarget, tag);
            }
        }




        #endregion ---------- Rlts
    }
}
