﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Structures.Core.Command;
    using LECore.Structures.LECoreInterface;

    using TextureMgrCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.ITextureMgr>;
    //
    /// <summary>
    /// TextureMgrManipulator の概要の説明です。
    /// </summary>
    public class TextureMgrManipulator
    {
        TextureMgr       _target = null;
        TextureMgrCommandCreater _textureMgrCommandCreater;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextureMgrManipulator()
        {
            _textureMgrCommandCreater = new LECore.Structures.Core.Command.MementoCommandFactory<ITextureMgr>(
                delegate( ITextureMgr src ) { return new TextureMgrMemento( src ); } );
        }

        public void BindTarget( ITextureMgr textureMgr )
        {
           _target = textureMgr as TextureMgr;
            Debug.Assert( _target != null );
        }

        public void RemoveTexture( ITextureImage textureImage )
        {
            _textureMgrCommandCreater.Modify( _target, delegate()
            {
                _target.BeginUpdate();
                _target.RemoveTexture( textureImage );
                _target.EndUpdate();
            } );
        }

        public void RemoveTextureSet( ITextureImage[] textureImageSet )
        {
            _textureMgrCommandCreater.Modify( _target, delegate()
            {
                _target.BeginUpdate();
                foreach( ITextureImage textureImage in textureImageSet )
                {
                    _target.RemoveTexture( textureImage );
                }
                _target.EndUpdate();
            } );
        }

        public void BeginUpdate()
        {
            _target.BeginUpdate();
        }

        public void EndUpdate()
        {
            _target.EndUpdate();
        }

        // 登録
        public ITextureImage[] RegisterITextureImageSetFromFile( string[] filePathSet)
        {
            ITextureImage[] result = null;
            _textureMgrCommandCreater.Modify( _target, delegate()
            {
                result = _target.RegisterITextureImageSetFromFile( filePathSet );

                // 登録できたときだけコマンドスタックにつむ
                return result.Length > 0;
            } );
            return result;
        }

        //
        public ITextureImage CreateITextureImageFromFileWithoutRegistering( string filePath )
        {
            ITextureImage result = null;
            _textureMgrCommandCreater.Modify( _target, delegate()
            {
                result = _target.CreateITextureImageFromFileWithoutRegistering( filePath );
            } );
            return result;
        }

        // キャプチャテクスチャを登録
        public ITextureImage RegisterCaptureTexture(IPane pane, string textureName, bool isHidingFromList, CaptureTextureUsage usage)
        {
            ITextureImage result = null;
            _textureMgrCommandCreater.Modify(_target, delegate ()
            {
                try
                {
                    _target.BeginUpdate();
                    result = _target.RegisterCaptureTexture(pane, textureName, isHidingFromList, usage);
                }
                finally
                {
                    _target.EndUpdate();
                }
            });
            return result;
        }

        // キャプチャテクスチャを登録
        public ITextureImage RegisterCaptureTexture( IPane pane, string textureName, bool isHidingFromList )
        {
            ITextureImage result = null;
            _textureMgrCommandCreater.Modify( _target, delegate()
            {
                try
                {
                    _target.BeginUpdate();
                    result = _target.RegisterCaptureTexture(pane, textureName, isHidingFromList);
                }
                finally
                {
                    _target.EndUpdate();
                }
            } );
            return result;
        }

        public bool UpdateTextureImage( ITextureImage iTexImg )
        {
            bool bResult = false;
            _textureMgrCommandCreater.Modify( _target, delegate()
            {
                bResult = _target.UpdateTextureImage(iTexImg);
            });
            return bResult;
        }

        public bool UpdateTextureImage( ITextureImage iTexImg, string newFilePath )
        {
            bool bResult = false;
            _textureMgrCommandCreater.Modify( _target, delegate()
            {
                bResult = _target.UpdateTextureImage(iTexImg, newFilePath);
            });
            return bResult;
        }

        public void SetTexturePixelFmt(ITextureImage img, TexImagePixelFmt fmt)
        {
            if( img.PixelFmt != fmt )
            {
                _textureMgrCommandCreater.Modify( _target, delegate()
                {
                    img.PixelFmt = fmt;
                } );
            }
        }

    }
}
