﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Drawing;
using System.Diagnostics;
using System.IO;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.Core.Command;

    /// <summary>
    /// SceneManipulator の概要の説明です。
    /// </summary>
    public class SceneManipulator
    {

        Scene _target = null;

        #region プロパティ

        /// <summary>
        /// 操作対象のサブシーンを取得します。
        /// </summary>
        public IScene IScene
        {
            get{ return _target;}
        }

        /// <summary>
        /// カレントサブシーンを設定します。
        /// </summary>
        public ISubScene CurrentISubScene
        {
            set
            {
                if (!object.ReferenceEquals(_target.CurrentISubScene, value))
                {
                    if (value != null)
                    {
                        (value as SubScene).FirstTimeInitialize();
                    }

                    _target.CurrentISubScene = value;
                }
            }
        }

        #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="target"></param>
        public SceneManipulator()
        {
        }

        /// <summary>
        /// 操作対象を関連付けます
        /// </summary>
        /// <param name="target"></param>
        public void BindTarget( IScene target )
        {
            _target = target as Scene;
            Debug.Assert( _target != null );
        }

        /// <summary>
        /// サブシーンを生成します。
        /// </summary>
        public ISubScene CreateSubScne()
        {
            return _target.CreateSubScene();
        }

        /// <summary>
        /// シーンから、サブシーンを削除します。
        /// </summary>
        /// <param name="subScene"></param>
        public void RemoveSubScene( ISubScene subScene )
        {
            Debug.Assert( subScene != null );
            _target.RemoveSubScene( subScene );
        }

        /// <summary>
        /// クリップボードをリセットします。
        /// </summary>
        public void ResetClipBoard()
        {
           LEClipboard.Instance.Reset();
        }

        /// <summary>
        /// 部品レイアウトを更新します。
        /// </summary>
        public void RefreshPartsSubScenes(string partsRootPath)
        {
            // 可能ならフルパス表現に修正して設定します。
            if (Directory.Exists(partsRootPath))
            {
                partsRootPath = new DirectoryInfo(partsRootPath).FullName;
            }

            _target.PartsRootPath = partsRootPath;
            _target.RefreshPartsSubScenes();
        }
    }
}
