﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using System.Drawing;
    using LECore.Structures;
    using LECore.Structures.Core.Command;
    using LECore.Structures.Nsrif.Attributes;
    using MaterialCommandFactory = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IMaterial>;
    using MaterialRVLCommandFactory = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IRevHWMaterial>;
    using Structures.Core;

    /// <summary>
    /// RevMaterialManipulator の概要の説明です。
    /// 通常のMaterialをターゲットとする、
    ///
    /// MaterialManipulator は現状では、ほとんど必要にならないため、
    /// 削除しました。今後、ハードウェアに依存する部分の切り出しを行う際に
    /// 再度、検討することとします。
    /// </summary>
    public class RevMaterialManipulator : BaseManipulator
    {
        // 操作対象
        RevHWMaterial          _target = null;
        MaterialRVLCommandFactory _MaterialRVLCommandFactory;
        MaterialCommandFactory	 _MaterialCommandFactory;

        // 操作対象
        public IRevHWMaterial         Target
        {
            get{ return _target;}
        }

        /// <summary>
        /// Tevステージ数を設定します。
        /// </summary>
        public int NumTevStages
        {
            set
            {
                TevStages tevStages = _target.TevData.TevStages;
                int oldValue = tevStages.NumStages;
                if( tevStages.NumStages != value )
                {
                    _MaterialRVLCommandFactory.Modify( _target, delegate()
                    {
                        tevStages.NumStages = value;
                    } );
                }
            }
        }

        /// <summary>
        /// Hオフセット
        /// </summary>
        public float HOffset
        {
            set { _MaterialCommandFactory.Modify(_target.IMaterial, delegate() { (_target.IMaterial as Material).HOffset = value; }); }
        }

        /// <summary>
        /// Sスケール
        /// </summary>
        public float SScale
        {
            set { _MaterialCommandFactory.Modify(_target.IMaterial, delegate() { (_target.IMaterial as Material).SScale = value; }); }
        }

        /// <summary>
        /// Vスケール
        /// </summary>
        public float BScale
        {
            set { _MaterialCommandFactory.Modify(_target.IMaterial, delegate() { (_target.IMaterial as Material).BScale = value; }); }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RevMaterialManipulator()
        {
            _MaterialRVLCommandFactory = new MementoCommandFactory<IRevHWMaterial>(
              delegate( IRevHWMaterial src ) { return new MaterialRVLMemento( src ); } );

            // テクスチャ追加、削除操作UndoRedoには、IMaterialのスナップショットを利用する
            // 必要があります。(IRevHWMaterialはテクスチャリストを記憶しません。)
            _MaterialCommandFactory = new MementoCommandFactory<IMaterial>(
              delegate( IMaterial src ) { return new MaterialMemento( src ); } );
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="targetTextBox"></param>
        public void BindTarget( IRevHWMaterial targetMaterial )
        {
            _target = targetMaterial as RevHWMaterial;
            Debug.Assert( _target != null );
        }

        public void RegisterMatTexture(string texName, int index)
        {
            RegisterMatTexture(texName, index, AttrTextureResourceType.LocalFile);
        }

        /// <summary>
        /// マテリアルにテクスチャを登録します。
        /// </summary>
        public void RegisterMatTexture( string texName, int index, AttrTextureResourceType resType )
        {
            // _MaterialRVLCommandFactoryではなく、
            // _MaterialCommandFactoryを利用して、
            // コマンドを生成します。内部で作成される、
            // オブジェクトのスナップショットの型が異なり、それによって
            // 保存されるデータの種類が変わってきます。
            _MaterialCommandFactory.Modify( _target.IMaterial, delegate()
            {
                if (resType == AttrTextureResourceType.LocalCaptured)
                {
                    IPane ownerPane = _target.IMaterial.OwnerPane;
                    if (texName == PaneHelper.FramebufferCaptureDummyTextureName)
                    {
                        TextureMgr mgr = ownerPane.OwnerSubScene.ITextureMgr as TextureMgr;
                        ITextureImage tex = PaneHelper.GetCaptureTextureFromPane(ownerPane);
                        if (tex == null)
                        {
                            TextureMgrManipulator mnp = new TextureMgrManipulator();

                            mnp.BindTarget(mgr);
                            CaptureTexture newTex = mnp.RegisterCaptureTexture(ownerPane, ownerPane.PaneName, false) as CaptureTexture;
                        }

                        _target.RegisterMatTexture(PaneHelper.GetFullyQualifiedCaptureTextureName(ownerPane), index, resType);
                    }
                    else
                    {
                        string prefix = PaneHelper.GetCaptureTexturePrefix(ownerPane);
                        _target.RegisterMatTexture(PaneHelper.MakeCaptureTextureName(prefix, texName), index, resType);
                    }
                }
                else
                {
                    _target.RegisterMatTexture(texName, index, resType);
                }
            } );
        }

        /// <summary>
        /// マテリアルが参照しているテクスチャ名を置き換えます。
        /// </summary>
        public void RenameMatReferencedTexture(string texName, int index, AttrTextureResourceType resType)
        {
            // _MaterialRVLCommandFactoryではなく、
            // _MaterialCommandFactoryを利用して、
            // コマンドを生成します。内部で作成される、
            // オブジェクトのスナップショットの型が異なり、それによって
            // 保存されるデータの種類が変わってきます。
            _MaterialCommandFactory.Modify(_target.IMaterial, delegate ()
            {
                _target.RegisterMatTexture(texName, index, resType);
            });
        }

        /// <summary>
        /// 指定番号のマテリアルテクスチャを消去します。
        ///
        /// 注意：
        /// とりあえず、
        /// テクスチャ番号の付け替え、
        /// 参照しているTEVパラメータの変更等の処理は行わない事とします。
        /// </summary>
        public void RemoveMatTexture( int index )
        {
            if( index < _target.IMaterialTexMapSet.Length )
            {
                _MaterialCommandFactory.Modify( _target.IMaterial, delegate()
                {
                    _target.RemoveMatTexture( index );
                } );
            }
        }

        /// <summary>
        /// カラー割り当てを更新します。
        /// </summary>
        public void ModifyColorAssign
        (
            AttrColorAssignTarget     diffuseColorAssignTarget,
            AttrColorAssignTarget     alphaColorAssignTarget
        )
        {
            ColorAssignSet dstColorAssignSet = _target.IColorAssignSet as ColorAssignSet;

            if( dstColorAssignSet.DiffuseAssignTarget != diffuseColorAssignTarget ||
                dstColorAssignSet.AlphaAssignTarget != alphaColorAssignTarget )
            {
                _MaterialRVLCommandFactory.Modify( _target, delegate()
                {
                    dstColorAssignSet.DiffuseAssignTarget = diffuseColorAssignTarget;
                    dstColorAssignSet.AlphaAssignTarget = alphaColorAssignTarget;
                } );
            }
        }

#if false
        /// <summary>
        /// チャンネル設定を更新します。
        /// </summary>
        public void ModifyColorChannel
        (
            AttrColorSource     diffuseCtrlColorSrc,
            AttrColorSource     alphaCtrlColorSrc,
            RGBAColor           matColReg0
        )
        {
#if false
             ChannelData dstChannelData = _target.IChannelData as ChannelData;

             if( dstChannelData.DiffuseCtrlColorSrc != diffuseCtrlColorSrc ||
                 dstChannelData.AlphaCtrlColorSrc != alphaCtrlColorSrc ||
                 dstChannelData.MatColReg0 != matColReg0 )
             {
                 _MaterialRVLCommandFactory.Modify( _target, delegate()
                 {
                     dstChannelData.DiffuseCtrlColorSrc = diffuseCtrlColorSrc;
                     dstChannelData.AlphaCtrlColorSrc = alphaCtrlColorSrc;
                     dstChannelData.MatColReg0 = matColReg0;
                 } );
             }
#endif
        }
#endif

        /// <summary>
        /// PE設定を更新します。
        /// </summary>
        public void ModifyPEData(
            PEACompare aCompare,
            PEBlend blend,
            PEBlend blendAlpha,
            AttrTransparencyMode drawMode,
            bool useDefaultSettings,
            bool useDefaultAlphaTestSettings)
        {
            PEData dstPEData = _target.IPEData as PEData;

            if( !dstPEData.ACompare.Equals( aCompare ) ||
                !dstPEData.Blend.Equals( blend ) ||
                !dstPEData.BlendAlpha.Equals(blendAlpha) ||
                /*!dstPEData.ZCompare.Equals( zCompare ) ||*/
                dstPEData.DrawMode != drawMode ||
                dstPEData.UseDefaultBlendSettings != useDefaultSettings ||
                dstPEData.UseDefaultAlphaTestSettings != useDefaultAlphaTestSettings)
            {
                ManipulatorUtility.ModifyPropWithCreatingCmd(
                    () => dstPEData.ACompare.Comp,
                    x => dstPEData.ACompare.Comp = x,
                    aCompare.Comp
                    );
                ManipulatorUtility.ModifyAttrWithCreatingCmd((AnmAttribute)(dstPEData.ACompare.RefAnmAttr), aCompare.Ref);
                ManipulatorUtility.ModifyPropWithCreatingCmd(
                    () => dstPEData.ACompare.Op,
                    x => dstPEData.ACompare.Op = x,
                    aCompare.Op
                    );

                _MaterialRVLCommandFactory.Modify( _target, delegate()
                {
                    dstPEData.Blend.Set( blend );
                    dstPEData.BlendAlpha.Set(blendAlpha);
                    dstPEData.DrawMode = drawMode;
                    dstPEData.UseDefaultBlendSettings = useDefaultSettings;
                    dstPEData.UseDefaultAlphaTestSettings = useDefaultAlphaTestSettings;
                } );
            }
        }

        /// <summary>
        /// Tevカラーの変更
        /// </summary>
        public FloatColor WhiteColor
        {
            set
            {

                if (_target.IMaterial.WhiteColor != value)
                {
                    // 分解して処理する
                    FVec4 rgba = new FVec4(value.R, value.G, value.B, value.A);
                    bool byteConvert = value.ByteConvert;

                    ManipulatorUtility.ModifyAttrWithCreatingCmd((AnmAttribute)_target.IMaterial.WhiteColorIAnmAttr, rgba);
                    ManipulatorUtility.ModifyPropWithCreatingCmd(
                        () => ((Material)_target.IMaterial).WhiteColorByteConvert,
                        x => ((Material)_target.IMaterial).WhiteColorByteConvert = x,
                        byteConvert);
                }
            }
        }

        /// <summary>
        /// Tevカラーの変更
        /// </summary>
        public FloatColor BlackColor
        {
            set
            {
                if (_target.IMaterial.BlackColor != value)
                {
                    // 分解して処理する
                    FVec4 rgba = new FVec4(value.R, value.G, value.B, value.A);
                    bool byteConvert = value.ByteConvert;
                    ManipulatorUtility.ModifyAttrWithCreatingCmd((AnmAttribute)_target.IMaterial.BlackColorIAnmAttr, rgba);
                    ManipulatorUtility.ModifyPropWithCreatingCmd(
                        () => ((Material)_target.IMaterial).BlackColorByteConvert,
                        x => ((Material)_target.IMaterial).BlackColorByteConvert = x,
                        byteConvert);
                }
            }
        }

        /// <summary>
        /// Tevカラーの変更
        /// </summary>
        public bool IsThresholdingAlphaInterpolationEnabled
        {
            set
            {
                if (_target.IMaterial.IsThresholdingAlphaInterpolationEnabled != value)
                {
                    _MaterialCommandFactory.Modify(_target.IMaterial, delegate()
                    {
                        (_target.IMaterial as Material).IsThresholdingAlphaInterpolationEnabled = value;
                    });
                }
            }
        }

        public bool LowLevelCombinerSettingsEnabled
        {
            set
            {
                if (_target.LowLevelCombinerSettingsEnabled != value)
                {
                    _MaterialRVLCommandFactory.Modify(_target, delegate()
                    {
                        (_target as RevHWMaterial).LowLevelCombinerSettingsEnabled = value;
                    });
                }
            }
        }

        public bool CombinerUserShaderSettingsEnabled
        {
            set
            {
                if (_target.CombinerUserShaderSettingsEnabled != value)
                {
                    _MaterialRVLCommandFactory.Modify(_target, delegate ()
                    {
                        (_target as RevHWMaterial).CombinerUserShaderSettingsEnabled = value;
                    });
                }
            }
        }

        /// <summary>
        /// TevStageパラメータの更新
        /// </summary>
        public void ModifyTevStage(
           AttrRasOrder      rasOrder,
           int               texMapOrder,
           int               texCoordOrder,
           int               rasSwap,
           int               texMapSwap,
           TevDirectStage    colorStage,
           TevDirectStage    alphaStage,
           int               index )
        {
            Debug.Assert( index < _target.ITevData.NumStages );
            TevData dstTevData = _target.ITevData as TevData;
            TevStage dstStage = dstTevData.TevStages[index];

            if( dstStage.RasOrder       != rasOrder ||
                dstStage.TexMapOrder    != texMapOrder ||
                dstStage.TexCoordOrder  != texCoordOrder ||
                dstStage.RasSwap        != rasSwap ||
                dstStage.TexMapSwap     != texMapSwap ||
                !dstStage.ColorStage.IsSame( colorStage ) ||
                !dstStage.AlphaStage.IsSame( alphaStage )
              )

            {
                _MaterialRVLCommandFactory.Modify( _target, delegate()
                {
                    dstStage.RasOrder = rasOrder;
                    dstStage.TexMapOrder = texMapOrder;
                    dstStage.TexCoordOrder = texCoordOrder;
                    dstStage.RasSwap = rasSwap;
                    dstStage.TexMapSwap = texMapSwap;

                    // アニメーションカーブを持つ要素（IAnmAttribute）は
                    // 別途専用のコマンド機能を使って、BaseValue の更新処理を行わなければならない。
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(dstStage.ColorStage.IndirectTexMtx.RotateIAnmAttr as AnmAttribute, colorStage.IndirectRotate);
                    ManipulatorUtility.ModifyAttrWithCreatingCmd(dstStage.ColorStage.IndirectTexMtx.ScaleIAnmAttr as AnmAttribute, colorStage.IndirectScale);

                    dstStage.ColorStage.Set( colorStage );
                    dstStage.AlphaStage.Set( alphaStage );
                } );
            }
        }

        public void ModifyCombinerUserShader(string fileName)
        {
            CombinerUserShader dstCombinerUserData = _target.ICombinerUserShader as CombinerUserShader;

            if (!_target.CombinerUserShader.FileName.Equals(fileName))
            {
                _MaterialRVLCommandFactory.Modify(_target, delegate ()
                {
                    dstCombinerUserData.FileName = fileName;
                });
            }
        }

        /// <summary>
        /// TEVカラーを設定します。
        /// </summary>
        public void SetTevColors(FloatColor color, int index)
        {
            if (_target.TevData.GetTevColor(index) != color)
            {
                // 分解して処理する
                FVec4 rgba = new FVec4(color.R, color.G, color.B, color.A);
                bool byteConvert = color.ByteConvert;

                var attr = ((TevData)_target.TevData).GetTevColorAttr(index);

                ManipulatorUtility.ModifyAttrWithCreatingCmd(attr, rgba);
                ManipulatorUtility.ModifyPropWithCreatingCmd(
                    () => ((TevData)_target.TevData).GetTevColorByteConvert(index),
                    x => ((TevData)_target.TevData).SetTevColorByteConvert(x, index),
                    byteConvert);
            }
        }

        /// <summary>
        /// 変更通知だけを行うコマンドを設定します。
        /// </summary>
        public void PushEmptyCommand(SceneModifyEventArgs.Kind kind)
        {
            _target.NotifyModifyEvent(_target, (int)kind);
            Scene.CurrentSubScene.CommandFactory.SetActionCommand(
                () => {
                    _target.NotifyModifyEvent(_target, (int)kind);
                }
                );
        }
    }
}
