﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Structures.Nsrif.Attributes;

    /// <summary>
    /// GroupMgrManipulator の概要の説明です。
    /// </summary>
    public class GroupMgrManipulator : BaseManipulator
    {
        // 操作対象
        LEGroupMgr              _target = null;

        readonly EventHandler   _OnGroupMgrModifyHandler = null;
        bool                      _groupEditing = false;

        #region プロパティ

        /// <summary>
        /// 操作対象サブシーンを取得します。
        /// </summary>
        ISubScene _CurrentISubScene
        {
            get{ return Scene.CurrentSubScene;}
        }


        #endregion プロパティ
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GroupMgrManipulator()
        {
           _OnGroupMgrModifyHandler = OnGroupMgrModify_;
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="targetTextBox"></param>
        public void BindTarget( ILEGroupMgr targetGroupMgr )
        {
            _target = targetGroupMgr as LEGroupMgr;

            Debug.Assert( _target != null );
        }


        /// <summary>
        /// ペインセットから、グループを新規作成します。
        /// </summary>
        public ILEGroup MakeNewGroupFromPaneSet( IPane[] paneSet, string defaultGroupName)
        {
            string groupName =
                GroupHelper.GetUniqueGroupName(_target, defaultGroupName);

            _target.OnGroupMgrModify += _OnGroupMgrModifyHandler;

            // グループを作成します。
            ILEGroup group = _target.MakeNewGroup( groupName, paneSet );

            _target.OnGroupMgrModify -= _OnGroupMgrModifyHandler;

            return group;
        }

        /// <summary>
        /// グループ編集を開始します。
        /// </summary>
        public void BeginEditGroup()
        {
            _CurrentISubScene.BeginMassiveModify();
            _target.OnGroupMgrModify += _OnGroupMgrModifyHandler;
            _groupEditing = true;
        }

        /// <summary>
        /// グループ編集を終了します。
        /// </summary>
        public void EndEditGroup()
        {
            _groupEditing = false;
            _target.OnGroupMgrModify -= _OnGroupMgrModifyHandler;
            _CurrentISubScene.EndMassiveModify();
        }

        /// <summary>
        /// グループメンバを削除します。
        /// </summary>
        /// <param name="member"></param>
        public void RemoveGroup( ILEGroup member )
        {
            Debug.Assert( _groupEditing );
            _target.RemoveGroup( member );
        }


        /// <summary>
        /// グループメンバを削除します。
        /// </summary>
        /// <param name="member"></param>
        public void RemoveMemberFromAll( IPane member )
        {
            Debug.Assert( _groupEditing );
            _target.RemoveMemberFromAll( member );
            _target.RemoveEmptyGroup();
        }

        /// <summary>
        /// 空のグループを削除します。
        ///
        /// TODO:空のグループを有効とし、廃止を検討します。
        /// </summary>
        public void RemoveEmptyGroup()
        {
            Debug.Assert( _groupEditing );
            _target.RemoveEmptyGroup();
        }

        /// <summary>
        /// グループの順番を変更します。
        /// </summary>
        public int ChangeGroupOrder( ILEGroup group, int newIndex )
        {
            Debug.Assert( _groupEditing );
            return _target.ChangeGroupOrder( group, newIndex );
        }

       #region グループに対する操作
        /// <summary>
        /// メンバーを登録します。
        /// </summary>
        public void AddMember( ILEGroup target, IPane member )
        {
            Debug.Assert( _groupEditing );
            (target as LEGroup).AddMember( member );
        }

        /// <summary>
        /// メンバーを登録します。
        /// </summary>
        public void RemoveMember( ILEGroup target, IPane member )
        {
            Debug.Assert( _groupEditing );
            ( target as LEGroup ).RemoveMember( member );
        }

        /// <summary>
        /// メンバーの順番を変更します。
        /// </summary>
        public int ChangeChildOrder( ILEGroup target, IPane member, int newIndex )
        {
            Debug.Assert( _groupEditing );
            return ( target as LEGroup ).ChangeMemberOrder( member, newIndex );
        }
       #endregion グループに対する操作

        /// <summary>
        /// マネージャ変更イベントハンドラ
        /// Undoコマンドを登録します。
        /// </summary>
        void OnGroupMgrModify_(object sender, EventArgs e)
        {
            if( sender == _target )
            {
                // Undoコマンドの登録
                if( e is LEGroupMgrModifyEventArgs )
                {
                    _CommandFactory.MakeSceneGroupModCmd( e as LEGroupMgrModifyEventArgs );
                }
            }
        }
    }
}


