﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW4R_FROM_TOOL
#include <windows.h>

#include "nw4r/misc.h"
#include "TagProcessorBase.h"
#include "TextWriterBase.h"


namespace NW4R
{
namespace Font
{
namespace UnManaged
{







/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TagProcessorBase::TagProcessorBase()

  Description:  コンストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
TagProcessorBase<CharType>::TagProcessorBase()
{
}

/*---------------------------------------------------------------------------*
  Name:         xxxxx

  Description:  デストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
TagProcessorBase<CharType>::~TagProcessorBase()
{
}




/* ------------------------------------------------------------------------
        タグ処理
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TagProcessorBase::Process( u16, PrintContext* )

  Description:  文字列中のタグを処理します。

  Arguments:    code:       タグの基点となる文字コード(0x00～0x1F)
                context:    描画処理のコンテキスト情報へのポインタ。

  Returns:      タグ処理の結果、文字列描画に及ぼすべき影響を返します。
 *---------------------------------------------------------------------------*/
template <typename CharType>
typename TagProcessorBase<CharType>::Operation
TagProcessorBase<CharType>::Process(
    u16             code,
    ContextType*    context
)
{
    NW4R_ASSERT(code < ' ');
    NW4R_POINTER_ASSERT(context);

    switch( code )
    {
    case '\n':
        {
            ProcessLinefeed(context);
        }
        return OPERATION_NEXT_LINE;

    case '\t':
        {
            ProcessTab(context);
        }
        return OPERATION_NO_CHAR_SPACE;

    default:
        {
            // nothing to do;
        }
        break;
    }

    return OPERATION_DEFAULT;
}

/*---------------------------------------------------------------------------*
  Name:         TagProcessorBase::CalcRect( Rect*, u16, PrintContext* )

  Description:  文字列中のタグを処理した場合に影響する矩形領域を
                計算します。

  Arguments:    pRect:      影響する矩形領域を格納するためのバッファへのポインタ。
                code:       タグの基点となる文字コード(0x00～0x1F)
                context:    描画処理のコンテキスト情報へのポインタ。

  Returns:      タグ処理の結果、文字列描画に及ぼすべき影響を返します。
 *---------------------------------------------------------------------------*/
template <typename CharType>
typename TagProcessorBase<CharType>::Operation
TagProcessorBase<CharType>::CalcRect(
    Rect*           pRect,
    u16             code,
    ContextType*    context
)
{
    NW4R_POINTER_ASSERT(pRect);
    NW4R_ASSERT(code < ' ');
    NW4R_POINTER_ASSERT(context);

    switch( code )
    {
    case '\n':
        {
            TextWriterBase<CharType>& writer = *context->writer;

            pRect->right    = writer.GetCursorX();
            pRect->top      = writer.GetCursorY();

            ProcessLinefeed(context);

            // ProcessLinefeed はカーソルを移動するだけで次の行の高さは含まれていないので
            // 次の行の高さとして FontHeight を足す。
            pRect->left     = writer.GetCursorX();
            pRect->bottom   = writer.GetCursorY() + context->writer->GetFontHeight();

            pRect->Normalize();
        }
        return OPERATION_NEXT_LINE;

    case '\t':
        {
            TextWriterBase<CharType>& writer = *context->writer;

            pRect->left     = writer.GetCursorX();

            ProcessTab(context);

            pRect->right    = writer.GetCursorX();
            pRect->top      = writer.GetCursorY();
            pRect->bottom   = pRect->top + writer.GetFontHeight();

            pRect->Normalize();
        }
        return OPERATION_NO_CHAR_SPACE;

    default:
        {
            // nothing to do;
        }
        break;
    }

    return OPERATION_DEFAULT;
}




/* ------------------------------------------------------------------------
        private タグ処理
   ------------------------------------------------------------------------ */
template <typename CharType>
void
TagProcessorBase<CharType>::ProcessLinefeed(
    ContextType*    context
)
{
    NW4R_POINTER_ASSERT(context);

    TextWriterBase<CharType>& writer = *context->writer;

    const float x   = context->xOrigin;
    const float y   = writer.GetCursorY() + writer.GetLineHeight();

    writer.SetCursor(x, y);
}

template <typename CharType>
void
TagProcessorBase<CharType>::ProcessTab(
    ContextType* context
)
{
    NW4R_POINTER_ASSERT(context);

    TextWriterBase<CharType>& writer = *context->writer;
    const int tabWidth = writer.GetTabWidth();

    if( tabWidth > 0 )
    {
        const float aCharWidth  = writer.IsWidthFixed() ? writer.GetFixedWidth(): writer.GetFontWidth();
        const float dx          = writer.GetCursorX() - context->xOrigin;
        const float tabPixel    = tabWidth * aCharWidth;
        const int   numTab      = static_cast<int>(dx / tabPixel) + 1;
        const float x           = context->xOrigin + tabPixel * numTab;

        writer.SetCursorX(x);
    }
}




/* ------------------------------------------------------------------------
        明示的実体化
   ------------------------------------------------------------------------ */

template class TagProcessorBase<char>;
template class TagProcessorBase<wchar_t>;


} /* UnManaged */
} /* namespace ut */
} /* namespace nw4r */
