﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW4R_FROM_TOOL
#include <windows.h>

#include "nw4r/misc.h"
#include "nw4r/ut/RomFont.h"
#include "nw4r/ut/inlines.h"

namespace nw4r {
    namespace ut {



namespace {

    /*---------------------------------------------------------------------------*
      Name:         IsCP1252Char( CharCode )

      Description:  対象の文字コードが非日本向け内蔵フォントに含まれている
                    文字かどうか判定します。

      Arguments:    c:  判定対象の文字コード。

      Returns:      条件に合致すれば true。
     *---------------------------------------------------------------------------*/
    inline bool
    IsCP1252Char(CharCode c)
    {
        return ( (0x20 <= c) && (c <= 0xFF) );
    }

    /*---------------------------------------------------------------------------*
      Name:         IsSJISHalfWidthChar( CharCode )

      Description:  対象の文字コードが日本向け内蔵フォントに含まれている
                    半角文字かどうか判定します。

      Arguments:    c:  判定対象の文字コード。

      Returns:      条件に合致すれば true。
     *---------------------------------------------------------------------------*/
    inline bool
    IsSJISHalfWidthChar(CharCode c)
    {
        if( 0xFF < c )
        {
            return false;
        }

        return ( ((0x20 <= c) && (c <= 0x7E))
                || ((0xA1 <= c) && (c <= 0xCF)) );
    }

    /*---------------------------------------------------------------------------*
      Name:         IsSJISFullWidthChar( CharCode )

      Description:  対象の文字コードが日本向け内蔵フォントに含まれている
                    全角文字かどうか判定します。

      Arguments:    c:  判定対象の文字コード。

      Returns:      条件に合致すれば true。
     *---------------------------------------------------------------------------*/
    inline bool
    IsSJISFullWidthChar(CharCode c)
    {
        u8 lead     = BitExtract<u8>(c, 8, 8);
        u8 trail    = BitExtract<u8>(c, 0, 8);

        return ((0x81 <= lead) && (lead <= 0x98))
                && ((0x40 <= trail) && (trail <= 0xFC));
    }

} /* namespace */


/* ------------------------------------------------------------------------
        クラス変数定義
   ------------------------------------------------------------------------ */

u16 RomFont::mFontEncode = 0xFFFF;



/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         RomFont::RomFont()

  Description:  コンストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
RomFont::RomFont()
: mFontHeader(NULL),
  mAlternateChar('?')
{
    mDefaultWidths.left         = 0;
    mDefaultWidths.glyphWidth   = 0;
    mDefaultWidths.charWidth    = 0;
}


/*---------------------------------------------------------------------------*
  Name:         RomFont::~RomFont()

  Description:  デストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
RomFont::~RomFont()
{
}




/* ------------------------------------------------------------------------
        構築
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         RomFont::Load( void* )

  Description:  内蔵フォントをロードしてこのフォントを使用できるようにします。

  Arguments:    buffer: GetRequireBufferSize() 以上のサイズのバッファへの
                        ポインタ。
                        Unload() するまでRomFontが使用します。

  Returns:      ロードに成功したら true を返します。
 *---------------------------------------------------------------------------*/
bool
RomFont::Load(void* buffer)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( buffer );
    NW4R_ALIGN32_ASSERT( buffer );
    OSFontHeader* fontHeader = reinterpret_cast<OSFontHeader*>(buffer);
    BOOL bSuccess;

    //---- 既にロード済みである
    if( mFontHeader != NULL )
    {
        return false;
    }

    bSuccess = ::OSInitFont(fontHeader);
    if( bSuccess )
    {
        mFontEncode                 = ::OSGetFontEncode();
        mFontHeader                 = fontHeader;
        mDefaultWidths.left         = 0;
        mDefaultWidths.glyphWidth   = static_cast<u8>(GetCellWidth());
        mDefaultWidths.charWidth    = static_cast<s8>(GetMaxCharWidth());

        InitReaderFunc(GetEncoding());
    }

    return bSuccess != FALSE;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::Unload()

  Description:  ロードした内蔵フォントデータを解放します。

  Arguments:    なし。

  Returns:      Load(void*) の引数として渡したバッファへのポインタを返す。
                Load されていないか、Load 後に既に Unload した場合は
                NULL を返す。
 *---------------------------------------------------------------------------*/
void*
RomFont::Unload()
{
    NW4R_POINTER_ASSERT( this );
    NW4R_WARNING(
        mFontHeader != NULL,
        "RomFont::Unload(): Rom font is not loaded.\n"
    );
    void* buffer = mFontHeader;
    mFontHeader = NULL;
    return buffer;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetRequireBufferSize()

  Description:  Load(void*) の第一引数で渡すべきバッファのサイズを取得します。

  Arguments:    なし。

  Returns:      バッファのサイズ。
 *---------------------------------------------------------------------------*/
u32
RomFont::GetRequireBufferSize()
{
    const u16 fontEncode = ::OSGetFontEncode();

    switch( fontEncode )
    {
    case OS_FONT_ENCODE_ANSI:   return OS_FONT_SIZE_ANSI;
    case OS_FONT_ENCODE_SJIS:   return OS_FONT_SIZE_SJIS;
    default:
        NW4R_FATAL_ERROR(
            "RomFont::GetRequireBufferSize(): Unknown OS_FONT_ENCODE_*(=%d)\n",
            fontEncode
        );
        return 0;
    }
}





/* ------------------------------------------------------------------------
        フォント全体情報アクセサ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetBaselinePos()

  Description:  フォントのベースライン位置を取得します。

  Arguments:    なし。

  Returns:      フォントのベースライン位置(セル上端からのピクセル数)
 *---------------------------------------------------------------------------*/
int
RomFont::GetBaselinePos() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontHeader );
    return mFontHeader->ascent;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetCellHeight()

  Description:  フォントのセルの高さを取得します。

  Arguments:    なし。

  Returns:      フォントのセルの高さ(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
RomFont::GetCellHeight() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontHeader );
    return mFontHeader->cellHeight;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetCellWidth()

  Description:  フォントのセルの幅を取得します。

  Arguments:    なし。

  Returns:      フォントのセルの幅を取得します(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
RomFont::GetCellWidth() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontHeader );
    return mFontHeader->cellWidth;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetHeight()

  Description:  フォントの高さを取得します。

  Arguments:    なし。

  Returns:      フォントの高さ(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
RomFont::GetHeight() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontHeader );
    return mFontHeader->cellHeight;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetMaxCharWidth()

  Description:  フォント中に含まれる文字の最大文字幅を取得します

  Arguments:    なし。

  Returns:      最大文字幅(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
RomFont::GetMaxCharWidth() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontHeader );
    return mFontHeader->width;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetType()

  Description:  フォントのタイプを取得します。

  Arguments:    なし。

  Returns:      Font::TYPE_ROM を返します。
 *---------------------------------------------------------------------------*/
Font::Type
RomFont::GetType() const
{
    return TYPE_ROM;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetTextureFormat()

  Description:  フォントのテクスチャフォーマットを帰します。

  Arguments:    なし。

  Returns:      フォントのテクスチャフォーマット。
 *---------------------------------------------------------------------------*/
GXTexFmt
RomFont::GetTextureFormat() const
{
    return GX_TF_I4;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetLineFeed()

  Description:  1行の高さを返します。

  Arguments:    なし。

  Returns:      1行の高さ(ピクセル単位)。
 *---------------------------------------------------------------------------*/
int
RomFont::GetLineFeed() const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontHeader );
    return mFontHeader->leading;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetDefaultCharWidths()

  Description:  デフォルトの文字幅を返します。

  Arguments:    なし。

  Returns:      文字幅情報。
 *---------------------------------------------------------------------------*/
const CharWidths
RomFont::GetDefaultCharWidths() const
{
    NW4R_POINTER_ASSERT( this );
    return mDefaultWidths;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::SetDefaultCharWidths( const CharWidths& )

  Description:  デフォルトの文字幅を設定します。

  Arguments:    widths: 新しいデフォルトの文字幅。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
RomFont::SetDefaultCharWidths(
     const CharWidths& widths
 )
{
    NW4R_POINTER_ASSERT( this );
    NW4R_REFERENCE_ASSERT( widths );
    mDefaultWidths = widths;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::SetAlternateChar( CharCode )

  Description:  代替文字を設定します。

  Arguments:    c:  新しい代替文字の文字コード。

  Returns:      代替文字の変更に成功した場合に true。
                設定しようとしている代替文字がフォントに含まれていない場合に
                失敗します。
 *---------------------------------------------------------------------------*/
bool
RomFont::SetAlternateChar(
     CharCode c )
{
    NW4R_POINTER_ASSERT( this );
    const CharCode oldValue = mAlternateChar;
    CharCode check;

    mAlternateChar = INVALID_CHARACTER_CODE;
    check = HandleUndefinedChar(c);

    if( check != INVALID_CHARACTER_CODE )
    {
        mAlternateChar = c;
        return true;
    }
    else
    {
        mAlternateChar = oldValue;
        return false;
    }
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::SetLineFeed( int )

  Description:  1行の高さを設定します。

  Arguments:    linefeed:   新しい1行の高さ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
RomFont::SetLineFeed( int linefeed )
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontHeader );
    NW4R_U16_RANGE_ASSERT(linefeed);
    mFontHeader->leading = static_cast<u16>(linefeed);
}



/* ------------------------------------------------------------------------
        文字単体情報アクセサ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetCharWidth( CharCode )

  Description:  文字の文字幅を取得します。

  Arguments:    c:  文字幅を取得する文字の文字コード。

  Returns:      文字幅(ピクセル単位)
 *---------------------------------------------------------------------------*/
int
RomFont::GetCharWidth( CharCode c ) const
{
    s32 width;
    char buf[CHAR_PTR_BUFFER_SIZE];

    MakeCharPtr(buf, c);
    (void)::OSGetFontWidth(buf, &width);

    return width;
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetCharWidths( CharCode )

  Description:  文字の文字幅情報を取得します。

  Arguments:    c:  文字幅情報を取得する文字の文字コード

  Returns:      文字幅情報。
 *---------------------------------------------------------------------------*/
const CharWidths
RomFont::GetCharWidths( CharCode c ) const
{
    const int width = GetCharWidth(c);
    CharWidths widths;

    //---- すべての文字で左右スペースは 0 とする
    widths.left         = 0;
    widths.glyphWidth   = static_cast<u8>(width);
    widths.charWidth    = static_cast<s8>(width);

    return widths;
}

char* OSGetFontTexture( char const* pBuff, void** pTex, s32* pX, s32* pY, s32* pW )
{
    return NULL;
}
/*---------------------------------------------------------------------------*
  Name:         RomFont::GetGlyph( Glyph*, CharCode )

  Description:  文字のグリフデータを取得します。

  Arguments:    glyph:  グリフデータを受け取るバッファへのポインタ。
                c:      グリフデータを取得する文字の文字コード。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
RomFont::GetGlyph(
     Glyph*     glyph,
     CharCode   c
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( mFontHeader );
    void* pTexture;
    s32 x, y, width;
    char buf[CHAR_PTR_BUFFER_SIZE];

    MakeCharPtr(buf, c);
    (void)::OSGetFontTexture(buf, &pTexture, &x, &y, &width);

    glyph->pTexture             = pTexture;
    glyph->widths.left          = 0;
    glyph->widths.glyphWidth    = static_cast<u8>(width);
    glyph->widths.charWidth     = static_cast<s8>(width);
    glyph->height               = static_cast<u8>(mFontHeader->cellHeight);
    glyph->texFormat            = GX_TF_I4;
    glyph->texWidth             = mFontHeader->sheetWidth;
    glyph->texHeight            = mFontHeader->sheetHeight;
    glyph->cellX                = static_cast<u16>(x);
    glyph->cellY                = static_cast<u16>(y);
}


/* ------------------------------------------------------------------------
        文字ストリーム
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         RomFont::GetEncoding()

  Description:  フォントが対応している文字列エンコーディング形式を取得します。

  Arguments:    なし。

  Returns:      フォントが対応している文字列エンコーディング形式。
 *---------------------------------------------------------------------------*/
FontEncoding
RomFont::GetEncoding() const
{
    switch( mFontEncode )
    {
    case OS_FONT_ENCODE_ANSI:   return FONT_ENCODING_CP1252;
    case OS_FONT_ENCODE_SJIS:   return FONT_ENCODING_SJIS;
    default:
        NW4R_FATAL_ERROR(
            "RomFont::GetEncoding(): Unknown OS_FONT_ENCODE_*(=%d)\n",
            mFontEncode
        );
        return FONT_ENCODING_CP1252;
    }
}



/* ------------------------------------------------------------------------
        private
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         RomFont::MakeCharPtr( char*, CharCode )

  Description:  文字コードからその文字1文字のみを含むNUL終端文字列を作成します。

  Arguments:    buffer: 作成した文字列を格納するバッファへのポインタ。
                        CHAR_PTR_BUFFER_SIZE 以上のサイズでなければならない。
                c:      文字列化する文字コード。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
RomFont::MakeCharPtr(char* buffer, CharCode c) const
{
    c = HandleUndefinedChar(c);

    if( BitExtract<u32>(c, 8, 8) == 0 )
    {
        buffer[0] = BitExtract<char>(c, 0, 8);
        buffer[1] = '\0';
    }
    else
    {
        buffer[0] = BitExtract<char>(c, 8, 8);
        buffer[1] = BitExtract<char>(c, 0, 8);
        buffer[2] = '\0';
    }
}

/*---------------------------------------------------------------------------*
  Name:         RomFont::HandleUndefinedChar( CharCode )

  Description:  内蔵フォントに含まれていない文字コードを代替文字の文字コードに
                置き換えます。

  Arguments:    c:  文字コード。

  Returns:      c が内蔵フォントに含まれていれば c を、そうでなければ
                代替文字の文字コードを返します。
 *---------------------------------------------------------------------------*/
CharCode
RomFont::HandleUndefinedChar(CharCode c) const
{
    NW4R_POINTER_ASSERT( this );
    bool bDefined;

    switch( mFontEncode )
    {
    case OS_FONT_ENCODE_ANSI:
        {
            bDefined = IsCP1252Char(c);
        }
        break;

    case OS_FONT_ENCODE_SJIS:
        {
            bDefined = ( IsSJISHalfWidthChar(c)
                            || IsSJISFullWidthChar(c) );
        }
        break;

    default:
        NW4R_INTERNAL_ERROR(
            "RomFont::GetRequireBufferSize(): Unknown OS_FONT_ENCODE_*(=%d)\n",
            mFontEncode
        );
    }

    return bDefined ? c: mAlternateChar;
}




    } /* namespace ut */
} /* namespace nw4r */
