﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "ArchiveFontBase.h"
#include "inlines.h"
#include "nw4r/misc.h"
#include "nw4r/math/arithmetic.h"
#include "ByteSwapUtil.h"
#include "FontEnvironment.h"

#include <cstring>
#include <algorithm>

namespace NW4R
{
namespace Font
{
namespace UnManaged
{
namespace detail {


const char ArchiveFontBase::LOAD_GLYPH_ALL[1] = "";


/* =======================================================================
        ArchiveFontBase public
   ======================================================================== */

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ArchiveFontBase()

  Description:  コンストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
ArchiveFontBase::ArchiveFontBase()
: mpGlyphIndexAdjustArray(NULL)
{
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::~ArchiveFontBase()

  Description:  デストラクタ

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
ArchiveFontBase::~ArchiveFontBase()
{
}



/* ------------------------------------------------------------------------
        Fontクラスオーバーライド
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ArchiveFont::GetCharWidths( CharCode )

  Description:  文字の文字幅情報を取得します。

  Arguments:    c:  文字幅情報を取得する文字の文字コード

  Returns:      文字幅情報。
 *---------------------------------------------------------------------------*/
const CharWidths
ArchiveFontBase::GetCharWidths( CharCode c ) const
{
    GlyphIndex index = GetGlyphIndex(c);

    // GetCharWidthsFromIndex に渡すグリフインデックスは補正前のインデックスだが
    // シートが存在しない場合は代替文字の文字幅を取得しなければならないため
    // AdjustIndex を使ってシートが存在するかテストする
    if( AdjustIndex(index) == GLYPH_INDEX_NOT_FOUND )
    {
        index = GetFINF()->alterCharIndex;
    }

    return GetCharWidthsFromIndex(index);
}





/* =======================================================================
        ArchiveFontBase protected
   ======================================================================== */

/* ------------------------------------------------------------------------
        構築/破棄
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::SetResourceBuffer(
                                            void*, FontInformation, u16*)

  Description:  割り当てられたバッファとそこに配置されている FINF ブロックへの
                ポインタ, グリフインデックス調整テーブルを設定します。

  Arguments:    pUserBuffer:    割り当てられたバッファへのポインタ。
                pFontInfo:      FINF ブロックへのポインタ。
                pAdjustTable:   グリフインデックス調整テーブルへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ArchiveFontBase::SetResourceBuffer(
    void*               pUserBuffer,
    FontInformation*    pFontInfo,
    u16*                pAdjustTable )
{
    NW4R_POINTER_ASSERT( pUserBuffer );
    NW4R_POINTER_ASSERT( pFontInfo );
    NW4R_POINTER_ASSERT( pAdjustTable );
    NW4R_ASSERT( mpGlyphIndexAdjustArray == NULL );

    ResFontBase::SetResourceBuffer(pUserBuffer, pFontInfo);
    mpGlyphIndexAdjustArray = pAdjustTable;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::RemoveResourceBuffer()

  Description:  SetResourceBuffer で設定されたパラメータをリセットし、
                割り当てられていたバッファへのポインタを返します。

  Arguments:    なし。

  Returns:      設定されていたバッファへのポインタ。
 *---------------------------------------------------------------------------*/
void*
ArchiveFontBase::RemoveResourceBuffer()
{
    mpGlyphIndexAdjustArray = NULL;
    return ResFontBase::RemoveResourceBuffer();
}



/* ------------------------------------------------------------------------
        グリフインデックス
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::AdjustIndex( u16 )

  Description:  CMAP ブロックから取得したグリフインデックスに対して
                ロードしていないシート分の補正を行います。
                CWDH ブロックでの文字幅情報取得には補正前のグリフインデックスを、
                シートからのグリフイメージ切り出しには補正後の
                グリフインデックスを使用します

  Arguments:    index:  補正するグリフインデックス

  Returns:      補正後のグリフインデックスを返します。
                GLYPH_INDEX_NOT_FOUND ではないグリフインデックスが
                補正によって GLYPH_INDEX_NOT_FOUND になることがあります。
 *---------------------------------------------------------------------------*/
u16
ArchiveFontBase::AdjustIndex(u16 index) const
{
    const FontTextureGlyph& tg  = *(GetFINF()->pGlyph);
    const int glyphsPerSheet    = tg.sheetRow * tg.sheetLine;
    const int sheetNo           = index / glyphsPerSheet;

    const u16 adjustor = mpGlyphIndexAdjustArray[sheetNo];

    return (adjustor == ADJUST_OFFSET_SHEET_NOT_LOADED) ?
                GLYPH_INDEX_NOT_FOUND: static_cast<u16>(index - adjustor);
}



/* ------------------------------------------------------------------------
        GLGRブロック走査
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::IncludeName( const char*, const char* )

  Description:  nameList に name が含まれているか判断します。

  Arguments:    nameList:   文字列のカンマ区切りリスト。
                name:       探索する文字列。

  Returns:      nameList に name が含まれているなら true を返します。
 *---------------------------------------------------------------------------*/
bool
ArchiveFontBase::IncludeName(const char* nameList, const char* name)
{
    const u32 nameLen = std::strlen(name);
    const char* found = nameList - 1;

    // return $nameList =~ m/(^|, *)$name *(,|$)/;
    for(;;)
    {
        found = std::strstr(found+1, name);

        if( found == NULL )
        {
            return false;
        }

        // 先頭か？
        if( found != nameList )
        {
            // 先頭でなくても直前にデリミタがあればOK
            const char* pos = found - 1;
            while( nameList < pos && *pos == ' ' )
            {
                pos--;
            }

            if( *pos != ',' )
            {
                continue;
            }
        }

        // 次のデリミタを探す
        const char* delimitor = std::strchr(found, ',');
        u32 foundLen;

        // 見つかった文字列の長さを求める
        if( delimitor != NULL )
        {
            foundLen = static_cast<u32>(delimitor - found);
        }
        else
        {
            foundLen = std::strlen(found);
        }

        // 長さが同じかその末尾に空白が加わっただけなら発見
        {
            const char* pos = found + nameLen;
            const char* end = found + foundLen;
            while( pos < end && *pos == ' ' )
            {
                pos++;
            }
            if( pos == end )
            {
                return true;
            }
        }
    }

    //return false;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::IsValidResource( const void*, u32 )

  Description:  与えられたバイナリデータが圧縮済みフォントリソースの
                GLGR ブロック部分までを含んでいるか判断します。

  Arguments:    brfnt:      判断するバイナリデータへのポインタ。
                dataSize:   brfnt で渡されたデータのサイズ。

  Returns:      brfnt が GLGR ブロックを完全に含んだ圧縮済みフォントリソースなら
                true を返します。
 *---------------------------------------------------------------------------*/
bool
ArchiveFontBase::IsValidResource(const void* brfnt, u32 dataSize)
{
    NW4R_MIN_ASSERT( dataSize, sizeof(BinaryFileHeader) + sizeof(FontGlyphGroupsBlock) );
    const BinaryFileHeader* fileHeader = reinterpret_cast<const BinaryFileHeader*>(brfnt);

    // バイナリファイルヘッダをチェック
    if( ! IsValidBinaryFile(fileHeader, BINFILE_SIG_FONTA, FONT_FILE_VERSION, 2) )
    {
        // サポートしていない形式
        NW4R_WARNING(FALSE, "Invalid font resource.");
        return false;
    }

    const FontGlyphGroupsBlock* ggBlock = reinterpret_cast<const FontGlyphGroupsBlock*>(fileHeader + 1);

    // 先頭ブロックのタイプをチェック
    if( ggBlock->header.kind != BINBLOCK_SIG_GLGR )
    {
        NW4R_WARNING(false, "invalid block header(%c%c%c%c) expect(%c%c%c%c)",
            (ggBlock->header.kind >> 24) & 0xFF,
            (ggBlock->header.kind >> 16) & 0xFF,
            (ggBlock->header.kind >>  8) & 0xFF,
            (ggBlock->header.kind >>  0) & 0xFF,
            (BINBLOCK_SIG_GLGR >> 24) & 0xFF,
            (BINBLOCK_SIG_GLGR >> 16) & 0xFF,
            (BINBLOCK_SIG_GLGR >>  8) & 0xFF,
            (BINBLOCK_SIG_GLGR >>  0) & 0xFF
        );
        return false;
    }

    // 先頭ブロックを全て読み込めているかチェック
    if( sizeof(BinaryFileHeader) + ggBlock->header.size > dataSize )
    {
        NW4R_WARNING(false, "First stream block must include all of GLGP block. "
                            "Please use laerger stream buffer.");
        return false;
    }

    return true;
}



/* ------------------------------------------------------------------------
        構築処理
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::RequestData(
                                ConstructContext*, CachedStreamReader, u32)

  Description:  入力ストリームのデータが足りない場合の処理を行います。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。
                size:

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::RequestData( ArchiveFontBase::ConstructContext*   pContext,
                              ArchiveFontBase::CachedStreamReader* pStream,
                              u32 size )
{
    pContext->streamOffset += pStream->GetOffset();
    bool bSuccess = pStream->RequestData(pContext, size);
    return bSuccess ? CONSTRUCT_MORE_DATA: CONSTRUCT_ERROR;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpDispatch(
                                    ConstructContext*, CachedStreamReader)

  Description:  バイナリファイル中のブロックごとに処理を振り分けます。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpDispatch(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    pStream )
{
    if( ! pContext->HasMoreBlock() )
    {
        return CONSTRUCT_FINISH;
    }

    const u32 requireSize = sizeof(BinaryBlockHeader);
    if( pStream->GetRemain() < requireSize )
    {
        return RequestData(pContext, pStream, requireSize);
    }

    pStream->CopyTo(&pContext->header, sizeof(pContext->header));

    char a[ 5 ];
    a[ 0 ] = (pContext->header.kind >> 24) & 0xFF;
    a[ 1 ] = (pContext->header.kind >> 16) & 0xFF;
    a[ 2 ] = (pContext->header.kind >>  8) & 0xFF;
    a[ 3 ] = (pContext->header.kind >>  0) & 0xFF;
    a[ 4 ] = 0;

    switch( pContext->header.kind )
    {
    case BINBLOCK_SIG_GLGR: pContext->op = ConstructContext::ANALYZE_GLGR;    break;
    case BINBLOCK_SIG_FINF: pContext->op = ConstructContext::ANALYZE_FINF;    break;
    case BINBLOCK_SIG_CMAP: pContext->op = ConstructContext::ANALYZE_CMAP;    break;
    case BINBLOCK_SIG_CWDH: pContext->op = ConstructContext::ANALYZE_CWDH;    break;
    case BINBLOCK_SIG_TGLP: pContext->op = ConstructContext::ANALYZE_TGLP;    break;
    default:
        NW4R_ASSERTMSG(false, "The font has unknown block(%c%c%c%c)",
            (pContext->header.kind >> 24) & 0xFF,
            (pContext->header.kind >> 16) & 0xFF,
            (pContext->header.kind >>  8) & 0xFF,
            (pContext->header.kind >>  0) & 0xFF );
        pContext->op = ConstructContext::FATAL_ERROR;
        return CONSTRUCT_ERROR;
    }

    // カレントブロック番号を次に進める
    pContext->NextBlock();

    // ブロックデータの先頭は 4 byte アライメントが必要
    pContext->Align(4);

    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpAnalyzeFileHeader(
                                    ConstructContext*, CachedStreamReader)

  Description:  バイナリファイルヘッダを処理します。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpAnalyzeFileHeader(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    pStream )
{
    const u32 requireSize = sizeof(BinaryFileHeader);

    if( pStream->GetRemain() < requireSize )
    {
        return RequestData(pContext, pStream, requireSize);
    }
    if( pContext->GetRemain() < requireSize )
    {
        // 出力先バッファのサイズが足らなかった
        NW4R_WARNING(false, "buffer size too small");
        return CONSTRUCT_ERROR;
    }

    // 一時的にコピー
    // CopyTo(pContext, ...) だとバッファ位置が進んでしまうので CopyTo(void*, ...) を使う
    pStream->CopyTo(pContext->GetCurrentPtr<void*>(), requireSize);
    pContext->op = ConstructContext::DISPATCH;

    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpAnalyzeGLGR(
                                    ConstructContext*, CachedStreamReader)

  Description:  GLGR ブロックを処理します。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpAnalyzeGLGR(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    pStream )
{
    u8* pTempFileHead;
    u8* pTempGLGRTail;

    // GLGR ブロックの確保
    {
        const BinaryFileHeader& header = *pContext->GetCurrentPtr<const BinaryFileHeader*>();
        const u32 requireSize = pContext->header.size - sizeof(BinaryBlockHeader);

        if( header.signature != BINFILE_SIG_FONTA )
        {
            // バイナリファイルヘッダがロードされていない or これはフォントリソースではない
            NW4R_WARNING(false, "invalid brfnt file.");
            return CONSTRUCT_ERROR;
        }
        if( pStream->GetRemain() < requireSize )
        {
            return RequestData(pContext, pStream, requireSize);
        }
        if( pContext->GetRemain() < requireSize )
        {
            // 出力先バッファのサイズが足らなかった
            NW4R_WARNING(false, "buffer size too small");
            return CONSTRUCT_ERROR;
        }

        // GLGR ブロックを一時的にコピー
        // CopyTo(pContext, ...) だとバッファ位置が進んでしまうので CopyTo(void*, ...) を使う
        const u32 infoSize = sizeof(BinaryFileHeader) + pContext->header.size;

        pTempFileHead      = pContext->GetCurrentPtr<u8*>();
        u8* blockHeaderPos = pTempFileHead + sizeof(BinaryFileHeader);
        u8* blockBodyPos   = blockHeaderPos + sizeof(BinaryBlockHeader);
        pTempGLGRTail      = blockHeaderPos + pContext->header.size;
        std::memcpy(blockHeaderPos, &pContext->header, sizeof(BinaryBlockHeader));
        pStream->CopyTo(blockBodyPos, requireSize);

        //---- 妥当性のチェック
        if( ! IsValidResource(pTempFileHead, infoSize) )
        {
            return CONSTRUCT_ERROR;
        }
    }

    u16 numSheet;
    u16 numGlyphsPerSheet;
    u32 numBlock;
    u16* pTempAdjustArray;
    u32 sizeAdjustTable;

    //---- グリフインデックス調整テーブルの構築
    {
        FontGlyphGroupsAcs gg(pTempFileHead);

        numSheet          = gg.GetNumSheet();
        numGlyphsPerSheet = gg.GetNumGlyphsPerSheet();
        numBlock          = gg.GetNumBlock();

        sizeAdjustTable = RoundUp(gg.GetNumSheet() * sizeof(u16), 4);   // 4 バイトアライメント
        pTempAdjustArray = reinterpret_cast<u16*>(RoundDown(pTempFileHead + pContext->GetRemain() - sizeAdjustTable, 2));
        if( pContext->GetRemain() < (pTempGLGRTail - pTempFileHead) + sizeAdjustTable )
        {
            // 出力先バッファのサイズが足らなかった
            NW4R_WARNING(false, "buffer size too small");
            return CONSTRUCT_ERROR;
        }

        // 以下 AdjustTable を一時的にフラグ配列として使う。
        // フラグを全て落とす。
        for( int sheetNo = 0; sheetNo < gg.GetNumSheet(); ++sheetNo )
        {
            pTempAdjustArray[sheetNo] = 0;
        }

        // 使用するシートに対応するフラグを立てていく。
        for( int setNo = 0; setNo < gg.GetNumSet(); ++setNo )
        {
            const char* setName = gg.GetSetName(setNo);

            if( IsNullString(pContext->GetGlyphGroups()) || IncludeName(pContext->GetGlyphGroups(), setName) )
            {
                for( int sheetNo = 0; sheetNo < gg.GetNumSheet(); ++sheetNo )
                {
                    if( gg.IsUseSheet(setNo, sheetNo) )
                    {
                        pTempAdjustArray[sheetNo] = 1;
                    }
                }
            }
        }

        // フラグを元に AdjustTable を構築する。
        u16 adjust = 0;
        for( int sheetNo = 0; sheetNo < gg.GetNumSheet(); ++sheetNo )
        {
            u16 &entry = pTempAdjustArray[sheetNo];

            if( entry == 1 )
            {
                entry = adjust;
            }
            else
            {
                entry = 0xFFFF;
                adjust += gg.GetNumGlyphsPerSheet();
            }
        }
    }

    // 以下では一時的にコピーした GLGR ブロックを破壊するので以降アクセスする事はできない。

    // Context の構築
    {
        u16* pTrueAdjustArray = pContext->GetCurrentPtr<u16*>();
        pContext->Advance(sizeAdjustTable);

        std::copy(pTempAdjustArray, pTempAdjustArray + numSheet, pTrueAdjustArray);

        pContext->SetGLGR( pTrueAdjustArray,
                           numSheet,
                           numGlyphsPerSheet,
                           numBlock );

        pContext->op = ConstructContext::DISPATCH;
    }

    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpAnalyzeFINF(
                                    ConstructContext*, CachedStreamReader)

  Description:  FINF ブロックを処理します。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpAnalyzeFINF(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    /*pStream*/ )
{
//#pragma unused(pStream)
    // バイナリブロックヘッダはコピーしない
    u32 copySize = pContext->header.size - sizeof(BinaryBlockHeader);

    if( pContext->GetRemain() < copySize )
    {
        // 出力先バッファのサイズが足らなかった
        NW4R_WARNING(false, "buffer size too small");
        return CONSTRUCT_ERROR;
    }

    // コピータスク起動
    pContext->pFINF = pContext->GetCurrentPtr<FontInformation*>();
    pContext->SetupCopyTask(copySize);
    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpAnalyzeCMAP(
                                    ConstructContext*, CachedStreamReader)

  Description:  CMAP ブロックを処理します。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpAnalyzeCMAP(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    /*pStream*/ )
{
//#pragma unused(pStream)
    // 前の CMAP ブロックまたは FINF ブロックのリンクを更新する
    FontCodeMap* pDstCMAP = pContext->GetCurrentPtr<FontCodeMap*>();

    if( pContext->pPrevCMAP != NULL )
    {
        pContext->pPrevCMAP->pNext = pDstCMAP;
    }
    else
    {
        NW4R_POINTER_ASSERT( pContext->pFINF );
        pContext->pFINF->pMap = pDstCMAP;
    }
    pContext->pPrevCMAP = pDstCMAP;

    // バイナリブロックヘッダはコピーしない
    u32 copySize = pContext->header.size - sizeof(BinaryBlockHeader);

    if( pContext->GetRemain() < copySize )
    {
        // 出力先バッファのサイズが足らなかった
        NW4R_WARNING(false, "buffer size too small");
        return CONSTRUCT_ERROR;
    }

    // データ本体のコピータスク
    pContext->SetupCopyTask(copySize);
    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpAnalyzeCWDH(
                                    ConstructContext*, CachedStreamReader)

  Description:  CWDH ブロックを処理します。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpAnalyzeCWDH(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    /*pStream*/ )
{
//#pragma unused(pStream)
    // 前の CWDH ブロックまたは FINF ブロックのリンクを更新する
    FontWidth* pDstCWDH = pContext->GetCurrentPtr<FontWidth*>();

    if( pContext->pPrevCWDH != NULL )
    {
        pContext->pPrevCWDH->pNext = pDstCWDH;
    }
    else
    {
        NW4R_POINTER_ASSERT( pContext->pFINF );
        pContext->pFINF->pWidth = pDstCWDH;
    }
    pContext->pPrevCWDH = pDstCWDH;

    // バイナリブロックヘッダはコピーしない
    u32 copySize = pContext->header.size - sizeof(BinaryBlockHeader);

    if( pContext->GetRemain() < copySize )
    {
        // 出力先バッファのサイズが足らなかった
        NW4R_WARNING(false, "buffer size too small");
        return CONSTRUCT_ERROR;
    }

    // データ本体のコピータスク
    pContext->SetupCopyTask(copySize);
    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpAnalyzeTGLP(
                                    ConstructContext*, CachedStreamReader)

  Description:  TGLP ブロックを処理します。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpAnalyzeTGLP(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    pStream )
{
    const u32 requireSize = sizeof(FontTextureGlyph);
    const u32 copySize    = sizeof(FontTextureGlyph);

    if( pStream->GetRemain() < requireSize )
    {
        return RequestData(pContext, pStream, requireSize);
    }
    if( pContext->GetRemain() < copySize )
    {
        // 出力先バッファのサイズが足らなかった
        NW4R_WARNING(false, "buffer size too small");
        return CONSTRUCT_ERROR;
    }
    NW4R_POINTER_ASSERT( pContext->pFINF );

    // FINF ブロックのリンクを更新する
    pContext->pFINF->pGlyph = pContext->GetCurrentPtr<FontTextureGlyph*>();

    // シート以外の部分をコピー
    //pStream->CopyTo(pContext, sizeof(FontTextureGlyph));
    pStream->CopyTo(pContext, copySize);

    // シートが圧縮されているかを取得
    const bool bSheetCompressed = ((pContext->pFINF->pGlyph->sheetFormat & FONT_SHEETFORMAT_COMPRESSED_FLAG) != 0);
    pContext->pFINF->pGlyph->sheetFormat &= FONT_SHEETFORMAT_TEXFMT_MASK;

    // シート数を更新
    {
        u16 numLoadSheet = 0;
        for( int i = 0; i < pContext->GetNumSheets(); ++i )
        {
            if( pContext->IsRequireSheet(i) )
            {
                numLoadSheet++;
            }
        }
        pContext->pFINF->pGlyph->sheetNum = numLoadSheet;
    }

    // シートのストリーム内オフセットを取得
    u32 sheetOffset  = reinterpret_cast<u32>(pContext->pFINF->pGlyph->sheetImage);

    // シートにはターゲット指定のアライメントが必要
    pContext->Align( NW::Font::NWFontEnvironment::TextureDataAlignSize);
    pContext->pFINF->pGlyph->sheetImage = pContext->GetCurrentPtr<u8*>();

    // シートが圧縮されているかに応じて分岐
    const ConstructContext::Operation nextOp
        = bSheetCompressed ? ConstructContext::PREPAIR_EXPAND_SHEET:
                             ConstructContext::PREPAIR_COPY_SHEET;

    // シートのある位置までスキップ
    u32 streamOffset = pContext->streamOffset + pStream->GetOffset();
    u32 skipSize     = sheetOffset - streamOffset;
    pContext->SetupSkipTask(skipSize, nextOp);

    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpPrepairCopySheet(
                                    ConstructContext*, CachedStreamReader)

  Description:  シートのコピー処理をセットアップします。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpPrepairCopySheet(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    /*pStream*/ )
{
//#pragma unused(pStream)
    // 全てのシートに対して処理が終わっているか？
    if( ! pContext->HasMoreSheet() )
    {
        pContext->op = ConstructContext::DISPATCH;
        return CONSTRUCT_CONTINUE;
    }

    const u32 copySize = pContext->pFINF->pGlyph->sheetSize;

    // ロードしなければならないシートか？
    if( pContext->IsRequireSheet() )
    {
        // ロードする
        if( pContext->GetRemain() < copySize )
        {
            NW4R_WARNING(false, "buffer size too small");
            return CONSTRUCT_ERROR;
        }

        // コピー
        pContext->SetupCopyTask(copySize, ConstructContext::PREPAIR_COPY_SHEET);
    }
    else
    {
        // ロードしない
        pContext->SetupSkipTask(copySize, ConstructContext::PREPAIR_COPY_SHEET);
    }

    // カレントシート番号を更新
    pContext->NextSheet();

    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpPrepairExpandSheet(
                                    ConstructContext*, CachedStreamReader)

  Description:  シートの展開処理をセットアップします。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpPrepairExpandSheet(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    pStream )
{
    // 全てのシートに対して処理が終わっているか？
    if( ! pContext->HasMoreSheet() )
    {
        pContext->op = ConstructContext::DISPATCH;
        return CONSTRUCT_CONTINUE;
    }

    u32 compSize = 0;
    u32 huffHead = 0;

    const u32 requireSize = sizeof(compSize) + sizeof(huffHead);

    if( pStream->GetRemain() < requireSize )
    {
        return RequestData(pContext, pStream, requireSize);
    }

    // 圧縮データサイズを読み込む
    pStream->CopyTo(&compSize, sizeof(compSize));
    std::memcpy(&huffHead, pStream->Get(0), sizeof(huffHead));
    // Get(0) はよくないがストリーム位置を移動させないため
    //
    // なお、compSizeとは異なり、huffHead については、
    // 特殊な格納方法が使用されているため、
    // バイトスワップの必要はありません。
    //

    const u32 srcSize = compSize;
    const u32 dstSize = CXGetUncompressedSize(&huffHead);

    // ロードしなければならないシートか？
    if( pContext->IsRequireSheet() )
    {
        // ロードする

        if( pContext->GetRemain() < srcSize + sizeof(*pContext->GetUncompContext()) )
        {
            NW4R_WARNING(false, "buffer size too small");
            return CONSTRUCT_ERROR;
        }

        // データ展開
        CXInitUncompContextHuffman( pContext->GetUncompContext(),
                                    pContext->GetCurrentPtr<u8*>() );
        pContext->SetupExtendTask(srcSize, ConstructContext::PREPAIR_EXPAND_SHEET);

        // ExtendTask では出力バッファ位置が更新されないので
        // 展開されるサイズ分あらかじめ進めておく必要がある
        pContext->Advance(dstSize);
    }
    else
    {
        // ロードしない
        pContext->SetupSkipTask(srcSize, ConstructContext::PREPAIR_EXPAND_SHEET);
    }

    // カレントシート番号を更新
    pContext->NextSheet();

    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpCopy(
                                    ConstructContext*, CachedStreamReader)

  Description:  データを入力ストリームから構築コンテキストの
                出力バッファへコピーします。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpCopy(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    pStream )
{
    const u32 requireSize = pStream->ManagedCopy(pContext);

    if( requireSize == 0 )
    {
        // コピー終了
        pContext->EndTask();
    }
    else
    {
        // 更なるデータが必要
        return RequestData(pContext, pStream, requireSize);
    }

    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpSkip(
                                    ConstructContext*, CachedStreamReader)

  Description:  入力ストリームを読み捨てます。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpSkip(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    pStream )
{
    u32 skipSize = NW4R::Font::UnManaged::Min(pContext->TaskRemain(), pStream->GetRemain());

    pStream->Advance(skipSize);
    pContext->TaskProceed(skipSize);

    if( pContext->TaskRemain() == 0 )
    {
        // スキップ終了
        pContext->EndTask();
    }
    else
    {
        // スキップ継続
        return RequestData(pContext, pStream, pContext->TaskRemain());
    }

    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpExpand(
                                    ConstructContext*, CachedStreamReader)

  Description:  入力ストリームを展開しつつ構築コンテキストの
                出力バッファに書き出します。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpExpand(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    pStream )
{
    NW4R_ASSERT( ! pStream->HasBufData() );
    u32 extendSize = NW4R::Font::UnManaged::Min(pContext->TaskRemain(), pStream->GetRemain());

    s32 remain = CXReadUncompHuffman( pContext->GetUncompContext(),
                                      reinterpret_cast<const u8*>(pStream->Get(extendSize)),
                                      extendSize );

    pContext->TaskProceed(extendSize);

    if( pContext->TaskRemain() == 0 )
    {
        // 展開終了
        NW4R_ASSERT( remain == 0 );
        pContext->EndTask();
    }
    else
    {
        // 更なるデータが必要
        NW4R_ASSERT( remain != 0 );
        return RequestData(pContext, pStream, pContext->TaskRemain());
    }

    return CONSTRUCT_CONTINUE;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::ConstructOpFatalError(
                                    ConstructContext*, CachedStreamReader)

  Description:  既に致命的なエラーが発生している場合にも関わらず
                処理が継続された場合に呼び出されます。

  Arguments:    pContext:   ストリーム構築コンテキスト。
                pStream:    入力ストリーム。

  Returns:      処理状態を返します。
 *---------------------------------------------------------------------------*/
/* static */ ArchiveFontBase::ConstructResult
ArchiveFontBase::ConstructOpFatalError(
    ArchiveFontBase::ConstructContext*      pContext,
    ArchiveFontBase::CachedStreamReader*    /*pStream*/ )
{
//#pragma unused(pStream)
    pContext->op = ConstructContext::FATAL_ERROR;
    return CONSTRUCT_ERROR;
}





/* =======================================================================
        CachedStreamReader
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::Init()

  Description:  初期化します。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ArchiveFontBase::CachedStreamReader::Init()
{
    mStreamBegin        = NULL;
    mStreamPos          = NULL;
    mStreamEnd          = NULL;
    mpTempStrmBuf       = NULL;
    mpTempStrmBufPos    = NULL;
    mpTempStrmBufEnd    = NULL;
    mRequireSize        = 0;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::Attach(const void*, u32)

  Description:  バッファに読み出されたストリームの一部を取り込みます。

  Arguments:    stream:     読み出されたストリームデータ。
                streamSize: stream に格納されているバッファのバイト数。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ArchiveFontBase::CachedStreamReader::Attach(const void* stream, u32 streamSize)
{
    mStreamBegin    = reinterpret_cast<const u8*>(stream);
    mStreamPos      = mStreamBegin;
    mStreamEnd      = mStreamBegin + streamSize;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::GetRemain()

  Description:  読み出されたストリームデータの残りサイズを取得します。

  Arguments:    なし。

  Returns:      読み出されたストリームデータの残りサイズを返します。
 *---------------------------------------------------------------------------*/
u32
ArchiveFontBase::CachedStreamReader::GetRemain() const
{
    return static_cast<u32>((mStreamEnd - mStreamPos) + (mpTempStrmBufEnd - mpTempStrmBufPos));
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::Advance(u32)

  Description:  ストリームを読み捨てます。

  Arguments:    dx: 読み捨てるバイト数。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ArchiveFontBase::CachedStreamReader::Advance(u32 dx)
{
    const int sizeTempStrm = mpTempStrmBufEnd - mpTempStrmBufPos;
    if( sizeTempStrm > static_cast<int>(dx) )
    {
        mpTempStrmBufPos += dx;
    }
    else
    {
        mpTempStrmBufPos = mpTempStrmBufEnd;
        mStreamPos += (dx - sizeTempStrm);
    }
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::CopyTo(
                                                    ConstructContext*, u32)

  Description:  ConstructContext にストリームをコピーします。

  Arguments:    pContext:   コピー先の構築コンテキスト
                size:       コピーするサイズ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ArchiveFontBase::CachedStreamReader::CopyTo(ArchiveFontBase::ConstructContext* pContext, u32 size)
{
    NW4R_ASSERT( pContext->GetRemain() >= size );

    CopyTo(pContext->GetCurrentPtr<void*>(), size);
    pContext->Advance(size);
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::CopyTo(void*, u32)

  Description:  ストリームをコピーします。

  Arguments:    buf:    コピー先のバッファ。
                size:   コピーするサイズ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ArchiveFontBase::CachedStreamReader::CopyTo(void* buf, u32 size)
{
    NW4R_ASSERT( size <= GetRemain() );
    const int sizeTempStrm = mpTempStrmBufEnd - mpTempStrmBufPos;
    u8* buf8 = reinterpret_cast<u8*>(buf);

    if( sizeTempStrm >= static_cast<int>(size) )
    {
        std::memcpy(buf8, mpTempStrmBufPos, size);
        mpTempStrmBufPos += size;
    }
    else
    {
        const u32 streamCopySize = size - sizeTempStrm;
        std::memcpy(buf8, mpTempStrmBufPos, static_cast<u32>(sizeTempStrm));
        std::memcpy(buf8 + sizeTempStrm, mStreamPos, streamCopySize);

        mpTempStrmBufPos = mpTempStrmBufEnd;
        mStreamPos += streamCopySize;
    }
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::MoveTo(void*, u32)

  Description:  ストリームを移動します。
                キャッシュされているストリームをキャッシュに書き込む場合に
                使用します。

  Arguments:    buf:    移動先のバッファ。
                size:   移動するサイズ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ArchiveFontBase::CachedStreamReader::MoveTo(void* buf, u32 size)
{
    NW4R_ASSERT( size <= GetRemain() );
    const int sizeTempStrm = mpTempStrmBufEnd - mpTempStrmBufPos;
    u8* buf8 = reinterpret_cast<u8*>(buf);

    if( sizeTempStrm >= static_cast<int>(size) )
    {
        std::memmove(buf8, mpTempStrmBufPos, size);
        mpTempStrmBufPos += size;
    }
    else
    {
        const u32 streamCopySize = size - sizeTempStrm;
        std::memmove(buf8, mpTempStrmBufPos, static_cast<u32>(sizeTempStrm));
        std::memmove(buf8 + sizeTempStrm, mStreamPos, streamCopySize);

        mpTempStrmBufPos = mpTempStrmBufEnd;
        mStreamPos += streamCopySize;
    }
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::ManagedCopy(
                                                            ConstructContext*)

  Description:  構築コンテキストへ必要な分のデータをコピーします。

  Arguments:    pContext:   コピー先の構築コンテキスト。

  Returns:      必要な分に足りないサイズを返します。
 *---------------------------------------------------------------------------*/
u32
ArchiveFontBase::CachedStreamReader::ManagedCopy(ArchiveFontBase::ConstructContext* pContext)
{
    NW4R_ASSERT( ! HasBufData() );
    u32 copySize = NW4R::Font::UnManaged::Min(static_cast<u32>(mStreamEnd - mStreamPos), pContext->TaskRemain());

    std::memcpy(pContext->GetCurrentPtr<void*>(), mStreamPos, copySize);
    mStreamPos += copySize;
    pContext->Advance(copySize);
    pContext->TaskProceed(copySize);

    return pContext->TaskRemain();
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::HasBufData()

  Description:  キャッシュされたデータが存在するか判定します。

  Arguments:    なし。

  Returns:      キャッシュされたデータが存在するなら true を返します。
 *---------------------------------------------------------------------------*/
bool
ArchiveFontBase::CachedStreamReader::HasBufData() const
{
    return mpTempStrmBufPos < mpTempStrmBufEnd;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::Get(u32)

  Description:  ストリーム中のデータを直接参照します。
                キャッシュされたデータが存在する場合は使用できません。

  Arguments:    size:   参照するデータサイズです。
                        参照されるデータは既に読み取られた物として扱われます。

  Returns:      ストリームデータへのポインタを返します。
 *---------------------------------------------------------------------------*/
const void*
ArchiveFontBase::CachedStreamReader::Get(u32 size)
{
    NW4R_ASSERT( size <= GetRemain() );
    const void* p = mStreamPos;
    mStreamPos += size;
    return p;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::RequestData()

  Description:  ストリームにデータの追加を要求します。

  Arguments:    pContext:   データ要求を管理する構築コンテキスト。
                size:       必要なデータサイズ。

  Returns:      要求に成功した場合に true を返します。
 *---------------------------------------------------------------------------*/
bool
ArchiveFontBase::CachedStreamReader::RequestData(
    ArchiveFontBase::ConstructContext*  pContext,
    u32                                 size )
{
    // ストリームにはデータが足りない状態でなければならない
    NW4R_ASSERT( GetRemain() < size );

    if( GetRemain() == 0 )
    {
        // ストリームデータが全くない場合はテンポラリバッファを作成しない
        mpTempStrmBuf       = NULL;
        mpTempStrmBufPos    = NULL;
        mpTempStrmBufEnd    = NULL;
        mRequireSize        = 0;
    }
    else
    {
        // テンポラリバッファに必要な空き領域があるかチェック
        if( pContext->GetRemain() < size * 2 )
        {
            // 無ければエラー
            return false;
        }

        u8* tempBuf = pContext->GetCurrentPtr<u8*>() + size;
        u32 curSize = GetRemain();

        // 既にテンポラリバッファが作られている場合があるので CopyTo は使えない
        MoveTo(tempBuf, curSize);

        mpTempStrmBuf       = tempBuf;
        mpTempStrmBufPos    = mpTempStrmBuf;
        mpTempStrmBufEnd    = mpTempStrmBuf + curSize;
        mRequireSize        = size;
    }

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         ArchiveFontBase::CachedStreamReader::GetOffset()

  Description:  現在管理しているストリーム断片中の現在位置を返します。

  Arguments:    なし。

  Returns:      ストリーム断片先頭から現在までに読み取ったバイト数。
 *---------------------------------------------------------------------------*/
u32
ArchiveFontBase::CachedStreamReader::GetOffset() const
{
    return static_cast<u32>((mStreamPos - mStreamBegin) + (mpTempStrmBufPos - mpTempStrmBuf));
}


        } /* namespace detail */
    } /* UnManaged */
    } /* namespace ut */
} /* namespace nw4r */
