﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifdef NW_LAYOUTEDITOR_CAFE

#include <scfont_ScalableFont.h>

namespace nw
{
namespace scfont
{

ScalableFont::InitializeArg::InitializeArg()
 : textureCache(NULL)
 , fontSize(40)
 , fontFace(0)
 , isAlternateCharSpaceWithOriginalWidth(false)
 , isAlternateCharSpaceWithOriginalWidthForNotReadyChar(false)
 , alternateChar('?')
 , lineFeedOffset(0)
{
}

ScalableFont::ScalableFont()
 : font::Font()
 , m_TextureCache(NULL)
 , m_FontSize(40)
 , m_FontHeight(50)
 , m_FontAscent(40)
 , m_FontBaselinePos(40)
 , m_FontLineFeed(50)
 , m_AlternateCharMode(USE_ALTERNATE_CHAR)
 , m_FontFace(0)
 , m_FontAlternateChar('?')
{
    m_FontDefaultCharWidth.left = 0;
    m_FontDefaultCharWidth.glyphWidth = 40;
    m_FontDefaultCharWidth.charWidth = 40;
}

ScalableFont::~ScalableFont()
{
}

void ScalableFont::Initialize(const InitializeArg& arg)
{
    if (arg.textureCache == NULL)
    {
        NW_ERR("arg.textureCache must be set.");
        return;
    }

    m_TextureCache = arg.textureCache;
    m_FontSize = arg.fontSize;
    m_FontFace = arg.fontFace;
    {
        const TextureCache::FontMetrics& metrics = m_TextureCache->GetFontMetrics(m_FontFace);
        // ResFontの表示に近くなるように、切り捨てではなく四捨五入する
        m_FontAscent = static_cast<int>(metrics.ascent_ratio * m_FontSize + 0.5f);
        m_FontHeight = static_cast<int>(metrics.height_ratio * m_FontSize + 0.5f);
        m_FontBaselinePos = static_cast<int>(metrics.boundingBoxAscentRatio * m_FontSize + 0.5f);
    }
    m_FontLineFeed = m_FontHeight + arg.lineFeedOffset;
    m_FontDefaultCharWidth.left = 0;
    m_FontDefaultCharWidth.glyphWidth = static_cast<u8>(m_FontSize);
    m_FontDefaultCharWidth.charWidth = static_cast<u8>(m_FontSize);

    if (arg.isAlternateCharSpaceWithOriginalWidth) {
        m_AlternateCharMode = USE_WHITESPACE;
        m_FontAlternateChar = CHARCODE_ALTERNATE_WHITESPACE;
    } else if (arg.isAlternateCharSpaceWithOriginalWidthForNotReadyChar) {
        m_AlternateCharMode = USE_WHITESPACE_FOR_NOT_READY_CHAR;
        m_FontAlternateChar = static_cast<font::CharCode>(arg.alternateChar);
    } else {
        m_AlternateCharMode = USE_ALTERNATE_CHAR;
        m_FontAlternateChar = static_cast<font::CharCode>(arg.alternateChar);
    }

    // 代替文字を登録
    RegisterAlternateCharGlyph();

    // 古い基底クラスの仕様と合わせるために必要。
    InitReaderFunc(GetEncoding());
}

void ScalableFont::RegisterAlternateCharGlyph() const
{
    RegisterGlyph(m_FontAlternateChar);
    {
        nw::scfont::GlyphNode* pGlyphNode = m_TextureCache->FindGlyphNode(m_FontAlternateChar, m_FontSize, m_FontFace);
        if (pGlyphNode != NULL)
        {
            pGlyphNode->SetSystemReserved(true);
        }
    }
    if (m_AlternateCharMode == USE_WHITESPACE_FOR_NOT_READY_CHAR) {
        // フォントにあってまだプロットされていない文字が表示する文字の幅の空白になるモードでは、通常の代替文字
        // の他に空白のプロットとシステム予約も必要
        RegisterGlyph(CHARCODE_ALTERNATE_WHITESPACE);
        GlyphNode* pGlyphNode = m_TextureCache->FindGlyphNode(CHARCODE_ALTERNATE_WHITESPACE, m_FontSize, m_FontFace);
        if (pGlyphNode != NULL)
        {
            m_TextureCache->FindGlyphNode(CHARCODE_ALTERNATE_WHITESPACE, m_FontSize, m_FontFace)->SetSystemReserved(true);
        }
    }
}

int ScalableFont::GetWidth() const
{
    return m_FontSize;
}

int ScalableFont::GetHeight() const
{
    return m_FontHeight;
}

int ScalableFont::GetAscent() const
{
    return m_FontAscent;
}

int ScalableFont::GetDescent() const
{
    return m_FontHeight - m_FontAscent;
}

int ScalableFont::GetMaxCharWidth() const
{
    return m_FontSize;
}

bool ScalableFont::HasKerning() const
{
    return m_TextureCache->HasKerning();
}

font::Font::Type ScalableFont::GetType() const
{
    return font::Font::Type::TYPE_SCALABLE;
}

font::FontType ScalableFont::GetFontType() const
{
    return font::FontType::FONT_TYPE_NNGCTEXTURE;
}

font::FontEncoding ScalableFont::GetEncoding() const
{
    return font::FontEncoding::FONT_ENCODING_UTF16;
}

GXTexFmt ScalableFont::GetTextureFormat() const
{
    return (GXTexFmt)GX_TF_I8;
}

int ScalableFont::GetLineFeed() const
{
    return m_FontLineFeed;
}

const font::CharWidths ScalableFont::GetDefaultCharWidths() const
{
    return m_FontDefaultCharWidth;
}

void ScalableFont::SetLineFeed(int linefeed)
{
    m_FontLineFeed = linefeed;
}

void ScalableFont::SetDefaultCharWidths(const font::CharWidths& widths)
{
    m_FontDefaultCharWidth = widths;
}

bool ScalableFont::SetAlternateChar(font::CharCode c)
{
    // 代替文字として文字幅と同じスペースを使う設定のときは、代替文字は変更不可能
    if (m_AlternateCharMode != USE_WHITESPACE && m_TextureCache->IsGlyphExistInFont(c, m_FontFace))
    {
        m_FontAlternateChar = c;
        return true;
    }
    else
    {
        return false;
    }
}

int ScalableFont::GetCharWidth(font::CharCode c) const
{
    const GlyphNode* node = m_TextureCache->FindGlyphNode(c, m_FontSize, m_FontFace);
    if (node && ! node->IsPlottingOrNotInFont())
    {
        return node->GetAdvanceX();
    }
    else if (m_AlternateCharMode != USE_ALTERNATE_CHAR && m_TextureCache->IsGlyphExistInFont(c, m_FontFace))
    {
        // 代替文字として文字幅と同じスペースを使う場合は、文字がプロットされておらず、かつ文字が
        // フォント内にあれば、直接スケーラブルフォントエンジンから取得する
        return m_TextureCache->CalcCharWidth(c, m_FontSize, m_FontFace);
    }
    else
    {
        // 代替文字を使う設定のとき、もしくは文字がフォントに含まれていなければ、
        // 代替文字をテクスチャキャッシュから探して幅を返す
        node = m_TextureCache->FindGlyphNode(m_FontAlternateChar, m_FontSize, m_FontFace);
        if (node == NULL)
        {
            return 0;
        }
        return node->GetAdvanceX();
    }
}

const font::CharWidths ScalableFont::GetCharWidths(font::CharCode c) const
{
    font::CharWidths widths;
    const GlyphNode* node = m_TextureCache->FindGlyphNode(c, m_FontSize, m_FontFace);
    if (node == NULL || node->IsPlottingOrNotInFont())
    {
        if (m_AlternateCharMode != USE_ALTERNATE_CHAR && m_TextureCache->IsGlyphExistInFont(c, m_FontFace))
        {
            // 代替文字として文字幅と同じスペースを使う場合は、文字がプロットされておらず、かつ文字が
            // フォント内にあれば、直接スケーラブルフォントエンジンから取得する
            int width = m_TextureCache->CalcCharWidth(c, m_FontSize, m_FontFace);
            widths.left = 0;
            widths.glyphWidth = static_cast<u8>(width);
            widths.charWidth = static_cast<u8>(width);
            return widths;
        }
        else
        {
            // 代替文字を使う設定のとき、もしくは文字がフォントに含まれていなければ、
            // 代替文字をテクスチャキャッシュから探す
            node = m_TextureCache->FindGlyphNode(m_FontAlternateChar, m_FontSize, m_FontFace);
        }
    }

    widths.left = static_cast<s8>(node->GetLeftOffset());
    widths.glyphWidth = static_cast<u8>(node->GetGlyphWidth());
    widths.charWidth = static_cast<u8>(node->GetAdvanceX());
    return widths;
}

void ScalableFont::GetGlyph(font::Glyph* pGlyph, font::CharCode c) const
{
    GlyphNode* node = m_TextureCache->FindGlyphNode(c, m_FontSize, m_FontFace);
    int charWidth;
    if (node == NULL || ! node->IsUsable())
    {
        if (m_AlternateCharMode != USE_ALTERNATE_CHAR && m_TextureCache->IsGlyphExistInFont(c, m_FontFace))
        {
            // 代替文字として文字幅と同じスペースを使う場合は、文字がプロットされておらず、かつ文字が
            // フォント内にあれば、スケーラブルフォントエンジンから文字幅を取得する
            charWidth = m_TextureCache->CalcCharWidth(c, m_FontSize, m_FontFace);
            // 文字としては空白を使用する
            node = m_TextureCache->FindGlyphNode(CHARCODE_ALTERNATE_WHITESPACE, m_FontSize, m_FontFace);
        }
        else
        {
            // 代替文字を使う設定のとき、もしくは文字がフォントに含まれていなければ、
            // 代替文字をテクスチャキャッシュから探す
            node = m_TextureCache->FindGlyphNode(m_FontAlternateChar, m_FontSize, m_FontFace);
            if (node == NULL)
            {
                // 代替文字のグリフが存在しない場合に例外が発生しないように
                // 文字幅が 0 の表示されないグリフ情報を返しておく。
                memset(pGlyph, 0, sizeof(font::Glyph));
                return;
            }
            else
            {
                charWidth = node->GetAdvanceX();
            }
        }
    }
    else
    {
        charWidth = node->GetAdvanceX();
    }

    pGlyph->widths.left = node->GetLeftOffset();
    pGlyph->widths.glyphWidth = node->GetGlyphWidth();
    pGlyph->widths.charWidth = static_cast<u16>(charWidth);
    pGlyph->widths.rawWidth = node->GetCacheWidth();
    // 文字の高さをそのまま渡すとテクスチャ補完の影響で下が切れてしまうため、+1する
    pGlyph->height = node->GetGlyphHeight() + 1;
    pGlyph->rawHeight = node->GetCacheHeight() + 1;
    pGlyph->cellX = node->GetCachePosX();
    pGlyph->cellY = node->GetCachePosY();
    pGlyph->pTexture = m_TextureCache->GetTextureCacheBitMap();
//  pGlyph->sheetIndex = 0; // 利用しないのでコメントアウト。
    pGlyph->texFormat = (GXTexFmt)GX_TF_I8;
    pGlyph->texWidth = static_cast<u16>(m_TextureCache->GetTextureCacheWidth());
    pGlyph->texHeight = static_cast<u16>(m_TextureCache->GetTextureCacheHeight());
    pGlyph->pTextureObject = m_TextureCache->GetTextureObject();
    pGlyph->baselineDifference = static_cast<s16>(node->GetBaselinePos() - m_FontBaselinePos);
    pGlyph->sheetNo = 0;
    pGlyph->isBitmapFont = false;
    //pGlyph->sheetFormat = 0; // 利用しないのでコメントアウト
    node->SetUsed(true);
}

bool ScalableFont::HasGlyph(font::CharCode c) const
{
    GlyphNode* node = m_TextureCache->FindGlyphNode(c, m_FontSize, m_FontFace);
    return (node && node->IsUsable());
}

int ScalableFont::GetKerning(font::CharCode c0, font::CharCode c1) const
{
    if (IsEnalbeKerning())
    {
        return m_TextureCache->CalcKerning(c0, c1, m_FontSize, m_FontFace);
    }
    else
    {
        return 0;
    }
}

font::FontEncoding ScalableFont::GetCharacterCode() const
{
    return font::FONT_ENCODING_UTF16;
}

int ScalableFont::GetBaselinePos() const
{
    return m_FontBaselinePos;
}

int ScalableFont::GetCellHeight() const
{
    return m_FontHeight;
}

int ScalableFont::GetCellWidth() const
{
    return m_FontSize;
}

void ScalableFont::EnableLinearFilter(bool atSmall, bool atLarge)
{
    (void)atSmall;
    (void)atLarge;
}

bool ScalableFont::IsLinearFilterEnableAtSmall() const
{
    return true;
}

bool ScalableFont::IsLinearFilterEnableAtLarge() const
{
    return true;
}

u32  ScalableFont::GetTextureWrapFilterValue() const
{
    // 使われていない
    return 0;
}

bool ScalableFont::IsColorBlackWhiteInterpolationEnabled() const
{
    return true;
}

bool ScalableFont::IsBorderEffectEnabled() const
{
    return (m_TextureCache) ? m_TextureCache->IsBorderEffectEnabled(m_FontFace) : false;
}

void ScalableFont::SetColorBlackWhiteInterpolationEnabled(bool /* flag */)
{
}

// 追加：.NET モジュールでテクスチャの検索を行う際に利用します。
int ScalableFont::GetSheetNum() const
{
    return 1;
}

} // namespace nw::scfont
} // namespace nw

#endif // NW_LAYOUTEDITOR_CAFE
