﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SCFONT_FREE_LIST_H_
#define NW_SCFONT_FREE_LIST_H_

#include <nw/types.h>

namespace nw
{
namespace scfont
{

/**
 *	固定サイズのメモリエリアを管理する仕組み（フリーリスト）です。
 */
class FreeList
{
public:
    FreeList() : mFree(NULL), mWork(NULL) {}

    /**
     *	フリーリストを初期化します。
     *
     *	@param work			ワーク領域。elem_size*numの大きさとしてください。４バイトアライメントである必要があります。
     *	@param elem_size	一つの要素のサイズ。４の倍数としてください。
     *	@param num			要素の最大数。
     */
    void Init(void* work, int elem_size, int num)
    {
        NW_ASSERT(work);
        NW_ASSERT(elem_size > 0 && (elem_size & 0x3) == 0);
        NW_ASSERT(num > 0);

        void** work_as_ptr = static_cast<void**>(work);
        int size_div4 = elem_size / sizeof(void*);

        mFree = work;
        /*
         *  フリーリストに繋がれている間は、領域をリンクリストとして使う。
         */
        for (int i = 0; i < num - 1; i++) {
            work_as_ptr[size_div4*i] = &work_as_ptr[size_div4*(i + 1)];
        }
        work_as_ptr[size_div4*(num - 1)] = NULL;

        mWork = work;
    }

    /**
     * フリーリストから要素を得ます。
     *
     * @return	新しい要素へのポインタ。領域のデータは不定です。
     *			フリーリストが空のときにはNULLが返ります。
     */
    void* Get()
    {
        void* ret = mFree;

        if (ret == NULL) {
            return NULL;
        } else {
            mFree = *static_cast<void**>(ret);	// 次の要素のポインタを辿ってmFreeに入れる
            return ret;
        }
    }

    /**
     *	不要な要素をフリーリストへ戻します。
     *
     *	@param elem		要素へのポインタ。
     */
    void Put(void* elem)
    {
        // リストに戻されたのだから、領域内にポインタを書き込んでもOK
        //
        // 戻されたものの先頭に、今のmFreeのアドレスを記憶しておく
        // そして、mFreeを、今返されたelemにする。
        *static_cast<void**>(elem) = mFree;
        mFree = elem;
    }

    /**
     *	フリーリストのワークエリアへのポインタを得ます。
     *
     *	@return	ワークエリアへのポインタ
     */
    void* GetWork() const
    {
        return mWork;
    }

    /**
     *	内部状態を生成時の状態に戻します。
     *
     *	initで与えたバッファを解放したときなど、内部に保持しているポインタが
     *	使用できなくなったとき等にご使用ください。
     *
     *	このメソッドを呼んだ後再び使うには、initを呼び出す必要があります。
     */
    void Cleanup()
    {
        mFree = NULL;
        mWork = NULL;
    }

    /**
     *	フリーリストが空になっているか否かを返します。
     *
     *	@retval true	空になっている（もう要素を確保できない）
     *	@retval false	まだ要素がある
     */
    bool IsEmpty() const
    {
        return (mFree == NULL);
    }


private:
    void* mFree;		// 未使用領域のリンクリストの先頭
    void* mWork;		// ワークエリアへのポインタ


};

} // namespace nw::scfont
} // namespace nw

#endif // NW_SCFONT_FREE_LIST_H_
