﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_UT_LIST_H_
#define NW4R_UT_LIST_H_

#include <stddef.h>
#include <dolphin/types.h>


namespace nw4r {
namespace ut   {

/*---------------------------------------------------------------------------*
  Name:         Link

  Description:  双方向リンクリストのノード構造体です。この構造体をリスト構造
                で繋ぎたい構造体のメンバとして格納します。
 *---------------------------------------------------------------------------*/
typedef struct
{
    void*       prevObject;     // 前に繋がれているオブジェクトへのポインタ。
    void*       nextObject;     // 次に繋がれているオブジェクトへのポインタ。

} Link;


/*---------------------------------------------------------------------------*
  Name:         lutList

  Description:  双方向リンクリスト構造体です。
 *---------------------------------------------------------------------------*/
typedef struct
{
    void*       headObject;     // 先頭に繋がれているオブジェクトへのポインタ。
    void*       tailObject;     // 後尾に繋がれているオブジェクトへのポインタ。
    u16         numObjects;     // リストに繋がれているオブジェクトの個数。
    u16         offset;         // NNSFndLink型の構造体メンバのオフセット。

} List;


/*---------------------------------------------------------------------------*
  Name:         NW4R_UT_LIST_INIT

  Description:  リスト構造体を初期化するためのマクロです。実際の初期化には、
                List_Init()関数により行われます。

                このマクロでは、指定された構造体名とLink型メンバ変数名から、
                offsetofマクロを使ってオフセットを求め、List_Init関数に渡して
                います。

  Arguments:    list:       リンク構造体へのポインタ。
                structName: リストに繋げたいオブジェクトの構造体名。
                linkName:   このオブジェクトのリンクに使用されるLink型
                            のメンバ変数名。

  Returns:      なし。
 *---------------------------------------------------------------------------*/

#define NW4R_UT_LIST_INIT(list, structName, linkName) \
            ::nw4r::ut::List_Init(list, offsetof(structName, linkName))

#define NW4R_UT_INIT_LIST(list, structName, linkName) \
            ::nw4r::ut::List_Init(list, offsetof(structName, linkName))


/*---------------------------------------------------------------------------*
    関数プロトタイプ。

 *---------------------------------------------------------------------------*/

void    List_Init(
                List*       list,
                u16         offset);

void    List_Append(
                List*       list,
                void*       object);

void    List_Prepend(
                List*       list,
                void*       object);

void    List_Insert(
                List*       list,
                void*       target,
                void*       object);

void    List_Remove(
                List*       list,
                void*       object);

void*   List_GetNext(
                List*       list,
                void*       object);

void*   List_GetPrev(
                List*       list,
                void*       object);

void*   List_GetNth(
                List*       list,
                u16         index);


/* namespace ut */
}
/* namespace nw4r */
}

/* NW4R_UT_LIST_H_ */
#endif
