﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_UT_ARCHIVEFONTBASE_H_
#define NW4R_UT_ARCHIVEFONTBASE_H_

#include <dolphin/types.h>
//#include <revolution/cx.h>
#include "ResFontBase.h"
#include "inlines.h"
#include "nw4r/misc.h"

#ifndef __CX_UNCOMPRESSION_H__
#include "UncompressHuffman.h"
#endif


namespace NW4R
{
namespace Font
{
namespace UnManaged
{
namespace detail
{

class ArchiveFontBase
    : public ResFontBase
{
public:
    struct ConstructContext;

protected:
    struct CachedStreamReader
    {
        friend struct ConstructContext;

    public:
        u32         GetRemain() const;
        void        Advance(u32 dx);
        void        CopyTo(ConstructContext* pContext, u32 size);
        void        CopyTo(void* buf, u32 size);
        void        MoveTo(void* buf, u32 size);
        u32         ManagedCopy(ConstructContext* pContext);
        bool        HasBufData() const;
        const void* Get(u32 size);
        void        Attach(const void* stream, u32 streamSize);
        bool        RequestData(ConstructContext* pContext, u32 size);
        u32         GetOffset() const;

    private:
        void Init();

    private:
        const u8* mStreamBegin;
        const u8* mStreamPos;
        const u8* mStreamEnd;
        u8*       mpTempStrmBuf;
        u8*       mpTempStrmBufPos;
        u8*       mpTempStrmBufEnd;

        u32       mRequireSize;
    };

public:
    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    // 構築処理の返り値
    enum ConstructResult
    {
        CONSTRUCT_MORE_DATA,
        CONSTRUCT_FINISH,
        CONSTRUCT_ERROR,
        CONSTRUCT_CONTINUE,
        NUM_OF_CONSTRUCT_RESULT
    };

    struct ConstructContext
    {
    public:
        enum Operation
        {
            DISPATCH,               // 0
            ANALYZE_FILE_HEADER,    // 1
            ANALYZE_GLGR,           // 2
            ANALYZE_FINF,           // 3
            ANALYZE_CMAP,           // 4
            ANALYZE_CWDH,           // 5
            ANALYZE_TGLP,           // 6
            PREPAIR_COPY_SHEET,     // 7
            PREPAIR_EXPAND_SHEET,   // 8
            COPY,                   // 9
            SKIP,                   // 10
            EXPAND,                 // 11
            FATAL_ERROR,            // 12
            NUM_OF_OPERATION,       //
            INVALID_OPERATION
        };

    public:
        void
        Init( void*         outBuffer,
              u32           outBufferSize,
              u16*          pAdjustTable,
              u16           nSheets,
              u16           nPerSheet,
              u32           numBlock )
        {
            Init(outBuffer, outBufferSize, NULL);
            SetGLGR(pAdjustTable, nSheets, nPerSheet, numBlock);
        }
        void
        Init( void*         outBuffer,
              u32           outBufferSize,
              const char*   glyphGroups )
        {
            streamReader.Init();
            pGlyphGroups = glyphGroups;
            pGlyphIndexAdjustTable = NULL;

            targetBuffer        = reinterpret_cast<u8*>(outBuffer);
            targetBufferEnd     = targetBuffer + outBufferSize;
            targetBufferCur     = targetBuffer;

            extendContext   = reinterpret_cast<CXUncompContextHuffman*>(
                                NW4R::Font::UnManaged::RoundDown(
                                    targetBufferEnd - sizeof(*extendContext), 4 ));

            op2             = INVALID_OPERATION;
            opSize          = 0;
            resNumBlock     = 1;    // GLGR ブロックに DISPATCH するために必要
            currentBlock    = 0;

            sheetIndex      = 0;
            numSheets       = 0;
            glyphsPerSheet  = 0;
            streamOffset    = 0;

            pFINF           = NULL;
            pPrevCWDH       = NULL;
            pPrevCMAP       = NULL;
            op              = INVALID_OPERATION;
        }
        void
        SetGLGR( u16*          pAdjustTable,
                 u16           nSheets,
                 u16           nPerSheet,
                 u32           numBlock )
         {
            pGlyphIndexAdjustTable  = pAdjustTable;
            numSheets               = nSheets;
            glyphsPerSheet          = nPerSheet;
            resNumBlock             = numBlock;
         }

        // 出力先バッファ位置走査
        template <typename PtrT>
        PtrT GetCurrentPtr() { return reinterpret_cast<PtrT>(targetBufferCur); }
        u32     GetRemain() const   { return static_cast<u32>(targetBufferEnd - targetBufferCur); }
        void    Align(u32 align)    { targetBufferCur = reinterpret_cast<u8*>(NW4R::Font::UnManaged::RoundUp(targetBufferCur, align)); }
        void    Advance(u32 dx)     { targetBufferCur += dx; }

    #ifdef NW4R_DEBUG
        u32     offset() const      { return static_cast<u32>(targetBufferCur - targetBuffer); }
        u32     sIndex() const  { return sheetIndex; }
    #endif

        // 展開完了
        void
        Finish(void** ppData, FontInformation** ppFontInfo, u16** ppAdjustTable)
        {
            // DCFlushRange(targetBuffer, static_cast<u32>(targetBufferEnd - targetBuffer));
            *ppData         = targetBuffer;
            *ppAdjustTable  = pGlyphIndexAdjustTable;
            *ppFontInfo     = pFINF;
        }

        // メンバ取得
        u16     GetNumSheets()         const    { return numSheets; }
        u16     GetNumGlyphsPerSheet() const    { return glyphsPerSheet; }
        CachedStreamReader& GetStreamReader()   { return streamReader; }
        CXUncompContextHuffman*
                GetUncompContext()              { return extendContext; }
        const char* GetGlyphGroups()   const    { return pGlyphGroups; }

        // メンバインクリメント
        void    NextBlock() { currentBlock++; }
        void    NextSheet() { sheetIndex++; }

        // 条件判定
        bool
        IsRequireSheet(int sheetNo) const
        {
            NW4R_ASSERT( sheetNo < numSheets );
            return pGlyphIndexAdjustTable[sheetNo] != 0xFFFF;
        }
        bool    IsRequireSheet() const  { return IsRequireSheet(sheetIndex); }
        bool    HasMoreSheet()   const  { return sheetIndex < numSheets; }
        bool    HasMoreBlock()   const  { return currentBlock < resNumBlock; }

        // 共通タスク
        void
        SetupCopyTask(u32 copySize, Operation nextOp=DISPATCH)
        {
            NW4R_BUFFERSIZE_ASSERT( copySize );
            opSize = copySize;
            op     = ConstructContext::COPY;
            op2    = nextOp;
        }
        void
        SetupSkipTask(u32 skipSize, Operation nextOp=DISPATCH)
        {
            NW4R_BUFFERSIZE_ASSERT( skipSize );
            opSize = skipSize;
            op     = ConstructContext::SKIP;
            op2    = nextOp;
        }
        void
        SetupExtendTask(u32 extendSize, Operation nextOp=DISPATCH)
        {
            NW4R_BUFFERSIZE_ASSERT( extendSize );
            opSize = extendSize;
            op     = ConstructContext::EXPAND;
            op2    = nextOp;
        }
        void    EndTask()               { op = op2; }
        u32     TaskRemain() const      { return opSize; }
        void    TaskProceed(u32 size)   { opSize -= Min(size, opSize); }


    public:
        FontInformation*    pFINF;
        FontWidth*          pPrevCWDH;
        FontCodeMap*        pPrevCMAP;

        Operation           op;
        BinaryBlockHeader   header;

        u32                 streamOffset;

    private:
        CachedStreamReader streamReader;
        CXUncompContextHuffman* extendContext;

        const char*     pGlyphGroups;
        u16*            pGlyphIndexAdjustTable;
        u8*             targetBuffer;
        u8*             targetBufferEnd;
        u8*             targetBufferCur;

        Operation       op2;
        u32             opSize;
        u32             resNumBlock;
        u32             currentBlock;

        u16             sheetIndex;
        u16             numSheets;
        u16             glyphsPerSheet;
    };


    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const char LOAD_GLYPH_ALL[1];


    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- コンストラクタ/デストラクタ
                                ArchiveFontBase();
    virtual                     ~ArchiveFontBase();

    //---- ResFontBase のオーバーライド
    virtual const CharWidths    GetCharWidths(
                                    CharCode c                      // 幅を求める文字
                                ) const;



    /* ------------------------------------------------------------------------
            型
       ------------------------------------------------------------------------ */
    // バイトスワップ処理をクラス外で行うために公開クラスとしました。
    class FontGlyphGroupsAcs
    {
    private:
        const BinaryFileHeader*         mpHeader;
        const FontGlyphGroupsBlock*     mpData;
        const u16*  mpNameOffsetArray;
        const u32*  mpSizeSheetsArray;
        const u32*  mpSizeCWDHArray;
        const u32*  mpSizeCMAPArray;
        const u32*  mpUseSheetArray;
        const u32*  mpUseCWDHArray;
        const u32*  mpUseCMAPArray;
        u32         mSizeSheetFlags;
        u32         mSizeCWDHFlags;
        u32         mSizeCMAPFlags;

    private:
        static bool
        GetBit(const u32* bits, u32 index)
        {
            u32 wordIndex = index / 32;
            u32 bitIndex  = index % 32;
            return ((bits[wordIndex] << bitIndex) & (1u << 31)) != 0;
        }

    public:
        FontGlyphGroupsAcs(const void* brfnt)
        {
            mpHeader = reinterpret_cast<const BinaryFileHeader*>(brfnt);
            mpData = reinterpret_cast<const FontGlyphGroupsBlock*>(mpHeader + 1);

            mSizeSheetFlags = sizeof(u32) * DivUp(mpData->body.numSheet, 32);
            mSizeCWDHFlags  = sizeof(u32) * DivUp(mpData->body.numCWDH, 32);
            mSizeCMAPFlags  = sizeof(u32) * DivUp(mpData->body.numCMAP, 32);

            // FontGlyphGroups はヘッダでは中途半端に定義されているため
            // 正確なサイズを得るためにオフセットで計算する
            const u32 sizePreBlock      = sizeof(BinaryFileHeader) + sizeof(BinaryBlockHeader);
            const u32 sizeStatic        = reinterpret_cast<u32>(reinterpret_cast<u16*>(reinterpret_cast<FontGlyphGroups*>(0)->nameOffsets));
            const u32 sizeNameOffsets   = sizeof(u16) * mpData->body.numSet;
            const u32 sizeSizeSheets    = sizeof(u32) * mpData->body.numSheet;
            const u32 sizeSizeCWDH      = sizeof(u32) * mpData->body.numCWDH;
            const u32 sizeSizeCMAP      = sizeof(u32) * mpData->body.numCMAP;
            const u32 sizeUseSheets     = mSizeSheetFlags * mpData->body.numSet;
            const u32 sizeUseCWDH       = mSizeCWDHFlags * mpData->body.numSet;
//            const u32 sizeUseCMAP       = mSizeCMAPFlags * numSet;

            const u32 offsetStatic      = RoundUp(sizePreBlock, 2);
            const u32 offsetNameOffsets = RoundUp(offsetStatic     + sizeStatic,        2);
            const u32 offsetSizeSheets  = RoundUp(offsetNameOffsets+ sizeNameOffsets,   4);
            const u32 offsetSizeCWDH    = RoundUp(offsetSizeSheets + sizeSizeSheets,    4);
            const u32 offsetSizeCMAP    = RoundUp(offsetSizeCWDH   + sizeSizeCWDH,      4);
            const u32 offsetUseSheets   = RoundUp(offsetSizeCMAP   + sizeSizeCMAP,      4);
            const u32 offsetUseCWDH     = RoundUp(offsetUseSheets  + sizeUseSheets,     4);
            const u32 offsetUseCMAP     = RoundUp(offsetUseCWDH    + sizeUseCWDH,       4);

            mpSizeSheetsArray   = reinterpret_cast<const u32*>(AddOffsetToPtr(mpHeader, offsetSizeSheets));
            mpSizeCWDHArray     = reinterpret_cast<const u32*>(AddOffsetToPtr(mpHeader, offsetSizeCWDH));
            mpSizeCMAPArray     = reinterpret_cast<const u32*>(AddOffsetToPtr(mpHeader, offsetSizeCMAP));
            mpUseSheetArray     = reinterpret_cast<const u32*>(AddOffsetToPtr(mpHeader, offsetUseSheets));
            mpUseCWDHArray      = reinterpret_cast<const u32*>(AddOffsetToPtr(mpHeader, offsetUseCWDH));
            mpUseCMAPArray      = reinterpret_cast<const u32*>(AddOffsetToPtr(mpHeader, offsetUseCMAP));
        }

        u16 GetNumBlock() const { return mpHeader->dataBlocks; }
        u32 GetFileSize() const { return mpHeader->fileSize; }
        u32 GetGLGRSize() const { return mpData->header.size; }

        const char* GetSetName(int setNo) const { return reinterpret_cast<const char*>(AddOffsetToPtr(mpHeader, reinterpret_cast<u16*>(mpData->body.nameOffsets)[setNo])); }


        u32 GetSizeCompSheet(int sheetNo) const { return mpSizeSheetsArray[sheetNo]; }
        u32 GetSizeCWDH(int cwdhNo) const { return mpSizeCWDHArray[cwdhNo]; }
        u32 GetSizeCMAP(int cmapNo) const { return mpSizeCMAPArray[cmapNo]; }

        // バイトスワップ処理を行うために、追加しました。
        void SetSizeCompSheet(int sheetNo, u32 size){ const_cast<u32*>(mpSizeSheetsArray)[sheetNo] = size; }
        void SetSizeCWDH(int cwdhNo, u32 size){ const_cast<u32*>(mpSizeCWDHArray)[cwdhNo] = size; }
        void SetSizeCMAP(int cmapNo, u32 size){ const_cast<u32*>(mpSizeCMAPArray)[cmapNo] = size; }


        bool IsUseSheet(int setNo, int sheetNo) const { return GetBit(mpUseSheetArray, setNo * mSizeSheetFlags * 8 + sheetNo); }
        bool IsUseCWDH (int setNo, int cwdhNo)  const { return GetBit(mpUseCWDHArray,  setNo * mSizeCWDHFlags  * 8 + cwdhNo); }
        bool IsUseCMAP (int setNo, int cmapNo)  const { return GetBit(mpUseCMAPArray,  setNo * mSizeCMAPFlags  * 8 + cmapNo); }

        u32 GetUseSheetFlags(int setNo, int flagSetNo) const { return mpUseSheetArray[setNo * mSizeSheetFlags / 4 + flagSetNo]; }
        u32 GetUseCWDHFlags (int setNo, int flagSetNo) const { return mpUseCWDHArray [setNo * mSizeCWDHFlags  / 4 + flagSetNo]; }
        u32 GetUseCMAPFlags (int setNo, int flagSetNo) const { return mpUseCMAPArray [setNo * mSizeCMAPFlags  / 4 + flagSetNo]; }

        // バイトスワップ処理を行うために、追加しました。
        void SetUseSheetFlags(int setNo, int flagSetNo, u32 flag){ const_cast<u32*>(mpUseSheetArray)[setNo * mSizeSheetFlags / 4 + flagSetNo] = flag; }
        void SetUseCWDHFlags (int setNo, int flagSetNo, u32 flag){ const_cast<u32*>(mpUseCWDHArray)[setNo * mSizeCWDHFlags  / 4 + flagSetNo] = flag; }
        void SetUseCMAPFlags (int setNo, int flagSetNo, u32 flag){ const_cast<u32*>(mpUseCMAPArray)[setNo * mSizeCMAPFlags  / 4 + flagSetNo] = flag; }


        u32 GetSheetSize()          const { return mpData->body.sheetSize; }
        u16 GetNumGlyphsPerSheet()  const { return mpData->body.glyphsPerSheet; }
        u16 GetNumSheet()           const { return mpData->body.numSheet; }
        u16 GetNumSet()             const { return mpData->body.numSet; }
        u16 GetNumCWDH()            const { return mpData->body.numCWDH; }
        u16 GetNumCMAP()            const { return mpData->body.numCMAP; }
        const BinaryBlockHeader* GetNextBlock() const
        { return reinterpret_cast<const BinaryBlockHeader*>(AddOffsetToPtr(mpData, mpData->header.size)); }
    };
protected:

    /* ------------------------------------------------------------------------
            定数
       ------------------------------------------------------------------------ */
    static const u16    ADJUST_OFFSET_SHEET_NOT_LOADED = 0xFFFF;


    /* ------------------------------------------------------------------------
            関数
       ------------------------------------------------------------------------ */
    //---- 構築/破棄
    void SetResourceBuffer(void* pUserBuffer, FontInformation* pFontInfo, u16* pAdjustTable);
    void* RemoveResourceBuffer();

    //---- グリフインデックス
    u16 AdjustIndex(u16 index) const;

    //---- GLGRブロック探査用
    static inline bool IsNullString(const char* str){ return *str == '\0'; }
    static bool IncludeName(const char* nameList, const char* name);
    static bool IsValidResource(const void* brfnt, u32 dataSize);

    //---- 構築処理
    static ConstructResult RequestData(ConstructContext* pContext, CachedStreamReader* pStream, u32 size);
    static ConstructResult ConstructOpDispatch          (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpAnalyzeFileHeader (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpAnalyzeGLGR       (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpAnalyzeFINF       (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpAnalyzeCMAP       (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpAnalyzeCWDH       (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpAnalyzeTGLP       (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpPrepairCopySheet  (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpPrepairExpandSheet(ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpCopy              (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpSkip              (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpExpand            (ConstructContext* pContext, CachedStreamReader* pStream);
    static ConstructResult ConstructOpFatalError        (ConstructContext* pContext, CachedStreamReader* pStream);


private:
    /* ------------------------------------------------------------------------
            変数
       ------------------------------------------------------------------------ */
    u16*                        mpGlyphIndexAdjustArray;
};



        } /* namespace detail */
    } /* UnManaged */
    } /* namespace ut */
} /* namespace nw4r */
#endif //  NW4R_UT_ARCHIVEFONTBASE_H_
