﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Imaging;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace NW4R.ColorPicker
{
    /// <summary>
    /// RGBカラースライダーコントロールクラス。
    /// </summary>
    internal class RGBSlider : ColorSlider
    {
        private System.ComponentModel.IContainer components = null;
        private ColorType			m_componentType	= ColorType.R;

        // HSVの保存
        private HSV m_hsv = new HSV(0, 0, 0);

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RGBSlider()
        {
            // この呼び出しは Windows フォーム デザイナで必要です。
            InitializeComponent();

            //	ちらつきの防止用
            this.SetStyle( ControlStyles.UserPaint | ControlStyles.AllPaintingInWmPaint | ControlStyles.DoubleBuffer, true );
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }

        #region デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            components = new System.ComponentModel.Container();
        }
        #endregion

        #region オーバーライド

        /// <summary>
        /// OnPaintのオーバーライド（ビットマップ表示）
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
            if( this.Enabled )
            {
                Graphics g = e.Graphics;
                using( Pen blackPen   = new Pen( Color.Black, 3 ) )
                {
                    g.DrawRectangle( blackPen, SliderValue-1, 1, 2, this.Size.Height-3 );
                    g.DrawRectangle( Pens.White, SliderValue-1, 1, 2, this.Size.Height-3 );
                }
            }
        }

        /// <summary>
        /// 指定位置色の取得
        /// </summary>
        /// <param name="p">カラーを取得する位置</param>
        /// <returns>選択色</returns>
        public override Color GetPointColor(int p)
        {
            int		col	= (int)Math.Round( p * 255d / (this.ClientSize.Width-1) );
            switch( m_componentType )
            {
                case ColorType.R:
                    m_color	= Color.FromArgb( col, m_color.G, m_color.B );
                    break;
                case ColorType.G:
                    m_color	= Color.FromArgb( m_color.R, col, m_color.B );
                    break;
                case ColorType.B:
                    m_color	= Color.FromArgb( m_color.R, m_color.G, col );
                    break;
                case ColorType.A:
                    m_color	= Color.FromArgb( col, m_color );
                    break;
                case ColorType.H:
                {
                    // HSVは保存したものを使うように変更
                    //HSV hsv = HSV.FromRGB(new RGB(m_color));
                    HSV hsv = m_hsv;
                    m_color = RGB.FromHSV(new HSV((int)Math.Round(p * 360d / (this.ClientSize.Width - 1)), hsv.S, hsv.V)).GetColor();
                    break;
                }
                case ColorType.S:
                {
                    // HSVは保存したものを使うように変更
                    //HSV hsv = HSV.FromRGB(new RGB(m_color));
                    HSV hsv = m_hsv;
                    m_color = RGB.FromHSV(new HSV(hsv.H, (int)Math.Round(p * 100d / (this.ClientSize.Width - 1)), hsv.V)).GetColor();
                    break;
                }
                default:
                {
                    // HSVは保存したものを使うように変更
                    //HSV hsv = HSV.FromRGB(new RGB(m_color));
                    HSV hsv = m_hsv;
                    m_color = RGB.FromHSV(new HSV(hsv.H, hsv.S, (int)Math.Round(p * 100d / (this.ClientSize.Width - 1)))).GetColor();
                    break;
                }
            }
            return m_color;
        }

        #endregion

        #region プロパティ

        /// <summary>
        /// RGBAのいずれか？
        /// </summary>
        public ColorType ComponentType
        {
            set {
                m_componentType	= value;
                MaxValue	= ( value == ColorType.H ) ? 360 :
                    ( ( value != ColorType.S && value != ColorType.V ) ? 255 : 100 );
                HSV	hsv	= HSV.FromRGB( new RGB( m_color ) );
                if( this.Created )
                    UpdateCurrentColorBitmap( m_color, hsv, false, true ) ;
            }
            get	{	return m_componentType;	}
        }
        #endregion

        /// <summary>
        /// 表示Bitmapの変更
        /// </summary>
        /// <param name="color">変更カラー</param>
        /// <param name="hsv">新しいカラー</param>
        /// <param name="resetPoint">選択している場所をリセットするか</param>
        /// <param name="flag">強制的に画像を書き換える</param>
        private void UpdateCurrentColorBitmap( Color color, HSV hsv, bool resetPoint, bool flag )
        {
            //	マウスダウン中でも値はアップデートする
            //if( m_isMouseDown )
            //	return;

            //	カラーの変更とビットマップの大きさの変更がなかった場合は何もしない
            Int32	w	= this.ClientSize.Width, x, col, total = 0, h = ( m_componentType == ColorType.A ) ? this.ClientSize.Height : 1;
            if( w <= 8 )
                w	= 8;
            if( resetPoint )
            {
                switch( m_componentType )
                {
                    case ColorType.R:	ColorValue	= color.R;	break;
                    case ColorType.G:	ColorValue	= color.G;	break;
                    case ColorType.B:	ColorValue	= color.B;	break;
                    case ColorType.A:	ColorValue	= color.A;	break;
                    case ColorType.H:	ColorValue	= hsv.H;	break;
                    case ColorType.S:	ColorValue	= hsv.S;	break;
                    default:			ColorValue	= hsv.V;	break;
                }
            }
            HSV	before	= HSV.FromRGB( new RGB( m_color ) );
            if( !flag && m_imageBMP != null && m_color.Equals( color ) && before.Equals( hsv ) && m_imageBMP.Size.Equals( new Size( w, h ) ) )
                return;
            m_color	= color;

            // HSVが更新されるので保存する
            m_hsv = hsv;

            //	マウスダウン中（マウスドラッグ中）は描画を変更しない
            if (m_isMouseDown)
                return;

            if( m_componentType == ColorType.A && !flag )
                return;
            //	コンポーネントタイプでHiとLow書き換え
            //	BMPをラインで書き換え
            double	add	= 255d / (w-1);
            if( m_imageBMP != null )
                m_imageBMP.Dispose();
            m_imageBMP	= new Bitmap( w, h );
            BitmapData	bitmapData	= m_imageBMP.LockBits( new Rectangle( this.DisplayRectangle.Location, new Size( this.DisplayRectangle.Width, h ) ),
                ImageLockMode.WriteOnly, PixelFormat.Format32bppRgb);
            IntPtr	pData	= bitmapData.Scan0;
            switch( m_componentType )
            {
                case ColorType.R:
                {
                    int		gg	= m_color.G;
                    int		bb	= m_color.B;
                    for( x = 0; x < w; x ++, total += 4 )
                    {
                        col		= ( (int)Math.Round(x*add) << 16 ) | ( gg << 8 ) | bb;
                        Marshal.WriteInt32( pData, total, col );
                    }
                    break;
                }
                case ColorType.G:
                {
                    int		rr	= m_color.R;
                    int		bb	= m_color.B;
                    for( x = 0; x < w; x ++, total += 4 )
                    {
                        col		= ( rr << 16 ) | ( (int)Math.Round(x*add) << 8 ) | bb;
                        Marshal.WriteInt32( pData, total, col );
                    }
                    break;
                }
                case ColorType.B:
                {
                    int		rr	= m_color.R;
                    int		gg	= m_color.G;
                    for( x = 0; x < w; x ++, total += 4 )
                    {
                        col		= ( rr << 16 ) | ( gg << 8 ) | (int)Math.Round(x*add);
                        Marshal.WriteInt32( pData, total, col );
                    }
                    break;
                }
                case ColorType.H:
                {
                    add	= 360d / (w-1);
                    int	s	= (int)Math.Round( hsv.S * 2.55d );
                    int	b	= (int)Math.Round( hsv.V * 2.55d );
                    int	num1	= (int)(b * (255 - s) / 255.0d);
                    for( x = 0; x < w; x ++, total += 4 )
                    {
                        if( s == 0 )
                            col	= ( b << 16 ) | ( b << 8 ) | b;
                        else
                        {
                            int	dif		= (int)( add*x * 6 ) % 360;
                            int	num2	= (int)(b * (255 - s * dif / 360d) / 255d);
                            int	num3	= (int)(b * (255 - s * (360 - dif) / 360d) / 255d);
                            switch( (int)(add*x) / 60 )
                            {
                                case 1:
                                    col		= ( num2 << 16 ) | ( b << 8 ) | num1;
                                    break;
                                case 2:
                                    col		= ( num1 << 16 ) | ( b << 8 ) | num3;
                                    break;
                                case 3:
                                    col		= ( num1 << 16 ) | ( num2 << 8 ) | b;
                                    break;
                                case 4:
                                    col		= ( num3 << 16 ) | ( num1 << 8 ) | b;
                                    break;
                                case 5:
                                    col		= ( b << 16 ) | ( num1 << 8 ) | num2;
                                    break;
                                default:
                                    col		= ( b << 16 ) | ( num3 << 8 ) | num1;
                                    break;
                            }
                        }
                        Marshal.WriteInt32( pData, total, col );
                    }
                    break;
                }
                case ColorType.S:
                {
                    int	b		= (int)Math.Round( hsv.V * 2.55d );
                    int	dif		= ( hsv.H * 6 ) % 360;
                    col	= ( b << 16 ) | ( b << 8 ) | b;

                    add	= 100d / (w-1);
                    switch( hsv.H / 60 )
                    {
                        case 1:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	s		= (int)Math.Round( x*add * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num2	= (int)(b * (255 - s * dif / 360d) / 255d);
                                if( s != 0 )
                                {
                                    col		= ( num2 << 16 ) | ( b << 8 ) | num1;
                                }
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                        case 2:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	s		= (int)Math.Round( x*add * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num3	= (int)(b * (255 - s * (360 - dif) / 360d) / 255d);
                                if( s != 0 )
                                {
                                    col		= ( num1 << 16 ) | ( b << 8 ) | num3;
                                }
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                        case 3:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	s		= (int)Math.Round( x*add * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num2	= (int)(b * (255 - s * dif / 360d) / 255d);
                                if( s != 0 )
                                {
                                    col		= ( num1 << 16 ) | ( num2 << 8 ) | b;
                                }
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                        case 4:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	s		= (int)Math.Round( x*add * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num3	= (int)(b * (255 - s * (360 - dif) / 360d) / 255d);
                                if( s != 0 )
                                {
                                    col		= ( num3 << 16 ) | ( num1 << 8 ) | b;
                                }
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                        case 5:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	s		= (int)Math.Round( x*add * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num2	= (int)(b * (255 - s * dif / 360d) / 255d);
                                if( s != 0 )
                                {
                                    col		= ( b << 16 ) | ( num1 << 8 ) | num2;
                                }
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                        default:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	s		= (int)Math.Round( x*add * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num3	= (int)(b * (255 - s * (360 - dif) / 360d) / 255d);
                                if( s != 0 )
                                {
                                    col		= ( b << 16 ) | ( num3 << 8 ) | num1;
                                }
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                    }
                    break;
                }
                case ColorType.V:
                {
                    int	s		= (int)Math.Round( hsv.S * 2.55d );
                    int	dif		= ( hsv.H * 6 ) % 360;

                    add	= 100d / (w-1);
                    switch( hsv.H / 60 )
                    {
                        case 1:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	b		= (int)Math.Round( add*x * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num2	= (int)(b * (255 - s * dif / 360d) / 255d);
                                if( s != 0 )
                                    col		= ( num2 << 16 ) | ( b << 8 ) | num1;
                                else
                                    col	= ( b << 16 ) | ( b << 8 ) | b;

                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                        case 2:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	b		= (int)Math.Round( add*x * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num3	= (int)(b * (255 - s * (360 - dif) / 360d) / 255d);
                                if( s != 0 )
                                    col		= ( num1 << 16 ) | ( b << 8 ) | num3;
                                else
                                    col	= ( b << 16 ) | ( b << 8 ) | b;
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                        case 3:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	b		= (int)Math.Round( add*x * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num2	= (int)(b * (255 - s * dif / 360d) / 255d);
                                if( s != 0 )
                                    col		= ( num1 << 16 ) | ( num2 << 8 ) | b;
                                else
                                    col	= ( b << 16 ) | ( b << 8 ) | b;
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                        case 4:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	b		= (int)Math.Round( add*x * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num3	= (int)(b * (255 - s * (360 - dif) / 360d) / 255d);
                                if( s != 0 )
                                    col		= ( num3 << 16 ) | ( num1 << 8 ) | b;
                                else
                                    col	= ( b << 16 ) | ( b << 8 ) | b;
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                        case 5:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	b		= (int)Math.Round( add*x * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num2	= (int)(b * (255 - s * dif / 360d) / 255d);
                                if( s != 0 )
                                    col		= ( b << 16 ) | ( num1 << 8 ) | num2;
                                else
                                    col	= ( b << 16 ) | ( b << 8 ) | b;
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                        default:
                            for( x = 0; x < w; x ++, total += 4 )
                            {
                                int	b		= (int)Math.Round( add*x * 2.55d );
                                int	num1	= (int)(b * (255 - s) / 255.0d);
                                int	num3	= (int)(b * (255 - s * (360 - dif) / 360d) / 255d);
                                if( s != 0 )
                                    col		= ( b << 16 ) | ( num3 << 8 ) | num1;
                                else
                                    col	= ( b << 16 ) | ( b << 8 ) | b;
                                Marshal.WriteInt32( pData, total, col );
                            }
                            break;
                    }
                    break;
                }
                default:
                {
                    bool	gray	= true;
                    for( x = 0; x < w; x ++, total += 4 )
                    {
                        int	col1	= 0xFFFFFF;
                        int	col2	= (int)Math.Round( 127.5d + x * 255d / (w-1) / 2 );
                        col2	= ( col2 << 16 ) | ( col2 << 8 ) | col2;
                        if( ( x % 5 ) == 0 )
                            gray	= !gray;
                        if( gray )
                        {
                            int		col3	= col2;
                            col2	= col1;
                            col1	= col3;
                        }
                        int		top	= total;
                        for( int y = 0; y < 5; y ++, top += bitmapData.Stride )
                            Marshal.WriteInt32( pData, top, col1 );
                        for( int y = 0; y < 5; y ++, top += bitmapData.Stride )
                            Marshal.WriteInt32( pData, top, col2 );
                        for( int y = 0; y < 5; y ++, top += bitmapData.Stride )
                            Marshal.WriteInt32( pData, top, col1 );
                    }
                    break;
                }
            }
            m_imageBMP.UnlockBits( bitmapData );
        }

        /// <summary>
        /// 指定されたカラーで描画するビットマップとカラーを更新する
        /// </summary>
        /// <param name="color">新しいカラー</param>
        /// <param name="hsv">新しいカラー</param>
        /// <param name="resetPoint">選択している場所をリセットするか</param>
        /// <param name="flag">強制的に画像を書き換える</param>
        public void UpdateColor( Color color, HSV hsv, bool resetPoint, bool flag )
        {
            this.UpdateCurrentColorBitmap( color, hsv, resetPoint, flag );
            if( !m_changePaint )
                this.Invalidate();
            m_changePaint	= true;
        }

    }

    /// <summary>
    /// スライダーのカラー設定。
    /// </summary>
    internal enum ColorType
    {
        /// <summary>RGBのR成分。</summary>
        R,
        /// <summary>RGBのG成分。</summary>
        G,
        /// <summary>RGBのB成分。</summary>
        B,
        /// <summary>HSVのH成分。</summary>
        H,
        /// <summary>HSVのS成分。</summary>
        S,
        /// <summary>HSVのV成分。</summary>
        V,
        /// <summary>A成分。</summary>
        A,
    }
}

