﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;

namespace NW4R.ColorPicker
{
    /// <summary>
    /// HSVカラー
    /// </summary>
    public class HSV
    {
        private int		m_h;
        private int		m_s;
        private int		m_v;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="h">H値（0～360）</param>
        /// <param name="s">S値（0～100）</param>
        /// <param name="v">V値（0～100）</param>
        public HSV( int h, int s, int v )
        {
            m_h	= h;	m_s	= s;	m_v	= v;
        }

        /// <summary>
        /// 比較演算子
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            bool	result	= false;
            if( obj is HSV )
            {
                HSV	tmp	= (HSV)obj;
                if( tmp.H == this.H && tmp.S == this.S && tmp.V == this.V )
                    result	= true;
            }
            return result;
        }

        /// <summary>
        /// GetHashCodeのオーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode ();
        }

        /// <summary>
        /// RGBからHSVを作成
        /// </summary>
        /// <param name="rgb">RGB値（各0～255）</param>
        /// <returns>HSV</returns>
        public static HSV FromRGB( RGB rgb )
        {
            int		r	= rgb.R;
            int		g	= rgb.G;
            int		b	= rgb.B;

            int		minValue	= Math.Min( Math.Min( r, g ), b );
            int		maxValue	= Math.Max( Math.Max( r, g ), b );
            int		delta		= maxValue - minValue;

            double	hh	= 0;
            int		ss	= 0;
            int		vv	= (int)Math.Round( maxValue / 2.55d );

            if( maxValue != 0 && delta != 0 )
            {
                if( minValue == 0 )
                    ss	= 100;
                else
                    ss	= (int)Math.Round( (double)delta / maxValue * 100 );
                if( r == maxValue )
                    hh	= (double)( g - b ) / delta ;
                else if( g == maxValue )
                    hh	= 2 + (double)( b - r ) / delta;
                else if( b == maxValue )
                    hh	= 4 + (double)( r - g ) / delta;
            }

            hh	*= 60;
            if( hh < 0 )
                hh	+= 360;

            return new HSV( (int)Math.Round( hh ), ss, vv );
        }

        /// <summary>
        /// HSVカラーH要素
        /// </summary>
        public int H
        {
            get	{	return m_h;		}
            set	{	m_h	= value;	}
        }
        /// <summary>
        /// HSVカラーS要素
        /// </summary>
        public int S
        {
            get	{	return m_s;		}
            set	{	m_s	= value;	}
        }
        /// <summary>
        /// HSVカラーV要素
        /// </summary>
        public int V
        {
            get	{	return m_v;		}
            set	{	m_v	= value;	}
        }
    }
    /// <summary>
    /// RGBカラー
    /// </summary>
    public class RGB
    {
        private int		m_r;
        private int		m_g;
        private int		m_b;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="r">R値（0～255）</param>
        /// <param name="g">G値（0～255）</param>
        /// <param name="b">B値（0～255）</param>
        public RGB( int r, int g, int b )
        {
            m_r	= r;	m_g	= g;	m_b	= b;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="col">Color</param>
        public RGB( Color col )
        {
            m_r	= col.R;	m_g	= col.G;	m_b	= col.B;
        }

        /// <summary>
        /// 比較演算子
        /// </summary>
        /// <param name="obj"></param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            bool	result	= false;
            if( obj is RGB )
            {
                RGB	tmp	= (RGB)obj;
                if( tmp.R == this.R && tmp.G == this.G && tmp.B == this.B )
                    result	= true;
            }
            return result;
        }

        /// <summary>
        /// GetHashCodeのオーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return base.GetHashCode ();
        }

        /// <summary>
        /// HSVからRGBを作成
        /// </summary>
        /// <param name="hsb">HSV値（H:0～360,S:0～100,V:0～100）</param>
        /// <returns>RGB</returns>
        public static RGB FromHSV( HSV hsb )
        {
            int	h		= hsb.H;
            int	s		= (int)Math.Round( hsb.S * 2.55d );
            int	b		= (int)Math.Round( hsb.V * 2.55d );
            if( s == 0 )
            {
                return new RGB( b, b, b );
            }
            int	dif		= ( h * 6 ) % 360;
            int	num1	= (int)(b * (255 - s) / 255.0d);
            int	num2	= (int)(b * (255 - s * dif / 360d) / 255d);
            int	num3	= (int)(b * (255 - s * (360 - dif) / 360d) / 255d);

            switch( h / 60 )
            {
                case 1:
                    return new RGB( num2, b, num1 );
                case 2:
                    return new RGB( num1, b, num3 );
                case 3:
                    return new RGB( num1, num2, b );
                case 4:
                    return new RGB( num3, num1, b );
                case 5:
                    return new RGB( b, num1, num2 );
                default:
                    return new RGB( b, num3, num1 );
            }
        }

        /// <summary>
        /// RGBカラーR要素
        /// </summary>
        public int R
        {
            get	{	return m_r;		}
            set	{	m_r	= value;	}
        }
        /// <summary>
        /// RGBカラーG要素
        /// </summary>
        public int G
        {
            get	{	return m_g;		}
            set	{	m_g	= value;	}
        }
        /// <summary>
        /// RGBカラーB要素
        /// </summary>
        public int B
        {
            get	{	return m_b;		}
            set	{	m_b	= value;	}
        }
        /// <summary>
        /// RGBカラーからColor型を取得（αは255）
        /// </summary>
        /// <returns></returns>
        public Color GetColor()
        {
            return Color.FromArgb( m_r, m_g, m_b );
        }
    }
}
