﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using NintendoWare.Font.Win32;
    using wchar_t = System.UInt16;

    public partial class GlCm
    {
        private const uint CodePageUnicode = 0xFFFFFFFF;

        // FontEncoding の順序と一致させる
        private static readonly uint[] EncodingToCodePage =
        {
            CodePageUnicode,     // UCS-2
            CodePageUnicode,     // UCS-2
            932,            // Shift_JIS
            1252,           // CP1252
            949,            // CP949
        };

        public static uint UnicodeToLocal(uint uni)
        {
            return UnicodeTo(CodePage.ACP, uni);
        }

        public static uint LocalToUnicode(uint local)
        {
            return ToUnicode(CodePage.ACP, local);
        }

        public static uint EncodingToUnicode(CharEncoding enc, uint c)
        {
            uint cp = EncodingToCodePage[(int)enc];
            return (cp == CodePageUnicode) ? c : ToUnicode(cp, c);
        }

        public static uint UnicodeToEncoding(CharEncoding enc, uint uni)
        {
            uint cp = EncodingToCodePage[(int)enc];
            return (cp == CodePageUnicode) ? uni : UnicodeTo(cp, uni);
        }

        //-------------------------------------
        // Unicode -> Local Code
        private static uint UnicodeTo(uint cp, uint uni)
        {
            uint local_code;
            byte[] buf = new byte[2];
            int len;
            bool isFailed;

            len = Kernel.WideCharToMultiByte(
                cp,
                WideChar.NoBestFitChars,
                new ushort[] { (ushort)uni },
                1,
                buf,
                buf.Length,
                null,
                out isFailed);

            if (isFailed)
            {
                return Runtime.RtConsts.InvalidCharCode;
            }

            switch (len)
            {
            case 1:
                local_code = buf[0];
                break;

            case 2:
                local_code = (uint)((buf[0] << 8) | buf[1]);
                break;

            case -1:
            default:
                local_code = Runtime.RtConsts.InvalidCharCode;
                break;
            }

            return local_code;
        }

        //-------------------------------------
        // Local Code -> Unicode
        private static uint ToUnicode(uint cp, uint local)
        {
            byte[] local_code = new byte[2];
            int local_len;
            int uni_len;
            ushort[] unicode = new ushort[1];

            if (local < 0x100)
            {
                local_code[0] = (byte)local;
                local_len = 1;
            }
            else
            {
                local_code[0] = (byte)(local >> 8);
                local_code[1] = (byte)local;
                local_len = 2;
            }

            uni_len = Kernel.MultiByteToWideChar(
                cp,
                MBChar.ErrInvalidChars,
                local_code,
                local_len,
                unicode,
                unicode.Length);

            if (uni_len != 1)
            {
                return Runtime.RtConsts.InvalidCharCode;
            }

            return (uint)unicode[0];
        }
    }
}
