﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    /// <summary>
    /// グリフデータタイプです。
    /// </summary>
    public enum GlyphDataType
    {
        /// <summary>
        /// BMP
        /// </summary>
        Glyph,

        /// <summary>
        /// 実機ネイティブテクスチャフォーマット
        /// ※ 当然ながら RVL, CTRではテクスチャフォーマットのビットイメージは異なります。
        /// </summary>
        Texture,

        /// <summary>
        /// 0 - 127 にアルファ、128 - 255 に 明度をパックした形式。シェーダ側で特殊な解釈を行って利用します。
        /// </summary>
        PackedTexture,
    }

    /// <summary>
    /// 文字列エンコーディングタイプです。
    /// </summary>
    public enum CharEncoding
    {
        UTF8,         // UTF-8
        UTF16,        // UTF-16
        SJIS,         // ShiftJIS
        CP1252,       // CP1252

        NUM
    }

    /// <summary>
    /// 文字コードマッピングタイプです。
    /// </summary>
    public enum FontMapMethod
    {
        Direct,      // インデックス = 文字コード - オフセット
        Table,       // インデックス = mapInfo[文字コード - オフセット]
        Scan,        // インデックス = search(mapInfo, 文字コード)
    }

    /// <summary>
    /// シートの形式 (テクスチャフォーマット)です。
    /// </summary>
    public enum FontSheetFormat
    {
        TexFmtMask = 0x3FFF,  // GXTexFmt
        LinearFormatFlag = 0x4000, // リニアなのでタイリング解除の必要がない
        CompressedFlag = 0x8000   // 1 なら圧縮されている
    }
}

namespace NintendoWare.Font.Runtime
{
    using System;
    using System.Diagnostics.CodeAnalysis;
    using System.Runtime.InteropServices;
    using RtPtr = System.UInt32;

    /// <summary>
    /// 文字の各種幅を保持する構造体です。
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 1)]
    public struct CharWidths
    {
        public const int Length = 3;

        /// <summary>
        /// 文字の左スペースの幅
        /// </summary>
        public sbyte Left;

        /// <summary>
        /// 文字のグリフ幅
        /// </summary>
        public byte GlyphWidth;

        /// <summary>
        /// 文字の幅 = 左スペース幅 + グリフ幅 + 右スペース幅
        /// </summary>
        public byte CharWidth;

        public CharWidths(sbyte left, byte glyphWidth, byte charWidth)
        {
            this.Left = left;
            this.GlyphWidth = glyphWidth;
            this.CharWidth = charWidth;
        }
    }

    /// <summary>
    /// 文字コードとグリフインデックスペア
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct CMapScanEntry
    {
        public static readonly int Length = Marshal.SizeOf(typeof(CMapScanEntry));

        /// <summary>
        /// 文字コード
        /// </summary>
        public uint Code;

        /// <summary>
        /// グリフインデックス
        /// </summary>
        public ushort Index;

        public ushort Padding;
    }

    /// <summary>
    /// MAPMETHOD_SCAN の場合の FontCodeMap.mapInfo
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct CMapInfoScan
    {
        public static readonly int Length = Marshal.SizeOf(typeof(CMapInfoScan));

        /// <summary>
        /// entries の要素数
        /// </summary>
        public ushort Num;

        public ushort Padding;
    }

    public partial class RtConsts
    {
        public static readonly string PackedFontFormatSuffix = "Packed";
        /* ------------------------------------------------------------------------
                シグネチャ定義
           ------------------------------------------------------------------------ */

        public static readonly Signature32 BinFileSigFONTCTR = new Signature32("CFNT");
        public static readonly Signature32 BinFileSigFONTACTR = new Signature32("CFNT");

        /// <summary>
        /// 通常フォントリソースファイルシグネチャ
        /// </summary>
        public static readonly Signature32 BinFileSigFONT = new Signature32(ConverterEnvironment.PlatformChar + "FNT");

        // アーカイブフォントリソースファイルシグネチャ
        public static readonly Signature32 BinFileSigFONTA = new Signature32(ConverterEnvironment.PlatformChar + "FNA");

        // バイナリブロックシグネチャ
        public static readonly Signature32 BinBlockSigFINF = new Signature32("FINF");
        public static readonly Signature32 BinBlockSigCGLP = new Signature32("CGLP");
        public static readonly Signature32 BinBlockSigTGLP = new Signature32("TGLP");
        public static readonly Signature32 BinBlockSigCWDH = new Signature32("CWDH");
        public static readonly Signature32 BinBlockSigCMAP = new Signature32("CMAP");
        public static readonly Signature32 BinBlockSigKRNG = new Signature32("KRNG");

        public static readonly Signature32 BinBlockSigGLGR = new Signature32("GLGR");
        public static readonly Signature32 BinBlockSigHTGL = new Signature32("HTGL");

        /* ------------------------------------------------------------------------
                リリースバージョン定義
           ------------------------------------------------------------------------ */
        // IPlatformTextureDetails から取得するように変更しました。

        /* ------------------------------------------------------------------------
                定数定義
           ------------------------------------------------------------------------ */
        public const uint InvalidCharCode = 0xFFFF;
        public const ushort InvalidGlyphIndex = 0xFFFF;
    }

    /* ------------------------------------------------------------------------
            ブロック本体定義
       ------------------------------------------------------------------------ */

    /// <summary>
    /// フォントを部分ロードするための情報を格納します。
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 2)]
    [SuppressMessage("Microsoft.StyleCop.CSharp.MaintainabilityRules", "SA1401:FieldsMustBePrivate", Justification = "Binay Image")]
    public class FontGlyphGroups
    {
        public static readonly int Length = Marshal.SizeOf(typeof(FontGlyphGroups));

        public uint SheetSize;

        public ushort GlyphsPerSheet;

        public ushort NumSet;

        public ushort NumSheet;

        public ushort NumCWDH;

        public ushort NumCMAP;

        //// ushort nameOffsets[];  // numSet
        /*
            以下は可変長データが続きます

            (4 byte align)

            uint sizeSheets[numSheet];       // 圧縮済みシートのサイズ
            uint sizeCWDH[numCWDH];          // CWDH ブロックのサイズ
            uint sizeCMAP[numCMAP];          // CMAP ブロックのサイズ
            uint useSheets[numSet][numSheet/32];
            uint useCWDH[numSet][numCWDH/32];
            uint useCMAP[numSet][numCMAP/32];

            char names[numSet][name length];
        */
    }

    /// <summary>
    /// フォントのグリフテクスチャを格納します。
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    [SuppressMessage("Microsoft.StyleCop.CSharp.MaintainabilityRules", "SA1401:FieldsMustBePrivate", Justification = "Binay Image")]
    public class FontTextureGlyph
    {
        public static readonly int Length = Marshal.SizeOf(typeof(FontTextureGlyph));

        /// <summary>
        /// セル幅=最大グリフ幅
        /// </summary>
        public byte CellWidth;

        /// <summary>
        /// セル高さ
        /// </summary>
        public byte CellHeight;

        /// <summary>
        /// シート数
        /// </summary>
        public byte SheetNum;

        /// <summary>
        /// 最大文字幅
        /// </summary>
        public byte MaxCharWidth;

        /// <summary>
        /// テクスチャシートデータサイズ(byte単位)
        /// </summary>
        public uint SheetSize;

        /// <summary>
        /// ベースライン位置
        /// </summary>
        public short BaselinePos;

        /// <summary>
        /// FontSheetFormat
        /// </summary>
        public ushort SheetFormat;

        /// <summary>
        /// シート内の横方向セル数
        /// </summary>
        public ushort SheetRow;

        /// <summary>
        /// シート内の縦方向セル数
        /// </summary>
        public ushort SheetLine;

        /// <summary>
        /// シート幅(ピクセル単位)
        /// </summary>
        public ushort SheetWidth;

        /// <summary>
        /// シート高さ(ピクセル単位)
        /// </summary>
        public ushort SheetHeight;

        /// <summary>
        /// テクスチャデータへのポインタ(byte*)
        /// </summary>
        public RtPtr SheetImage;
    }

    /// <summary>
    /// 各文字の文字幅情報を格納します。
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    [SuppressMessage("Microsoft.StyleCop.CSharp.MaintainabilityRules", "SA1401:FieldsMustBePrivate", Justification = "Binay Image")]
    public class FontWidth
    {
        public static readonly int Length = Marshal.SizeOf(typeof(FontWidth));

        /// <summary>
        /// widthTable の最初のエントリが対応するグリフインデックス
        /// </summary>
        public ushort IndexBegin;

        /// <summary>
        /// widthTable の最後のエントリが対応するグリフインデックス
        /// </summary>
        public ushort IndexEnd;

        /// <summary>
        /// 次の FontWidth へのポインタ(FontWidth*)
        /// </summary>
        public RtPtr PtrNext;

        //// CharWidths          widthTable[];   // 幅情報の配列
    }

    /// <summary>
    /// 文字コードから、文字コードに対応するグリフの
    /// グリフイメージ配列中のインデックスへのマッピングを規定します。
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    [SuppressMessage("Microsoft.StyleCop.CSharp.MaintainabilityRules", "SA1401:FieldsMustBePrivate", Justification = "Binay Image")]
    public class FontCodeMap
    {
        public static readonly int Length = Marshal.SizeOf(typeof(FontCodeMap));

        /// <summary>
        /// このブロックが担当する文字コード範囲の最初の文字コード
        /// </summary>
        public uint CodeBegin;

        /// <summary>
        /// このブロックが担当する文字コード範囲の最後の文字コード
        /// </summary>
        public uint CodeEnd;

        /// <summary>
        /// マッピング方法 (FontMappingMethod型)
        /// </summary>
        public ushort MappingMethod;

        /// <summary>
        /// 予約
        /// </summary>
        public ushort Reserved;

        /// <summary>
        /// 次の FontCodeMap へのポインタ(FontCodeMap*)
        /// </summary>
        public RtPtr PtrNext;

        //// ushort                 mapInfo[];      // 文字コードマッピング情報 具体的な内容は mappingMethod に依る
    }

    /// <summary>
    /// フォント全体に渡る情報を格納します。
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    [SuppressMessage("Microsoft.StyleCop.CSharp.MaintainabilityRules", "SA1401:FieldsMustBePrivate", Justification = "Binay Image")]
    public class FontInformation
    {
        public static readonly int Length = Marshal.SizeOf(typeof(FontInformation));

        /// <summary>
        /// グリフデータタイプ (FontType型)
        /// </summary>
        public byte FontType;

        /// <summary>
        /// フォントの高さ
        /// </summary>
        public byte Height;

        /// <summary>
        /// フォントの幅
        /// </summary>
        public byte Width;

        /// <summary>
        /// アセント
        /// </summary>
        public byte Ascent;

        /// <summary>
        /// (*)== leading
        /// </summary>
        public short Linefeed;

        /// <summary>
        /// (*)フォントに含まれない文字用のグリフのグリフインデックス
        /// </summary>
        public ushort AlterCharIndex;

        /// <summary>
        /// (*)文字幅情報を持たないグリフ用の文字幅情報
        /// </summary>
        public CharWidths DefaultWidth;

        /// <summary>
        /// 対応する文字列エンコーディング (FontEncoding型)
        /// </summary>
        public byte Encoding;

        /// <summary>
        /// 唯一の FontGlyph へのポインタ(FontTextureGlyph*)
        /// </summary>
        public RtPtr PtrGlyph;

        /// <summary>
        /// 最初の FontWidth へのポインタ(FontWidth*)
        /// </summary>
        public RtPtr PtrWidth;

        /// <summary>
        /// 最初の FontCodeMap へのポインタ(FontCodeMap*)
        /// </summary>
        public RtPtr PtrMap;
    }
}
