﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.IO;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Windows.Media;
    using NintendoWare.Font.Win32;
    using WaterTrans.TypeLoader;
    using Alpha = System.Byte;
    using HDC = System.IntPtr;
    using HFONT = System.IntPtr;
    using System.Windows;
    using System.Linq;
    using System.Runtime.InteropServices;

    public enum WidthType
    {
        GlyphOnly,
        GlyphOnlyKeepSpace,
        IncludeMargin,
        Fixed,

        NUM
    }

    public abstract class Letterer : IDisposable
    {
        private const int Margin = 1;
        private const int MarginLeft = Margin;
        private const int MarginRight = Margin;
        private const int MarginTop = Margin;
        private const int MarginBottom = Margin;
        private const int WidthSpWidth = Margin;
        private const int WidthHeight = Margin;

        private bool isUnicode;

        private WidthType widthType;

        private int bpp;

        private int fixedWidth;

        private int baseline;

        private int cellSize;

        private int fontHeight;

        private HDC deviceContext;

        private HFONT oldFont;

        private string fontFaceName;
        private string fontFacePath;
        private bool useScfontKerning;

        public Letterer()
        {
            this.isUnicode = false;
            this.bpp = 0;
            this.deviceContext = IntPtr.Zero;
            this.oldFont = IntPtr.Zero;
            this.fontHeight = 0;
        }

        protected HDC DeviceContext
        {
            get { return this.deviceContext; }
            set { this.deviceContext = value; }
        }

        protected HFONT OldFont
        {
            get { return this.oldFont; }
        }

        protected bool IsUnicode
        {
            get { return this.isUnicode; }
        }

        protected int Bpp
        {
            get { return this.bpp; }
        }

        protected int Baseline
        {
            get { return this.baseline; }
        }

        protected int CellSize
        {
            get { return this.cellSize; }
        }

        public void Dispose()
        {
            this.Dispose(true);
        }

        public abstract void Create(ref LOGFONT logfont, string fontPath, string inputFontFaceName);

        public abstract bool LetterChar(Glyph glyph, uint c);

        public int GetFontHeight()
        {
            return this.fontHeight;
        }

        public int GetAscent()
        {
            return this.baseline;
        }

        public void SetOption(bool isUnicode, int bpp, bool hasAlpha, WidthType widthType, int fixedWidth, bool useScfontKerning)
        {
            if (bpp < 1 || 8 < bpp)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_OUTOFRANGE_COLORBPP, bpp);
            }

            if (widthType < 0 || WidthType.NUM <= widthType)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_INVALID_WIDTH_TYPE, widthType);
            }

            this.isUnicode = isUnicode;
            this.bpp = (bpp == 1) ? 1 : ((bpp <= 4) ? 4 : 8);    // TODO
            this.widthType = widthType;
            this.fixedWidth = fixedWidth;
            this.useScfontKerning = useScfontKerning;
        }

        public KerningPair[] GetKerningPairs()
        {
            // カーニング要素数が返る。 カーニングがない場合は 0 が返る。
            uint numPairs = Gdi.GetKerningPairsW(DeviceContext, 0, null);
            if (numPairs != 0)
            {
                KerningPair[] pairs = new KerningPair[numPairs];
                {
                    KERNINGPAIR[] gdiKerningPair = new KERNINGPAIR[numPairs];
                    Gdi.GetKerningPairsW(DeviceContext, numPairs, gdiKerningPair);
                    for (int i = 0; i < numPairs; i++)
                    {
                        pairs[i].First = (uint)gdiKerningPair[i].wFirst;
                        pairs[i].Second = (uint)gdiKerningPair[i].wSecond;
                        pairs[i].KernAmount = gdiKerningPair[i].iKernAmount;
                    }
                }

                // 指定があれば、MonoTypeエンジンのカーニングで上書きする。
                if (this.useScfontKerning)
                {
                    MonoTypeHelper.FixKerningByMonoTypeKerning(fontFacePath, fontFaceName, fontHeight, pairs);
                }

                List<KerningPair> pairs2 = new List<KerningPair>(pairs);
                pairs2.RemoveAll((pair) => pair.KernAmount == 0);

                // OutLogFile_(pairs2.ToArray(), @"C:\Users\N1983\Desktop\GetKerningPairsW_1.txt");
                // OutLogFile_(this.GetKerningPairsFromGPOS_(), @"C:\Users\N1983\Desktop\GetKerningPairsW_2.txt");

                return pairs2.ToArray();
            }
            else
            {
                KerningPair[] pairs = this.GetKerningPairsFromGPOS_();

                // 指定があれば、MonoTypeエンジンのカーニングで上書きする。
                // memo : 手元でチェックしたマリオフォントの
                // GPOS カーニング値と、MonoTypeエンジンのカーニングは完全に一致していた。
                // GPOS カーニング を持つフォントでは、useScfontKerning は不要かもしれない。
                //
                // 上記から「GPOS カーニングを優先的に利用する」というオプションを用意して
                // useScfontKerning の代わりにできないかと考えたがこれはダメな模様。
                // (フォントによっては、GPOS を持いないが、iTpyeではカーニングが取れ、なおかつGetKerningPairsWと差異がある場合がある)
                if (this.useScfontKerning)
                {
                    MonoTypeHelper.FixKerningByMonoTypeKerning(fontFacePath, fontFaceName, fontHeight, pairs);
                }

                List<KerningPair> pairs2 = new List<KerningPair>(pairs);
                pairs2.RemoveAll((pair) => pair.KernAmount == 0);

                // OutLogFile_(pairs2.ToArray(), @"C:\Users\N1983\Desktop\GetKerningPairsW_2.txt");

                return pairs2.ToArray();
            }
        }

        /// <summary>
        /// デバッグ用のログ出力
        /// </summary>
        private static void OutLogFile_(KerningPair[] pairs, string filePath)
        {
            String content = "";
            foreach (var pair in pairs)
            {
                content += string.Format("[{0}({1:x4}), {2}({3:x4})] = {4}\n", (char)pair.First, pair.First, (char)pair.Second, pair.Second, pair.KernAmount);
            }

            File.WriteAllText(filePath, content);
        }

        /// <summary>
        /// GPOS テーブルから カーニングを取る
        /// </summary>
        private KerningPair[] GetKerningPairsFromGPOS_()
        {
            Typeface typeface = new Typeface(this.fontFaceName);
            GlyphTypeface glyphTypeface;
            if (!typeface.TryGetGlyphTypeface(out glyphTypeface))
            {
                return new KerningPair[0];
            }

            TypefaceInfo info = new TypefaceInfo(glyphTypeface.GetFontStream());
            PairAdjustmentMetrics kern = info.GetKerningAdjustmentMetrics();

            // 有効なカーニング情報が得られないなら中断
            if (kern == null || kern.GlyphIndexPairs.Count() <= 0)
            {
                return new KerningPair[0];
            }

            // グリフインデックス => 文字コードのテーブルを準備する。
            Dictionary<ushort, uint> glyphMapToCharacter = new Dictionary<ushort, uint>();
            foreach (var charCode in glyphTypeface.CharacterToGlyphMap.Keys)
            {
                ushort glyphIndex = glyphTypeface.CharacterToGlyphMap[charCode];
                if (!glyphMapToCharacter.ContainsKey(glyphIndex))
                {
                    glyphMapToCharacter.Add(glyphIndex, (uint)charCode);
                }
            }

            int noCode = 0;
            List<KerningPair> pairs = new List<KerningPair>();
            foreach (KeyValuePair<ushort, ushort> glyphIndexPair in kern.GlyphIndexPairs)
            {
                KerningPair pair = new KerningPair();

                // 左→右方向の文字では、
                // 一文字目の右端位置を調整するという考え方なので、FirstAdjustmentを使う。
                // http://forum.fontlab.com/opentype-layout-feature-development/screwed-arabic-gposkerning-in-indesign-me-cs3-t349.0.html
                var kern1st = kern.GetFirstAdjustment(glyphIndexPair.Key, glyphIndexPair.Value);

                // 四捨五入で整数値に変換する（やらないと GetKerningPairs() の値と大きく異なってしまいます）。
                // 生値をダンプするときは XAdvanceRaw などを代入するようにします。
                pair.KernAmount = (int)Math.Floor(kern1st.XAdvance * fontHeight + 0.5);

                uint code;
                if(!glyphMapToCharacter.TryGetValue(glyphIndexPair.Key, out code))
                {
                    noCode++;
                    continue;
                }
                pair.First = code;

                if (!glyphMapToCharacter.TryGetValue(glyphIndexPair.Value, out code))
                {
                    noCode++;
                    continue;
                }
                pair.Second = code;

                pairs.Add(pair);
            }

            pairs.Sort((l, r) => (l.First == r.First) ? (int)l.Second - (int)r.Second : (int)l.First - (int)r.First);
            return pairs.ToArray();
        }

        protected bool GetCharABCWidthsU(HDC hDC, uint first, uint last, ABC[] pabc)
        {
            return this.isUnicode ?
                Gdi.GetCharABCWidthsW(hDC, first, last, pabc) :
                Gdi.GetCharABCWidthsA(hDC, first, last, pabc);
        }

        protected bool GetCharWidth32U(HDC hDC, uint first, uint last, int[] pw)
        {
            return this.isUnicode ?
                Gdi.GetCharWidth32W(hDC, first, last, pw) :
                Gdi.GetCharWidth32A(hDC, first, last, pw);
        }

        protected abstract void Dispose(bool disposing);

        protected void CalcurateParameter()
        {
            // calculate metric info
            {
                TEXTMETRIC tm = new TEXTMETRIC();

                var isSuccess = Gdi.GetTextMetrics(this.deviceContext, tm);
                if (!isSuccess)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_GETTEXTMETRICS);
                }

                this.cellSize = Math.Max(tm.tmMaxCharWidth, tm.tmHeight);
                this.baseline = tm.tmAscent - tm.tmInternalLeading;

                // this.fontHeight == EmSize = CellHeight - tmInternalLeading
                // lf.lfHeight が正なら、= CellHeight.   CellHeight - tmInternalLeading  が em になる
                // 参照：http://support.microsoft.com/kb/32667/en-us
                this.fontHeight = tm.tmHeight - tm.tmInternalLeading;
            }
        }

        protected void SetupFont(ref LOGFONT logFont, string fontPath, string inputFontFaceName)
        {
            Debug.Assert(this.oldFont == IntPtr.Zero);
            Debug.Assert(this.deviceContext != IntPtr.Zero);

            var font = Gdi.CreateFontIndirect(ref logFont);
            if (font == IntPtr.Zero)
            {
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_CREATEFONT);
            }

            this.oldFont = Gdi.SelectFont(this.deviceContext, font);
            if (this.oldFont == IntPtr.Zero)
            {
                Gdi.DeleteFont(font);
                throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_SELECTOBJECT_NEWFONT);
            }

            if (inputFontFaceName != null)
            {
                // フォントフェース名がオプションで直接指定された場合は、
                // 期待したフォントがロードされているかどうかを検証する
                uint cbBuffer = Gdi.GetOutlineTextMetricsA(this.deviceContext, 0, IntPtr.Zero);
                if (cbBuffer == 0)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_MATCH_FONT_FACE_NAME, inputFontFaceName);
                }
                IntPtr buffer = Marshal.AllocHGlobal((int)cbBuffer);
                if (Gdi.GetOutlineTextMetricsA(this.deviceContext, cbBuffer, buffer) == 0)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_MATCH_FONT_FACE_NAME, inputFontFaceName);
                }
                OUTLINETEXTMETRIC metric = (OUTLINETEXTMETRIC)Marshal.PtrToStructure(buffer, typeof(OUTLINETEXTMETRIC));
                string metricFaceName = Marshal.PtrToStringAnsi(new IntPtr((long)buffer + (long)metric.otmpFaceName));
                if (inputFontFaceName != metricFaceName)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_MATCH_FONT_FACE_NAME, inputFontFaceName);
                }
            }

            Gdi.SetTextAlign(this.deviceContext, GdiDef.TA_LEFT | GdiDef.TA_TOP | GdiDef.TA_NOUPDATECP);
            Gdi.SetMapMode(this.deviceContext, GdiDef.MM_TEXT);

            fontFaceName = logFont.lfFaceName;
            fontFacePath = fontPath;
        }

        protected void CleanupFont()
        {
            Debug.Assert(this.oldFont != IntPtr.Zero);
            Debug.Assert(this.deviceContext != IntPtr.Zero);

            if (this.oldFont != IntPtr.Zero)
            {
                var font = Gdi.SelectFont(this.deviceContext, this.oldFont);
                if (font == IntPtr.Zero)
                {
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_FAIL_SELECTOBJECT_OLDFONT);
                }

                Gdi.DeleteFont(font);
            }
        }

        protected bool IsGlyphExists(uint c)
        {
            uint ret;
            var index = new ushort[2] { 0, 0 };

            if (this.isUnicode)
            {
                var buf = new ushort[] { (ushort)c };
                ret = Gdi.GetGlyphIndicesW(this.deviceContext, buf, 1, index, GdiDef.GGI_MARK_NONEXISTING_GLYPHS);
                return index[0] != Runtime.RtConsts.InvalidGlyphIndex;
            }
            else
            {
                var buf = new byte[2];
                int len;

                if (c < 0x100)
                {
                    buf[0] = (byte)c;
                    len = 1;
                }
                else
                {
                    buf[0] = (byte)(c >> 8);
                    buf[1] = (byte)(c & 0xFF);
                    len = 2;
                }

                ret = Gdi.GetGlyphIndicesA(this.deviceContext, buf, len, index, GdiDef.GGI_MARK_NONEXISTING_GLYPHS);

                return (ret == 1) && (index[0] != Runtime.RtConsts.InvalidGlyphIndex);
            }
        }

        protected void MakeAlpha(IndexImage image, int bpp)
        {
            int maxIdx = (1 << bpp) - 1;

            image.EnableAlpha();
            for (int y = 0; y < image.Height; y++)
            {
                for (int x = 0; x < image.Width; x++)
                {
                    uint idx = image.GetColor(x, y);
                    Alpha a = (byte)(idx * 255 / maxIdx);

                    image.SetAlpha(x, y, a);
                }
            }
        }

        // image: topdown
        protected void SetGlyph(Glyph glyph, IndexImage image, GlyphInfo info)
        {
            RECT trueGlyphRect = new RECT();
            IndexImage lastimage = new IndexImage();
            RgbImage rgbImage = new RgbImage();

            // グリフ領域を測定
            image.ScanOpacityRect(ref trueGlyphRect);

            int width = trueGlyphRect.right - trueGlyphRect.left;
            int height = trueGlyphRect.bottom - trueGlyphRect.top;
            int bline = info.GlyphAscent - trueGlyphRect.top;
            int left, right;

            // グリフ領域を抽出
            image.Extract(lastimage, trueGlyphRect.left, trueGlyphRect.top, width, height);
            lastimage.SetColorTable(image.GetColorTable(), image.GetColorTableEntryNum());

            /*
                |---- info.charWidth----|
                |info.glyphLeft
                |--|
                   ------**********---|   image
                   |     |- width -|
                   |     tGR.left  tGR.right
                   |
                   |- info.bmpWidth --|
            */

            // 幅を計算
            {
                switch (this.widthType)
                {
                case WidthType.GlyphOnly:
                    left = 0;
                    right = 0;
                    break;

                case WidthType.GlyphOnlyKeepSpace:
                    if (width > 0)
                    {
                        // グリフがある場合
                        left = 0;
                        right = 0;
                    }
                    else
                    {
                        // グリフがない場合
                        left = info.CharWidth;
                        right = 0;
                    }

                    break;

                case WidthType.IncludeMargin:
                    if (width > 0)
                    {
                        // グリフがある場合
                        left = info.GlyphLeft + trueGlyphRect.left;
                        right = info.CharWidth - width - left;
                    }
                    else
                    {
                        // グリフがない場合
                        left = info.CharWidth;
                        right = 0;
                    }

                    break;

                case WidthType.Fixed:
                    if (width > 0)
                    {
                        // グリフがある場合
                        left = (this.fixedWidth - width) / 2;
                        right = this.fixedWidth - left - width;
                    }
                    else
                    {
                        // グリフがない場合
                        left = this.fixedWidth;
                        right = 0;
                    }

                    break;

                default:
                    throw GlCm.ErrMsg(ErrorType.Internal, Strings.IDS_ERR_INVALID_WIDTH_TYPE);
                }
            }
            //// _RPT2(_CRT_WARN, "LETTER: M %d %d\n", left, right);

            // index -> RGB
            rgbImage.Create(lastimage.Width, lastimage.Height, 32);
            rgbImage.Clear(ImageBase.RgbBlack, ImageBase.AlphaTransparent);
            rgbImage.EnableAlpha();

            for (int y = 0; y < lastimage.Height; y++)
            {
                for (int x = 0; x < lastimage.Width; x++)
                {
                    uint c = lastimage.GetRGB(x, y);
                    Alpha a = lastimage.GetAlpha(x, y);
                    //// const uint          c = (a > 0) ? IImage::RGB_BLACK: IImage::RGB_WHITE;
                    //// rgbImage.SetAlpha(x+1, y+1, a);
                    rgbImage.SetColorAlpha(x, y, c, a);
                }
            }

            // グリフに設定
            // buf: topdown
            IntColor nullColor = 0;
            glyph.SetGlyphImage(rgbImage, left, right, bline, nullColor);
        }

        protected class GlyphInfo
        {
            public int BmpWidth { get; set; }

            public int BmpHeight { get; set; }

            public int GlyphLeft { get; set; }

            public int CharWidth { get; set; }

            public int GlyphAscent { get; set; }

            public byte Black { get; set; }

            //// byte bg;

            public byte White { get; set; }
        }
    }
}
