﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Reflection;
    using Alpha = System.Byte;

    public class ImageFontWriter : FontWriter
    {
        // ImageFileFormat と同じ順序でなければならない
        private readonly string[] FileExtension =
        {
            null,
            "bmp",
            "tga",
            "tif",
        };

        private readonly string File;
        private readonly string Order;
        private readonly ImageFileFormat Format;
        private readonly bool IsDrawGrid;
        private readonly bool IsOutKerningData;

        private readonly int CellMarginLeft;
        private readonly int CellMarginRight;
        private readonly int CellMarginTop;
        private readonly int CellMarginBottom;

        /// <summary>
        /// グリッド色    DrawGrid, DrawArrangeInfo
        /// </summary>
        private readonly uint colorGrid;

        /// <summary>
        /// 余白領域色    DrawGrid
        /// </summary>
        private readonly uint colorMargin;

        /// <summary>
        /// 幅線色        DrawOneGlyph
        /// </summary>
        private readonly uint colorWidthBar;

        /// <summary>
        /// nullセル色 DrawNullBlock
        /// </summary>
        private readonly uint colorNullCell;

        /// <summary>
        /// 背景色
        /// </summary>
        private readonly uint colorBackground;

        /// <summary>
        /// セル内でのグリフ領域配置位置のオフセット
        /// </summary>
        private int glpOffsetX;

        /// <summary>
        /// セル内でのグリフ領域配置位置のオフセット
        /// </summary>
        private int glpOffsetY;

        /// <summary>
        /// グリフ領域の幅
        /// </summary>
        private int glpWidth;

        /// <summary>
        /// グリフ領域の高さ
        /// </summary>
        private int glpHeight;

        /// <summary>
        /// セルの幅
        /// </summary>
        private int cellWidth;

        /// <summary>
        /// セルの高さ
        /// </summary>
        private int cellHeight;

        /// <summary>
        /// ブロックの幅
        /// </summary>
        private int blkWidth;

        /// <summary>
        /// ブロックの高さ
        /// </summary>
        private int blkHeight;

        /// <summary>
        /// 横方向ブロック数
        /// </summary>
        private int blkHNum;

        /// <summary>
        /// 縦方向ブロック数
        /// </summary>
        private int blkVNum;

        //// int m_bpp;

        public ImageFontWriter(
            string file,
            string order,
            ImageFileFormat format,
            bool isDrawGrid,
            bool isOutKerningData,
            int cellWidth,
            int cellHeight,
            int marginLeft,
            int marginTop,
            int marginRight,
            int marginBottom,
            uint colorGrid,
            uint colorMargin,
            uint colorWidthBar,
            uint colorNullCell,
            uint colorBackground)
        {
            this.File = file;
            this.Order = order;
            this.Format = format;
            this.IsDrawGrid = isDrawGrid;
            this.IsOutKerningData = isOutKerningData;


            this.CellMarginLeft = marginLeft;
            this.CellMarginRight = marginRight;
            this.CellMarginTop = marginTop;
            this.CellMarginBottom = marginBottom;

            this.cellWidth = cellWidth;
            this.cellHeight = cellHeight;

            this.colorGrid = colorGrid;
            this.colorMargin = colorMargin;
            this.colorWidthBar = colorWidthBar;
            this.colorNullCell = colorNullCell;
            this.colorBackground = colorBackground;

            Rpt._RPT4("{0} {1} {2} {3}\n", marginLeft, marginRight, marginTop, marginBottom);
            Rpt._RPT2("{0} {1}\n", cellWidth, cellHeight);
        }

        public override void WriteFontData(FontData fontData, GlyphOrder order)
        {
            // 保存形式決定
            var format = this.Format;
            if (format == ImageFileFormat.Ext)
            {
                var fileExt = Path.GetExtension(this.File);

                if (fileExt != string.Empty)
                {
                    fileExt = fileExt.Substring(1);
                }

                // i == 0 は FORMAT_EXT なのでとばす。
                for (int i = 1; i < this.FileExtension.Length; ++i)
                {
                    string ext = this.FileExtension[i];

                    if (string.Equals(ext, fileExt, StringComparison.InvariantCultureIgnoreCase))
                    {
                        format = (ImageFileFormat)i;
                        break;
                    }
                }

                if (format == ImageFileFormat.Ext)
                {
                    throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_UNKNOWN_IMAGE_EXT);
                }
            }

            switch (format)
            {
            case ImageFileFormat.Bmp:
                {
                    // 処理状態更新
                    ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_WRITE_BMP);
                    ProgressControl.GetInstance().ResetProgressBarPos();

                    // 画像生成
                    var image = this.CreateImage(fontData, order);
                    CheckImageSizeOverForBmpOrTga_(image);

                    ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_WRITE_BMP_FILE);
                    BitmapFile bmpfile = new BitmapFile(this.File);
                    bmpfile.Save(image);
                    image.Dispose();
                }
                break;
            case ImageFileFormat.Tif:
                {
                    // 処理状態更新
                    ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_WRITE_TIFF);
                    ProgressControl.GetInstance().ResetProgressBarPos();

                    // 画像生成
                    var image = this.CreateImage(fontData, order);
                    ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_WRITE_TIFF_FILE);

                    // まずBMPを書き出し...
                    string tempBmpFilePath = Path.Combine(Environment.GetEnvironmentVariable("TEMP"), Path.GetRandomFileName());
                    tempBmpFilePath = tempBmpFilePath + ".bmp";
                    BitmapFile bmpfile = new BitmapFile(tempBmpFilePath);
                    bmpfile.Save(image);
                    image.Dispose();

                    string tempTiffFilePath = Path.ChangeExtension(tempBmpFilePath, ".tif");
                    try
                    {
                        // その後TIFFに変換する。
                        BmpToTiff_(tempBmpFilePath);
                        if (System.IO.File.Exists(tempTiffFilePath))
                        {
                            string dstTiffFilePath = Path.ChangeExtension(this.File, ".tif");
                            System.IO.File.Copy(tempTiffFilePath, dstTiffFilePath, true);
                        }
                    }
                    finally
                    {
                        if (System.IO.File.Exists(tempTiffFilePath)) { System.IO.File.Delete(tempTiffFilePath); }
                        if (System.IO.File.Exists(tempBmpFilePath)) { System.IO.File.Delete(tempBmpFilePath); }
                    }
                }
                break;
            case ImageFileFormat.Tga:
                {
                    // 処理状態更新
                    ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_WRITE_TGA);
                    ProgressControl.GetInstance().ResetProgressBarPos();

                    // 画像生成
                    var image = this.CreateImage(fontData, order);
                    CheckImageSizeOverForBmpOrTga_(image);

                    // 処理状態更新
                    ProgressControl.GetInstance().SetStatusString(Strings.IDS_STATUS_WRITE_TGA_FILE);

                    // ファイル出力
                    TgaFile tgafile = new TgaFile(this.File);
                    tgafile.Save(image);
                    image.Dispose();
                }

                break;

            default:
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_UNKNOWN_IMAGE_FORMAT_INT);
            }

            // カーニング情報を書き出し
            if (IsOutKerningData)
            {
                // 書き出すカーニング情報がある場合だけ書き出す。
                // 存在しない場合は、ただスルーする。
                // (Image出力モードでは、ユーザに IsOutKerningData を ON/OFF させないので)
                if (fontData.KerningPairs != null && fontData.KerningPairs.Length > 0)
                {
                    using (var bw = new ByteOrderBinaryWriter(BinaryFile.Open(Path.ChangeExtension(this.File, ".bffkn"), FileMode.Create, FileAccess.Write), true))
                    {
                        // 文字コードが 32bit のカーニング情報であることを示すヘッダを書き込む
                        bw.Write(KerningPair.HeaderOf32BitModeKerning);

                        foreach (KerningPair pair in fontData.KerningPairs)
                        {
                            bw.Write(pair);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// BMP と TGA 向けのサイズチェックを行います。
        /// </summary>
        /// <param name="image"></param>
        private static void CheckImageSizeOverForBmpOrTga_(ImageBase image)
        {
            if (image.Width >= 30000 || image.Height >= 30000)
            {
                ProgressControl.Warning(Strings.IDS_WARN_OVERMAX_PIXCELSIZE);
            }
        }

        /// <summary>
        /// TIFF を出力します。
        /// </summary>
        private static void BmpToTiff_(string bmpFilePath)
        {
            Process proc = new Process();

            proc.StartInfo.FileName = Path.Combine(
                Path.GetDirectoryName(Assembly.GetEntryAssembly().Location), "TiffBmpCvtr.exe");
            proc.StartInfo.Arguments = bmpFilePath;
            proc.StartInfo.CreateNoWindow = true;
            proc.StartInfo.UseShellExecute = false;
            proc.Start();
            proc.WaitForExit();
        }

        public override void GetGlyphOrder(GlyphOrder order)
        {
            order.Load(this.Order, this.UseDtdValidation);
        }

        public override void ValidateInput()
        {
            // Bitmap File
            ValidateOutputPath(this.File);

            // Glyph Order
            ValidateOrderFile(this.Order, this.UseDtdValidation);

            // Draw Grid
            // nothing to do

            // Cell Size
            if (this.cellWidth == 0)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_UNDERMIN_CELLWIDTH, this.cellWidth);
            }

            if (this.cellHeight == 0)
            {
                throw GlCm.ErrMsg(ErrorType.Parameter, Strings.IDS_ERR_UNDERMIN_CELLHEIGHT, this.cellHeight);
            }
            //// if( (m_cellWidth <= 0) && (m_cellHeight <= 0) )
            //// {
            ////     if( m_cellMarginLeft < 0 )
            ////     {
            ////         throw ErrMsg(ERR_INFO, _T("Parameter Error:\n")
            ////                     _T(" Cell Margin Left(=%d) must be => 0."),
            ////                     m_cellMarginLeft );
            ////     }
            ////     if( m_cellMarginRight < 0 )
            ////     {
            ////         throw ErrMsg(ERR_INFO, _T("Parameter Error:\n")
            ////                     _T(" Cell Margin Right(=%d) must be => 0."),
            ////                     m_cellMarginRight );
            ////     }
            ////     if( m_cellMarginTop < 0 )
            ////     {
            ////         throw ErrMsg(ERR_INFO, _T("Parameter Error:\n")
            ////                     _T(" Cell Margin Top(=%d) must be => 0."),
            ////                     m_cellMarginTop );
            ////     }
            ////     if( m_cellMarginBottom < 0 )
            ////     {
            ////         throw ErrMsg(ERR_INFO, _T("Parameter Error:\n")
            ////                     _T(" Cell Margin Bottom(=%d) must be => 0."),
            ////                     m_cellMarginBottom );
            ////     }
            //// }
        }

        private ImageBase CreateImage(FontData fontData, GlyphOrder order)
        {
            // パラメータとイメージを初期化
            this.InitParameter(fontData, order);
            var image = this.CreatePlaneImage(fontData, order);

            // グリフ描画
            this.DrawGlyph(image, fontData, order);

            // グリッド描画
            if (this.IsDrawGrid)
            {
                this.DrawGrid(image, this.blkWidth, this.blkHeight);
            }

            // 配置情報描画
            ArrangeInfo ai = new ArrangeInfo();
            ai.BaselinePos = fontData.BaselinePos.Value;
            ai.Width = fontData.IsWidthExplicit() ? fontData.Width.Value : -1;
            ai.Ascent = fontData.IsHeightExplicit() ? fontData.Ascent.Value : -1;
            ai.Descent = fontData.IsHeightExplicit() ? fontData.Descent.Value : -1;
            this.DrawArrangeInfo(image, ai, this.IsDrawGrid);

            return image;
        }

        // パラメータ初期化
        private void InitParameter(FontData fontData, GlyphOrder order)
        {
            //---- フォント最大サイズ
            this.glpWidth = Math.Max(fontData.CellWidth.Value, fontData.MaxCharWidth.Value);
            this.glpHeight = Math.Max(fontData.CellHeight.Value, fontData.BaselinePos.Value);

            //---- セルサイズ
            if (this.cellWidth < 0)
            {
                //---- マージン指定
                this.cellWidth = this.CellMarginLeft + this.glpWidth + this.CellMarginRight;
                this.glpOffsetX = this.CellMarginLeft;
            }
            else
            {
                //---- セルサイズ指定
                if (this.CellMarginLeft < 0)
                {
                    //---- オフセットが指定されていない
                    this.glpOffsetX = (this.cellWidth - this.glpWidth + ((this.cellWidth > this.glpWidth) ? 1 : -1)) / 2;
                }
                else
                {
                    //---- オフセットが指定されている
                    this.glpOffsetX = this.CellMarginLeft;
                }
            }

            if (this.cellHeight < 0)
            {
                //---- マージン指定
                this.cellHeight = this.CellMarginTop + this.glpHeight + this.CellMarginBottom;
                this.glpOffsetY = this.CellMarginTop;
            }
            else
            {
                //---- セルサイズ指定
                if (this.CellMarginTop < 0)
                {
                    //---- オフセットが指定されていない
                    if (this.glpHeight <= this.cellHeight)
                    {
                        this.glpOffsetY = (this.cellHeight - this.glpHeight) / 2;
                    }
                    else
                    {
                        int margin = this.cellHeight - this.glpHeight;
                        int decent = this.glpHeight - fontData.BaselinePos.Value;

                        this.glpOffsetY = margin - (margin * decent / this.glpHeight);
                    }
                }
                else
                {
                    //---- オフセットが指定されている
                    this.glpOffsetY = this.CellMarginTop;
                }
            }

            // セルサイズが配置情報を出力するのに十分でなければ再調整する
            if (fontData.IsWidthExplicit())
            {
                if (fontData.Width.Value > this.cellWidth)
                {
                    int adj = fontData.Width.Value - this.cellWidth;
                    this.glpOffsetX += adj / 2;
                    this.cellWidth += adj;
                }
            }

            if (fontData.IsHeightExplicit())
            {
                int cellBaselineUpperSpace = this.glpOffsetY + fontData.BaselinePos.Value;
                int cellBaselineLowerSpace = this.cellHeight - cellBaselineUpperSpace;

                // アセントは入るか？
                if (fontData.Ascent.Value > cellBaselineUpperSpace)
                {
                    int adj = fontData.Ascent.Value - cellBaselineUpperSpace;
                    this.glpOffsetY += adj;
                    this.cellHeight += adj;
                }

                // ディセントは入るか？
                if (fontData.Descent.Value > cellBaselineLowerSpace)
                {
                    int adj = fontData.Descent.Value - cellBaselineLowerSpace;
                    this.cellHeight += adj;
                }
            }

            //---- ブロックサイズ
            this.blkWidth = this.cellWidth + FB.NoCellAreaH;
            this.blkHeight = this.cellHeight + FB.NoCellAreaV;

            //---- ブロック数
            this.blkHNum = order.GetHNum();
            this.blkVNum = order.GetVNum();
        }

        private ImageBase CreatePlaneImage(FontData font, GlyphOrder order)
        {
            int image_width = order.GetHNum() * this.blkWidth;
            int image_height = order.GetVNum() * this.blkHeight;

            var image = new RgbImage();

            image.Create(image_width, image_height, 32);
            image.EnableAlpha();
            image.Clear(this.colorBackground, ImageBase.AlphaTransparent);

            return image;
        }

        private void DrawGrid(ImageBase image, int blkWidth, int blkHeight)
        {
            int w = image.Width;
            int h = image.Height;
            Alpha a = ImageBase.AlphaTransparent;

            // margin
            for (int x = 1; x < w; x += blkWidth)
            {
                int x2 = x + blkWidth - 3;
                image.DrawRect(x, 0, 1, h, this.colorMargin, a); // 縦左
                image.DrawRect(x2, 0, 1, h, this.colorMargin, a); // 縦右
            }

            for (int y = 1; y < h; y += blkHeight)
            {
                int y2 = y + blkHeight - 5;
                int y3 = y + blkHeight - 3;
                image.DrawRect(0, y, w, 1, this.colorMargin, a); // 横上
                image.DrawRect(0, y2, w, 1, this.colorMargin, a); // 横中
                image.DrawRect(0, y3, w, 1, this.colorMargin, a); // 横下
            }

            // border
            for (int x = 0; x < w; x += blkWidth)
            {
                int x2 = x + blkWidth - 1;
                image.DrawRect(x, 0, 1, h, this.colorGrid, a);   // 縦左
                image.DrawRect(x2, 0, 1, h, this.colorGrid, a);   // 縦右
            }

            for (int y = 0; y < h; y += blkHeight)
            {
                int y2 = y + blkHeight - 1;
                image.DrawRect(0, y, w, 1, this.colorGrid, a);   // 横上
                image.DrawRect(0, y2, w, 1, this.colorGrid, a);   // 横下
            }
        }

        private void DrawArrangeInfo(ImageBase image, ArrangeInfo ai, bool isDrawGrid)
        {
            Alpha alpha = ImageBase.AlphaTransparent;
            uint color = isDrawGrid ? this.colorBackground : this.colorGrid;

            int widthL = FB.NoCellAreaLeft + ((this.cellWidth - ai.Width) / 2) - 1;
            int widthR = widthL + ai.Width + 1;
            int baselinePointY = FB.NoCellAreaTop + this.glpOffsetY + ai.BaselinePos;
            int ascentPointY = baselinePointY - ai.Ascent - 1;
            int descentPointY = baselinePointY + ai.Descent;

            //---- ベースライン位置
            image.SetColorAlpha(0, baselinePointY, color, alpha);

            //---- 幅
            if (ai.Width > 0
                && widthL >= FB.NoCellAreaLeft - 1
                && widthR <= FB.NoCellAreaLeft + this.cellWidth)
            {
                image.SetColorAlpha(widthL, 0, color, alpha);
                image.SetColorAlpha(widthR, 0, color, alpha);
            }

            //---- アセント, ディセント
            if (ai.Ascent >= 0 && ascentPointY >= FB.NoCellAreaTop - 1
             && ai.Descent >= 0 && descentPointY <= FB.NoCellAreaTop + this.cellHeight)
            {
                image.SetColorAlpha(0, ascentPointY, color, alpha);
                image.SetColorAlpha(0, descentPointY, color, alpha);
            }

            //---- 透過色指定
            image.SetColorAlpha(0, 0, this.colorBackground, alpha);
        }

        private void DrawOneGlyph(ImageBase image, FontData font, int x, int y, Glyph g)
        {
            int b = font.BaselinePos.Value;
            int cell_x = (x * this.blkWidth) + FB.NoCellAreaTop;
            int cell_y = (y * this.blkHeight) + FB.NoCellAreaLeft;
            int glyphWidth = g.Width();
            int barWidth = g.CharFeed();
            int left_0 = Math.Max(g.Left(), 0);
            int right_0 = Math.Max(g.Right(), 0);

            int margin_w = this.glpWidth - (left_0 + glyphWidth + right_0);
            int offset = margin_w / 2;

            // グリフ描画
            if (glyphWidth > 0)
            {
                RgbImage gimage = new RgbImage();

                g.ExtractGlyphImage(gimage, b);

                // m_glpOffsetX: フォント内のグリフ共通のオフセット
                // offset:       このグリフ独自のオフセット
                int dst_x = Math.Max(0, this.glpOffsetX + offset + left_0);
                int dst_y = Math.Max(0, this.glpOffsetY);
                int trs_w = Math.Min(this.cellWidth - dst_x, gimage.Width);
                int trs_h = Math.Min(this.cellHeight - dst_y, gimage.Height);
                int src_x = 0;
                int src_y = 0;

                image.Paste(
                    gimage,
                    src_x,
                    src_y,
                    cell_x + dst_x,
                    cell_y + dst_y,
                    trs_w,
                    trs_h);
            }

            // 幅線描画
            {
                Alpha a = ImageBase.AlphaTransparent;
                int barArea_x = cell_x;
                int barArea_y = cell_y + this.cellHeight + FB.PaddingWidth;
                int barArea_w = this.cellWidth;
                int barArea_h = FB.BarWidth;

                if (barWidth > 0)
                {
                    int bar_offset_x = Math.Max(0, this.glpOffsetX + offset - Math.Min(0, g.Left()));
                    int bar_offset_y = 0;
                    int bar_w = Math.Min(barArea_w - bar_offset_x, barWidth);
                    int bar_h = barArea_h;

                    // 幅線を描く
                    image.DrawRect(
                        barArea_x + bar_offset_x,
                        barArea_y + bar_offset_y,
                        bar_w,
                        bar_h,
                        this.colorWidthBar,
                        a);
                }
            }
        }

        private void DrawNullBlock(ImageBase image, FontData font, int x, int y)
        {
            int cell_x = (x * this.blkWidth) + FB.NoCellAreaTop;
            int cell_y = (y * this.blkHeight) + FB.NoCellAreaLeft;

            image.DrawRect(
                cell_x,
                cell_y,
                this.cellWidth,
                this.cellHeight,
                this.colorNullCell,
                ImageBase.AlphaTransparent);
        }

        // グリフ描画
        private void DrawGlyph(ImageBase image, FontData font, GlyphOrder order)
        {
            // プログレスバー設定
            ProgressControl.GetInstance().SetProgressBarMax(this.blkHNum * this.blkVNum);

            // 垂直方向ブロックについて
            for (int y = 0; y < this.blkVNum; y++)
            {
                // 水平方向ブロックについて
                for (int x = 0; x < this.blkHNum; x++)
                {
                    uint c = order.GetCharCode(x, y);

                    if (c != Runtime.RtConsts.InvalidCharCode)
                    {
                        // <null/> でない
                        Glyph g = font.GetGlyphList().GetByCode(c);

                        // グリフが無いなら出力しない
                        // TODO: 警告
                        if (g != null)
                        {
                            this.DrawOneGlyph(image, font, x, y, g);
                        }
                    }
                    else
                    {
                        // <null/> である
                        this.DrawNullBlock(image, font, x, y);
                    }

                    ProgressControl.GetInstance().StepProgressBar();
                }
            }
        }
    }
}
