﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using System.Threading.Tasks;  //---- アルファは256段階決めうち
    using NintendoWare.Font.Win32;
    using Alpha = System.Byte;

    public abstract class ImageBase : IEquatable<ImageBase>, IDisposable
    {
        public const int AlphaTransparent = 0;
        public const int AlphaMax = 255;
        public const int AlphaOpacity = AlphaMax;

        public static readonly IntColor RgbWhite = GlCm.BMP_RGB(0xFF, 0xFF, 0xFF);
        public static readonly IntColor RgbBlack = GlCm.BMP_RGB(0, 0, 0);

        public static readonly IntColor RgbaTransparent = GlCm.BMP_RGBA(0xFF, 0xFF, 0xFF, 0);
        public static readonly IntColor RgbaOpacity = GlCm.BMP_RGBA(0, 0, 0, 0xFF);

        private int width;
        private int height;
        private int bpp;

        private bool isValid;
        private bool isAlpha;
        private Pixel[,] image;

        public ImageBase()
        {
            this.width = -1;
            this.height = -1;
            this.bpp = 0;
            this.isValid = false;
            this.isAlpha = false;
        }

        public ImageBase(ImageBase right)
        {
            this.width = right.width;
            this.height = right.height;
            this.bpp = right.bpp;
            this.isValid = right.isValid;
            this.isAlpha = right.isAlpha;

            int pixels = right.width * right.height;

            this.image = new Pixel[right.height, right.width];
            Array.Copy(right.image, this.image, pixels);
        }

        /// <summary>
        /// 内容を破棄します。
        /// </summary>
        public void Dispose()
        {
            this.image = null;
        }

        public int Bpp { get { return this.bpp; } }

        public int Width { get { return this.width; } }

        public int Height { get { return this.height; } }

        /// <summary>
        /// ImageBaseが保持している画像メモリが実際には、
        /// 複数毎のテクスチャに分割して参照される場合にその枚数を指定します。
        /// </summary>
        public int NumSubImage { get; set; }

        public abstract bool IsIndexed { get; }

        public bool IsValid
        {
            get
            {
                return this.isValid
                        && (this.width >= 0) && (this.height >= 0)
                        && (this.bpp > 0) && (this.bpp <= 32);
            }
        }

        public bool IsEnableAlpha { get { return this.isAlpha; } }

        public int AlphaBpp { get { return 8; } }

        public int ColorBpp { get { return this.Bpp; } }

        public abstract ImageBase NewSameType();

        public abstract uint GetRGB(int x, int y);

        public Pixel GetPixel(int x, int y)
        {
            return this.image[y, x];
        }

        public void EnableAlpha()
        {
            this.isAlpha = true;
        }

        /// <summary>
        /// 作成初期化
        /// </summary>
        /// <param name="width"></param>
        /// <param name="height"></param>
        /// <param name="bpp"></param>
        public virtual void Create(int width, int height, int bpp)
        {
            Debug.Assert(IsValidBpp(bpp));
            this.width = width;
            this.height = height;
            this.bpp = bpp;
            this.NumSubImage = 1;
            this.isValid = true;
            this.isAlpha = false;

            this.image = new Pixel[height, width];
        }

        public virtual void Clear(uint color, Alpha alpha)
        {
            Debug.Assert(this.IsValid);
            Pixel pixel = new Pixel();

            pixel.Color = color;
            pixel.Alpha = alpha;

            for (int y = 0; y < this.Height; y++)
            {
                for (int x = 0; x < this.Width; x++)
                {
                    this.image[y, x] = pixel;
                }
            }
        }

        /*
            イメージ変更
        */

        public void Paste(byte[] buf)
        {
            this.Paste(buf, 0);
        }

        public void Paste(byte[] buf, int align)
        {
            Debug.Assert(this.IsValid);
            this.Paste(buf, 0, 0, this.Width, this.Height, this.Bpp, align);
        }

        public void Paste(
            byte[] buf,
            int x,
            int y,
            int width,
            int height,
            int bpp/*=0*/,
            int align/*=0*/)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(x >= 0);
            Debug.Assert(y >= 0);
            Debug.Assert(x + width <= this.Width);
            Debug.Assert(y + height <= this.Height);

            if (bpp == 0)
            {
                bpp = this.bpp;
            }

            int x_st = x;
            int x_ed = x + width;
            int y_st = y;
            int y_ed = y + height;
            int skipBits = CalcAlignSkipBits(width, bpp, align);

            if (bpp == 32 && skipBits == 0)
            {
                Parallel.For(y_st, y_ed, py =>
                {
                    int bufPosY = (py - y_st) * (width * 4);
                    for (int px = x_st; px < x_ed; px++)
                    {
                        int bufPosX = bufPosY + ((px - x_st) * 4);

                        uint read1 = buf[bufPosX + 0];
                        uint read2 = buf[bufPosX + 1];
                        uint read3 = buf[bufPosX + 2];
                        uint read4 = buf[bufPosX + 3];

                        this.image[py, px] = Pixel.FromUInt32((read4 << 24) | (read3 << 16) | (read2 << 8) | (read1));
                    }
                });
            }
            else
            {
                BitReader reader = new BitReader(buf);

                for (int py = y_st; py < y_ed; py++)
                {
                    for (int px = x_st; px < x_ed; px++)
                    {
                        uint p = reader.Read(bpp);
                        this.image[py, px] = Pixel.FromUInt32(ReverseBytes(p, bpp));
                    }

                    reader.Skip(skipBits);
                }
            }
        }

        public void Paste(ImageBase image, int x, int y)
        {
            Debug.Assert(this.IsValid);
            this.Paste(image, 0, 0, x, y, image.Width, image.Height);
        }

        public void Paste(
            ImageBase image,
            int sx,
            int sy,
            int dx,
            int dy,
            int width,
            int height)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(sx >= 0);            // 将来的に負にも対応可能
            Debug.Assert(sy >= 0);            // 将来的に負にも対応可能
            Debug.Assert(dx >= 0);            // 将来的に負にも対応可能
            Debug.Assert(dy >= 0);            // 将来的に負にも対応可能
            Debug.Assert(sx + width <= image.Width);
            Debug.Assert(sy + height <= image.Height);
            Debug.Assert(dx + width <= this.Width);
            Debug.Assert(dy + height <= this.Height);

            if (image.IsEnableAlpha)
            {
                this.EnableAlpha();
            }

            for (int y = 0; y < height; ++y)
            {
                for (int x = 0; x < width; ++x)
                {
                    this.image[dy + y, dx + x] = image.image[sy + y, sx + x];
                }
            }
        }

        /*
            点操作
        */

        public void SetAlpha(int x, int y, Alpha alpha)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(x >= 0);
            Debug.Assert(y >= 0);
            Debug.Assert(x < this.width);
            Debug.Assert(y < this.height);

            this.image[y, x].Alpha = alpha;
        }

        public void SetColor(int x, int y, uint color)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(x >= 0);
            Debug.Assert(y >= 0);
            Debug.Assert(x < this.width);
            Debug.Assert(y < this.height);

            this.image[y, x].Color = color;
        }

        public virtual void SetColorAlpha(int x, int y, uint c, Alpha a)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(x >= 0);
            Debug.Assert(y >= 0);
            Debug.Assert(x < this.width);
            Debug.Assert(y < this.height);
            Pixel pixel = new Pixel();

            pixel.Alpha = a;
            pixel.Color = c;

            this.image[y, x] = pixel;
        }

        public virtual void SetRGB(int x, int y, byte r, byte g, byte b)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(x >= 0);
            Debug.Assert(y >= 0);
            Debug.Assert(x < this.width);
            Debug.Assert(y < this.height);

            this.image[y, x].SetRGB(r, g, b);
        }

        public virtual void SetRGBGray(int x, int y, byte gray)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(x >= 0);
            Debug.Assert(y >= 0);
            Debug.Assert(x < this.width);
            Debug.Assert(y < this.height);

            this.image[y, x].GrayColor = gray;
        }

        public virtual void SetPixel(int x, int y, ImageBase.Pixel p)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(x >= 0);
            Debug.Assert(y >= 0);
            Debug.Assert(x < this.width);
            Debug.Assert(y < this.height);

            this.image[y, x] = p;
        }

        /// <summary>
        /// 描画
        /// </summary>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="w"></param>
        /// <param name="h"></param>
        /// <param name="color"></param>
        /// <param name="alpha"></param>
        public virtual void DrawRect(int x, int y, int w, int h, uint color, Alpha alpha/*=ALPHA_OPACITY*/)
        {
            Debug.Assert(this.IsValid);
            int x_st = x;
            int x_ed = x + w;
            int y_st = y;
            int y_ed = y + h;

            for (int py = y_st; py < y_ed; py++)
            {
                for (int px = x_st; px < x_ed; px++)
                {
                    this.SetColorAlpha(px, py, color, alpha);
                }
            }
        }

        /*
            抽出
        */

        /// <summary>
        /// (x, y) から width/height を bpp,align で buf に抽出する
        /// </summary>
        /// <param name="buf"></param>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="width"></param>
        /// <param name="height"></param>
        /// <param name="bpp"></param>
        /// <param name="align"></param>
        public virtual void Extract(byte[] buf, int x, int y, int width, int height, int bpp, int align)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(IsValidBpp(bpp));
            Debug.Assert(x >= 0);
            Debug.Assert(y >= 0);
            Debug.Assert(x + width <= this.Width);
            Debug.Assert(y + height <= this.Height);
            int y_st = y;
            int y_ed = y + height;
            int x_st = x;
            int x_ed = x + width;
            int skipBits = CalcAlignSkipBits(width, bpp, align);
            BitWriter writer = new BitWriter(buf);

            for (int py = y_st; py < y_ed; py++)
            {
                for (int px = x_st; px < x_ed; px++)
                {
                    Pixel pixel = this.image[py, px];
                    writer.Write(bpp, ReverseBytes(pixel.IntColor, bpp));
                }

                writer.Skip(skipBits);
            }
        }

        /// <summary>
        /// 全体を align で buf に抽出する
        /// </summary>
        /// <param name="buf"></param>
        /// <param name="align"></param>
        public virtual void Extract(byte[] buf, int align/*=0*/)
        {
            Debug.Assert(this.IsValid);
            this.Extract(buf, 0, 0, this.Width, this.Height, this.Bpp, align);
        }

        /// <summary>
        /// (x, y) から width/height を buf に抽出する
        /// </summary>
        /// <param name="image"></param>
        /// <param name="x"></param>
        /// <param name="y"></param>
        /// <param name="width"></param>
        /// <param name="height"></param>
        public virtual void Extract(ImageBase image, int x, int y, int width, int height)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(x >= 0);
            Debug.Assert(y >= 0);
            Debug.Assert(x + width <= this.width);
            Debug.Assert(y + height <= this.height);
            Debug.Assert(image != null);
            Debug.Assert(this.IsIndexed == image.IsIndexed);

            image.Create(width, height, this.bpp);

            if (this.IsEnableAlpha)
            {
                image.EnableAlpha();
            }

            for (int py = 0; py < height; py++)
            {
                for (int px = 0; px < width; px++)
                {
                    image.image[py, px] = this.image[py + y, px + x];
                }
            }
        }

        /*
            点取得
        */

        public Alpha GetAlpha(int x, int y)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(x >= 0);
            Debug.Assert(x < this.width);
            Debug.Assert(y >= 0);
            Debug.Assert(y < this.height);

            return this.image[y, x].Alpha;
        }

        public uint GetColor(int x, int y)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(x >= 0);
            Debug.Assert(x < this.width);
            Debug.Assert(y >= 0);
            Debug.Assert(y < this.height);

            return this.image[y, x].Color;
        }

        /*
            比較
        */

        public bool Equals(ImageBase right)
        {
            Debug.Assert(this.IsValid);
            Debug.Assert(right.IsValid);

            if ((object)right == null)
            {
                return false;
            }

            if (this.width != right.width)
            {
                return false;
            }

            if (this.height != right.height)
            {
                return false;
            }

            if (this.bpp != right.bpp)
            {
                return false;
            }

            if (this.isValid != right.isValid)
            {
                return false;
            }

            if (this.isAlpha != right.isAlpha)
            {
                return false;
            }

            if (this.IsEnableAlpha != right.IsEnableAlpha)
            {
                return false;
            }

            for (int y = 0; y < this.height; y++)
            {
                for (int x = 0; x < this.width; x++)
                {
                    if (this.GetColor(x, y) != right.GetColor(x, y))
                    {
                        return false;
                    }

                    if (this.IsEnableAlpha)
                    {
                        if (this.GetAlpha(x, y) != right.GetAlpha(x, y))
                        {
                            return false;
                        }
                    }
                }
            }

            return true;
        }

        public virtual void ScanOpacityRect(ref RECT rect)
        {
            Debug.Assert(this.IsValid);
            rect.left = this.width;
            rect.right = 0;
            rect.top = this.height;
            rect.bottom = 0;

            for (int y = 0; y < this.height; y++)
            {
                int left = this.width;
                int right = this.width;

                //---- 一番左の不透明ピクセルを探す
                for (int x = 0; x < this.width; x++)
                {
                    if (this.GetAlpha(x, y) != AlphaTransparent)
                    {
                        left = x;
                        break;
                    }
                }

                //---- 一番右の不透明ピクセルを探す
                for (int x = this.width - 1; x >= left; x--)
                {
                    if (this.GetAlpha(x, y) != AlphaTransparent)
                    {
                        right = x + 1;
                        break;
                    }
                }

                //---- 不透明ドットがライン中に存在した場合
                if (left < right)
                {
                    if (left < rect.left)
                    {
                        rect.left = left;
                    }

                    if (right > rect.right)
                    {
                        rect.right = right;
                    }

                    if (y < rect.top)
                    {
                        rect.top = y;
                    }

                    rect.bottom = y + 1;
                }
            }

            //---- 完全に透明な場合
            // m_width が 0 の場合に pRect.left == pRect.right == 0 と
            // なってしまうので top, bottom もチェックする
            if (rect.left > rect.right || rect.top > rect.bottom)
            {
                rect.left = 0;
                rect.right = 0;
                rect.top = 0;
                rect.bottom = 0;
            }
        }

        public virtual void ScanNonNullRect(ref RECT rect, IntColor nullColor)
        {
            Debug.Assert(this.IsValid);
            rect.left = this.width;
            rect.right = 0;
            rect.top = this.height;
            rect.bottom = 0;

            for (int y = 0; y < this.height; y++)
            {
                int left = this.width;
                int right = this.width;

                //---- 一番左の不透明ピクセルを探す
                for (int x = 0; x < this.width; x++)
                {
                    if (this.image[y, x].IntColor != nullColor)
                    {
                        left = x;
                        break;
                    }
                }

                //---- 一番右の不透明ピクセルを探す
                for (int x = this.width - 1; x >= left; x--)
                {
                    if (this.image[y, x].IntColor != nullColor)
                    {
                        right = x + 1;
                        break;
                    }
                }

                //---- 不透明ドットがライン中に存在した場合
                if (left < right)
                {
                    if (left < rect.left)
                    {
                        rect.left = left;
                    }

                    if (right > rect.right)
                    {
                        rect.right = right;
                    }

                    if (y < rect.top)
                    {
                        rect.top = y;
                    }

                    rect.bottom = y + 1;
                }
            }

            //---- 完全に透明な場合
            // m_width が 0 の場合に pRect.left == pRect.right == 0 と
            // なってしまうので top, bottom もチェックする
            if (rect.left > rect.right || rect.top > rect.bottom)
            {
                rect.left = 0;
                rect.right = 0;
                rect.top = 0;
                rect.bottom = 0;
            }
        }

        public override bool Equals(object obj)
        {
            if (obj == null)
            {
                return false;
            }

            var image = obj as ImageBase;
            return (object)image != null && this.Equals(image);
        }

        public override int GetHashCode()
        {
            Debug.Assert(this.IsValid);

            int hash =
                this.width
                ^ this.height
                ^ this.bpp
                ^ this.isValid.GetHashCode()
                ^ this.isAlpha.GetHashCode();

            for (int y = 0; y < this.height; y++)
            {
                for (int x = 0; x < this.width; x++)
                {
                    hash ^= this.GetColor(x, y).GetHashCode();
                    if (this.IsEnableAlpha)
                    {
                        hash ^= this.GetAlpha(x, y).GetHashCode();
                    }
                }
            }

            return hash;
        }

        protected static int CalcAlignSkipBits(int width, int bpp, int align)
        {
            if (align == 0)
            {
                return 0;
            }

            var bitAlign = align * 8;
            var result = (width * bpp) % bitAlign;
            return (bitAlign - result) % bitAlign;
        }

        protected static bool IsValidBpp(int bpp)
        {
            return bpp > 0 && ((bpp % 8 == 0) || bpp == 1 || bpp == 2 || bpp == 4);
        }

        protected static uint ReverseBytes(uint bits, int bitNum)
        {
            uint hbits = bits & 0xFF;

            if (bitNum <= 8)
            {
                return hbits;
            }

            hbits = (hbits << 8) | ((bits >> 8) & 0xFF);
            if (bitNum <= 16)
            {
                return hbits;
            }

            hbits = (hbits << 8) | ((bits >> 16) & 0xFF);
            if (bitNum <= 24)
            {
                return hbits;
            }

            hbits = (hbits << 8) | ((bits >> 24) & 0xFF);
            return hbits;
        }

        public struct Pixel
        {
            private byte b;
            private byte g;
            private byte r;
            private Alpha a;

            public byte B
            {
                get { return this.b; }
                set { this.b = value; }
            }

            public byte G
            {
                get { return this.g; }
                set { this.g = value; }
            }

            public byte R
            {
                get { return this.r; }
                set { this.r = value; }
            }

            public Alpha A
            {
                get { return this.a; }
                set { this.a = value; }
            }

            public byte GrayColor
            {
                set
                {
                    this.r = value;
                    this.g = value;
                    this.b = value;
                }
            }

            public uint Color
            {
                get
                {
                    return ((uint)this.r << 16) | ((uint)this.g << 8) | ((uint)this.b << 0);
                }

                set
                {
                    this.b = (byte)value;
                    this.g = (byte)(value >> 8);
                    this.r = (byte)(value >> 16);
                }
            }

            public Alpha Alpha
            {
                get { return this.a; }
                set { this.a = value; }
            }

            public uint IntColor
            {
                get
                {
                    return ((uint)this.a << 24) | this.Color;
                }

                set
                {
                    this.Color = value;
                    this.a = (Alpha)(value >> 24);
                }
            }

            public void SetRGB(byte r, byte g, byte b)
            {
                this.r = r;
                this.g = g;
                this.b = b;
            }

            public static Pixel FromUInt32(uint value)
            {
                var newObj = new Pixel();
                newObj.IntColor = value;
                return newObj;
            }
        }
    }
}
