﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using NintendoWare.Font.Win32;

    public class Glyph : IEquatable<Glyph>
    {
        public const int WidthLineSplitError = 1;

        private const int GlyphColorNull = 0;

        /// <summary>
        /// グリフイメージ。
        /// グリフイメージを囲む最小矩形のサイズで保持している。
        /// </summary>
        private readonly RgbImage Image;

        /// <summary>
        /// image 中のベースライン位置
        /// </summary>
        private int baseline;

        /// <summary>
        /// 左スペース幅
        /// </summary>
        private int leftSpace;

        /// <summary>
        /// 右スペース幅
        /// </summary>
        private int rightSpace;

        /// <summary>
        /// グリフインデックス
        /// </summary>
        private ushort index;

        /// <summary>
        /// 文字コード
        /// </summary>
        private uint code;

        public Glyph()
        {
            this.baseline = 0;
            this.leftSpace = 0;
            this.rightSpace = 0;
            this.Image = new RgbImage();
        }

        public Glyph(Glyph orig)
        {
            this.baseline = orig.baseline;
            this.leftSpace = orig.leftSpace;
            this.rightSpace = orig.rightSpace;
            this.Image = new RgbImage(orig.Image);
        }

        public void SetIndex(ushort index)
        {
            this.index = index;
        }

        public void SetCode(uint code)
        {
            this.code = code;
        }

        public ushort GetIndex()
        {
            return this.index;
        }

        public uint GetCode()
        {
            return this.code;
        }

        public int GetImageDataLen()
        {
            var result = this.Width() * this.Height() * this.Image.Bpp;
            return (result + 7) / 8;
        }

        public int CharFeed()
        {
            return this.leftSpace + this.Width() + this.rightSpace;
        }

        public int Width()
        {
            return this.Image.Width;
        }

        public int Height()
        {
            return this.Image.Height;
        }

        public int Left()
        {
            return this.leftSpace;
        }

        public int Right()
        {
            return this.rightSpace;
        }

        public int Ascent()
        {
            return this.baseline;
        }

        public int Descent()
        {
            return this.Height() - this.baseline;
        }

        /*
            image にはグリフと幅線が含まれる。
            左右スペースは幅線により与えられる。
            imgWidth にはスキャンライン長以上の意味は無い。
        */
        public void SetGlyphImageWithWidthBar(RgbImage image, RgbImage bar, int baseline, IntColor nullColor)
        {
            Debug.Assert(image.IsValid);
            Debug.Assert(bar.IsValid);
            Debug.Assert(bar.Height == 1);
            Win32.RECT rect = new Win32.RECT();

            image.ScanNonNullRect(ref rect, nullColor);
            this.ScanWidthBar(bar, ref rect, nullColor);
            this.StoreGlyph(image, ref rect, baseline);
        }

        /*
            image にはグリフのみが含まれる。
            // 左右スペースはimage内の左右スペースにより与えられる。
            左右スペースは引数により与えられる。
            // imgWidth == グリフ幅 が仮定される
            imgWidth にはスキャンライン長以上の意味は無い。
        */
        public void SetGlyphImage(RgbImage image, int l, int r, int baseline, IntColor nullColor)
        {
            Debug.Assert(image.IsValid);
            Win32.RECT rect = new Win32.RECT();

            image.ScanNonNullRect(ref rect, nullColor);

            this.leftSpace = l;
            this.rightSpace = r;

            this.StoreGlyph(image, ref rect, baseline);
        }

        /*
            image にはグリフのみが含まれる。
            左右スペースはimage内の左右スペースと引数により与えられる。
            imgWidth == グリフ幅 が仮定される
        */
        public void SetGlyphImageInclude(RgbImage image, int l, int r, int baseline)
        {
            Debug.Assert(image.IsValid);
            Win32.RECT rect = new Win32.RECT();

            image.ScanOpacityRect(ref rect);

            this.leftSpace = rect.left + l;
            this.rightSpace = image.Width - rect.right + r;

            this.StoreGlyph(image, ref rect, baseline);
        }

        /*
            image にはグリフのみが含まれる。
            左右スペースはimage内の左右スペースにより与えられる。
            imgWidth は左右スペースの計算に用いられる。
        */
        public void SetGlyphImage(RgbImage image, int baseline)
        {
            Debug.Assert(image.IsValid);
            Win32.RECT rect = new Win32.RECT();

            image.ScanOpacityRect(ref rect);

            this.leftSpace = rect.left;
            this.rightSpace = image.Width - rect.right;

            this.StoreGlyph(image, ref rect, baseline);
        }

        // buf: topdown
        // buf に bpp でベースライン位置を合わせてグリフをコピーする
        // 左右方向は左詰
        public void ExtractGlyphImage(RgbImage image, int baseline)
        {
            Debug.Assert(this.Width() >= 0 && this.Height() >= 0);

            int topSpace = Math.Max(baseline - this.Ascent(), 0);
            int extractHeight = topSpace + this.Height();

            image.Create(this.Width(), extractHeight, this.Image.Bpp);
            image.EnableAlpha();
            image.Clear(ImageBase.RgbWhite, ImageBase.AlphaTransparent);

            image.Paste(this.Image, 0, topSpace);
        }

        public bool Equals(Glyph r)
        {
            if ((object)r == null)
            {
                return false;
            }

            if (this.baseline != r.baseline)
            {
                return false;
            }

            if (this.leftSpace != r.leftSpace)
            {
                return false;
            }

            if (this.rightSpace != r.rightSpace)
            {
                return false;
            }

            if (!this.Image.Equals(r.Image))
            {
                return false;
            }

            return true;
        }

        public override bool Equals(object obj)
        {
            if (obj == null)
            {
                return false;
            }

            var r = obj as Glyph;
            return (object)r != null && this.Equals(r);
        }

        public override int GetHashCode()
        {
            return this.baseline
                ^ this.leftSpace
                ^ this.rightSpace
                ^ this.Image.GetHashCode();
        }

        // image: topdown
        // 注意: 白と黒だけで判断している。 nullColor を無視。
        private void ScanWidthBar(RgbImage bar, ref RECT r, IntColor nullColor)
        {
            int w = bar.Width;
            int left = -1;
            int right = w;
            int x = 0;

            // 左の空白を飛ばす
            while (x < w)
            {
                if (bar.GetRGB(x++, 0) != nullColor)
                {
                    left = x - 1;
                    break;
                }
            }

            // 幅線を読む
            while (x < w)
            {
                if (bar.GetRGB(x++, 0) == nullColor)
                {
                    right = x - 1;
                    break;
                }
            }

            // 線が分断されていないかチェック
            while (x < w)
            {
                if (bar.GetRGB(x++, 0) != nullColor)
                {
                    throw new GlyphException(WidthLineSplitError);
                }
            }

            /*
                 |------*****---------|  グリフ
                 |      |    |        |
                 |----***********-----|  幅線
                      | |    |   |
                      | |    |   right
                      | |    r.right
                      | r.left
                      left
            */

            if (left >= 0)
            {
                // 幅線がある場合
                if (r.right - r.left > 0)
                {
                    // グリフがある場合
                    // 左スペース = 左スペース
                    // 右スペース = 右スペース
                    this.leftSpace = r.left - left;
                    this.rightSpace = right - r.right;
                }
                else
                {
                    // グリフがない場合
                    // 左スペース = 文字幅
                    // 右スペース = 0
                    this.leftSpace = right - left;
                    this.rightSpace = 0;
                }
            }
            else
            {
                // 幅線がない場合
                // 左スペース = 0
                // 右スペース = グリフ幅
                this.leftSpace = 0;
                this.rightSpace = -(r.right - r.left);
            }
        }

        // image: topdown
        private void StoreGlyph(RgbImage image, ref RECT r, int baseline)
        {
            int width = r.right - r.left;
            int height = r.bottom - r.top;

            image.Extract(this.Image, r.left, r.top, width, height);
            this.baseline = baseline - r.top;
        }
    }

    internal class GlyphException : Exception
    {
        private int err;

        public GlyphException(int err)
        {
            this.err = err;
        }

        public int ErrorCode
        {
            get { return this.err; }
        }
    }
}
