﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.Font
{
    using System;
    using System.IO;
    using System.Reflection;
    using System.Runtime.InteropServices;

    public class ByteOrderBinaryReader : IDisposable
    {
        private readonly BinaryReader br;
        private readonly byte[] buf = new byte[8];
        private bool isLittleEndian;

        public ByteOrderBinaryReader(Stream input, bool isLittleEndian)
        {
            if (input == null)
            {
                throw new ArgumentNullException("input");
            }

            this.br = new BinaryReader(input);
            this.isLittleEndian = isLittleEndian;
        }

        public Stream BaseStream
        {
            get { return this.br.BaseStream; }
        }

        public void Dispose()
        {
            this.br.Close();
        }

        public byte ReadByte()
        {
            return this.br.ReadByte();
        }

        public sbyte ReadSByte()
        {
            return this.br.ReadSByte();
        }

        public short ReadInt16()
        {
            return (short)this.ReadUInt16();
        }

        public ushort ReadUInt16()
        {
            if (this.isLittleEndian)
            {
                return this.br.ReadUInt16();
            }
            else
            {
                this.br.Read(this.buf, 0, 2);
                return (ushort)(this.buf[0] << 8 | this.buf[1]);
            }
        }

        public int ReadInt32()
        {
            return (int)this.ReadUInt32();
        }

        public uint ReadUInt32()
        {
            if (this.isLittleEndian)
            {
                return this.br.ReadUInt32();
            }
            else
            {
                this.br.Read(this.buf, 0, 4);
                return (uint)(this.buf[0] << 24 | this.buf[1] << 16 | this.buf[2] << 8 | this.buf[3]);
            }
        }

        public long ReadInt64()
        {
            return (long)this.ReadUInt64();
        }

        public ulong ReadUInt64()
        {
            if (this.isLittleEndian)
            {
                return this.br.ReadUInt64();
            }
            else
            {
                this.br.Read(this.buf, 0, 8);
                var lo = (uint)(this.buf[0] << 24 | this.buf[1] << 16 | this.buf[2] << 8 | this.buf[3]);
                var hi = (uint)(this.buf[4] << 24 | this.buf[5] << 16 | this.buf[6] << 8 | this.buf[7]);
                return ((ulong)hi) << 32 | lo;
            }
        }

        public uint Read24Bits()
        {
            this.br.Read(this.buf, 0, 3);
            if (this.isLittleEndian)
            {
                return (uint)(this.buf[2] << 16 | this.buf[1] << 8 | this.buf[0]);
            }
            else
            {
                return (uint)(this.buf[0] << 16 | this.buf[1] << 8 | this.buf[2]);
            }
        }

        public void Read<T>(out T obj)
        {
            obj = (T)this.ReadStructImpl(typeof(T));
        }

        public int Read(byte[] buffer)
        {
            return this.br.Read(buffer, 0, buffer.Length);
        }

        public int Read(byte[] buffer, int index, int count)
        {
            return this.br.Read(buffer, index, count);
        }

        public long Seek(int offset, SeekOrigin origin)
        {
            return this.br.BaseStream.Seek(offset, origin);
        }

        private object ReadStructImpl(Type type)
        {
            return this.ReadStructImpl(null, type);
        }

        private object ReadStructImpl(FieldInfo fieldInfo, Type type)
        {
            if (type.IsPrimitive)
            {
                return this.ReadPrimitive(type);
            }
            else if (type.IsArray)
            {
                var marshalAsAttribute = (MarshalAsAttribute)Attribute.GetCustomAttribute(fieldInfo, typeof(MarshalAsAttribute));
                if (null == marshalAsAttribute)
                {
                    throw new InvalidOperationException("MarshalAs required");
                }

                if (UnmanagedType.ByValArray != marshalAsAttribute.Value)
                {
                    throw new InvalidOperationException("ByValArray only!");
                }

                var elementType = type.GetElementType();
                var array = Array.CreateInstance(elementType, marshalAsAttribute.SizeConst);

                if (elementType.IsPrimitive)
                {
                    this.ReadPrimitiveArray(elementType, array);
                }
                else
                {
                    for (int i = 0; i < array.Length; ++i)
                    {
                        array.SetValue(this.ReadStructImpl(null, elementType), i);
                    }
                }

                return array;
            }
            else
            {
                object obj = type.InvokeMember(null, BindingFlags.CreateInstance, null, null, null);
                foreach (var subFInfo in type.GetFields(BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.Instance))
                {
                    var val = this.ReadStructImpl(subFInfo, subFInfo.FieldType);
                    subFInfo.SetValue(obj, val);
                }

                return obj;
            }
        }

        private object ReadPrimitive(Type type)
        {
            if (type == typeof(sbyte))
            {
                return this.ReadSByte();
            }
            else if (type == typeof(byte))
            {
                return this.ReadByte();
            }
            else if (type == typeof(short))
            {
                return this.ReadInt16();
            }
            else if (type == typeof(ushort))
            {
                return this.ReadUInt16();
            }
            else if (type == typeof(int))
            {
                return this.ReadInt32();
            }
            else if (type == typeof(uint))
            {
                return this.ReadUInt32();
            }
            else
            {
                throw new InvalidOperationException(string.Format("Unsupport type. - {0}", type));
            }
        }

        private void ReadPrimitiveArray(Type type, Array array)
        {
            if (type == typeof(sbyte))
            {
                var tgt = (sbyte[])array;
                for (var i = 0; i < tgt.Length; ++i)
                {
                    tgt[i] = this.ReadSByte();
                }
            }
            else if (type == typeof(byte))
            {
                var tgt = (byte[])array;
                for (var i = 0; i < tgt.Length; ++i)
                {
                    tgt[i] = this.ReadByte();
                }
            }
            else if (type == typeof(short))
            {
                var tgt = (short[])array;
                for (var i = 0; i < tgt.Length; ++i)
                {
                    tgt[i] = this.ReadInt16();
                }
            }
            else if (type == typeof(ushort))
            {
                var tgt = (ushort[])array;
                for (var i = 0; i < tgt.Length; ++i)
                {
                    tgt[i] = this.ReadUInt16();
                }
            }
            else if (type == typeof(int))
            {
                var tgt = (int[])array;
                for (var i = 0; i < tgt.Length; ++i)
                {
                    tgt[i] = this.ReadInt32();
                }
            }
            else if (type == typeof(uint))
            {
                var tgt = (uint[])array;
                for (var i = 0; i < tgt.Length; ++i)
                {
                    tgt[i] = this.ReadUInt32();
                }
            }
            else
            {
                throw new InvalidOperationException(string.Format("Unsupport type. - {0}", type));
            }
        }
    }
}
