﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using u32 = System.UInt32;

    /// <summary>
    /// ビット列から毎回bit数を指定して取得するクラスです。
    /// </summary>
    public class BitReader
    {
        private byte[] buf;
        private byte data;
        private int bitAvailable;
        private int pos;

        public BitReader(byte[] buf)
        {
            this.buf = buf;
            this.bitAvailable = 0;
        }

        public void NextByte()
        {
            if (this.bitAvailable < 8)
            {
                this.bitAvailable = 0;
            }
        }

        public u32 Read(int bits)
        {
            u32 ret = 0;

            while (bits > 8)
            {
                ret <<= 8;
                ret |= this.Read8(8);
                bits -= 8;
            }

            ret <<= bits;
            ret |= this.Read8(bits);

            return ret;
        }

        public void Skip(int bits)
        {
            if (bits > this.bitAvailable)
            {
                bits -= this.bitAvailable;

                this.pos += bits / 8;
                bits %= 8;
                this.Reload();
            }

            this.bitAvailable -= bits;
        }

        private void Reload()
        {
            this.data = this.buf[this.pos++];
            this.bitAvailable = 8;
        }

        private byte Read8(int bits)
        {
            Debug.Assert(bits <= 8);
            byte ret = 0;

            if (this.bitAvailable < bits)
            {
                int lack = bits - this.bitAvailable;
                ret = (byte)(this.Read8(this.bitAvailable) << lack);
                this.Reload();
                ret |= this.Read8(lack);
                return ret;
            }
            else
            {
                ret = (byte)(this.data >> (this.bitAvailable - bits));
                ret &= (byte)(0xFF >> (8 - bits));

                this.bitAvailable -= bits;

                return ret;
            }
        }
    }
}
