﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Font
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Runtime.InteropServices;
    using NintendoWare.Font.Win32;

    public class BitmapFile
    {
        private const ushort FileSignature = (((int)'M') << 8) | (int)'B';

        private readonly string fileName;

        public BitmapFile(string fname)
        {
            this.fileName = fname;
        }

        public ImageBase Load()
        {
            ImageBase image;
            BITMAPHEADER bmh = new BITMAPHEADER();

            using (var binaryFile = BinaryFile.Open(this.fileName, FileMode.Open, FileAccess.Read))
            {
                var br = new ByteOrderBinaryReader(binaryFile, true);
                this.LoadHeader(br, bmh);
                this.ValidateFormat(bmh);
                this.ValidateFuture(bmh);

                bool isIndex = IsIndex(bmh.bmih.biBitCount);

                if (isIndex)
                {
                    IndexImage p = new IndexImage();
                    image = p;
                    this.LoadColorTable(br, p, bmh.bmih.biBitCount);
                }
                else
                {
                    image = new RgbImage();
                    if (bmh.bmih.biCompression == GdiDef.BI_BITFIELDS)
                    {
                    }
                }

                // read body
                {
                    var isBottomUp = bmh.bmih.biHeight > 0;
                    int height = Math.Abs(bmh.bmih.biHeight);
                    var dataLineSize = CalcScanLineSize(ref bmh.bmih);
                    var dataSize = CalcImageSize(ref bmh.bmih);
                    var bmpImage = new byte[dataSize];

                    br.Seek((int)bmh.bmfh.bfOffBits, SeekOrigin.Begin);

                    if (isBottomUp)
                    {
                        for (int i = 0; i < height; i++)
                        {
                            int pos = (height - i - 1) * dataLineSize;
                            br.Read(bmpImage, pos, dataLineSize);
                        }
                    }
                    else
                    {   // topdown
                        for (int i = 0; i < height; i++)
                        {
                            int pos = i * dataLineSize;
                            br.Read(bmpImage, pos, dataLineSize);
                        }
                    }

                    image.Create(bmh.bmih.biWidth, height, bmh.bmih.biBitCount);
                    image.Paste(bmpImage, 4);
                    if (bmh.bmih.biBitCount == 32)
                    {
                        image.EnableAlpha();
                    }
                }
            }

            return image;
        }

        public void Save(ImageBase image)
        {
            Debug.Assert(image.IsValid);
            var infoHeaderSize = Marshal.SizeOf(typeof(BITMAPINFOHEADER));
            int imageSize;
            BITMAPFILEHEADER fh = new BITMAPFILEHEADER();
            int colorUsed;
            int bitCount;
            IndexImage indexImage;
            int colorNum = 0;

            indexImage = image as IndexImage;

            if (indexImage != null)
            {
                colorUsed = indexImage.GetColorTableEntryNum();
                colorNum = 256;
                infoHeaderSize += colorNum * Marshal.SizeOf(typeof(RGBQUAD));
                bitCount = 8;
            }
            else
            {
                bitCount = image.Bpp; //// (image.GetBpp() > 16) ? 24: 16;
                colorUsed = 0;
            }

            imageSize = CalcImageSize(image.Width, image.Height, bitCount);
            BITMAPINFOHEADER bmiHeader = new BITMAPINFOHEADER();
            RGBQUAD[] bmiColors = null;

            // prepaire info
            {
                // 2834 pixel/meter = 72 dpi by Photoshop 8.0.1
                const int pixPerMeter = 2834;

                // bitmap file header
                {
                    // UNNES memset(&fh, 0, sizeof(fh));
                    fh.bfType = FileSignature;
                    fh.bfSize = (uint)(Marshal.SizeOf(fh.GetType()) + infoHeaderSize + imageSize);
                    fh.bfOffBits = (uint)(Marshal.SizeOf(fh.GetType()) + infoHeaderSize);
                }

                // bitmap info
                {
                    // UNNES memset(bi, 0, biSize);
                    bmiHeader.biSize = (uint)Marshal.SizeOf(bmiHeader.GetType());
                    bmiHeader.biWidth = image.Width;
                    bmiHeader.biHeight = image.Height;
                    bmiHeader.biPlanes = 1;
                    bmiHeader.biBitCount = (ushort)bitCount;
                    bmiHeader.biCompression = GdiDef.BI_RGB;
                    bmiHeader.biSizeImage = (uint)imageSize;
                    bmiHeader.biXPelsPerMeter = pixPerMeter;
                    bmiHeader.biYPelsPerMeter = pixPerMeter;
                    bmiHeader.biClrUsed = (uint)colorUsed;
                    bmiHeader.biClrImportant = (uint)colorUsed;
                }

                // color palette
                if (indexImage != null)
                {
                    bmiColors = new RGBQUAD[indexImage.GetColorTableEntryNum()];
                    for (int i = 0; i < bmiColors.Length; ++i)
                    {
                        var col = indexImage.GetColorTable()[i];
                        bmiColors[i].rgbRed = col.R;
                        bmiColors[i].rgbGreen = col.G;
                        bmiColors[i].rgbBlue = col.B;
                        bmiColors[i].rgbReserved = col.A;
                    }
                }
            }

            // save
            using (var bw = new ByteOrderBinaryWriter(BinaryFile.Open(this.fileName, FileMode.Create, FileAccess.Write), true))
            {
                // write header
                bw.Write(fh);
                bw.Write(bmiHeader);
                if (bmiColors != null)
                {
                    Array.ForEach(bmiColors, (col) => bw.Write(col));
                }

                // write body
                // 使用メモリ量を抑えるため、1 ラインづつ、変換しながら書き込む
                int lineSize = CalcScanLineSize(image.Width, bitCount);
                byte[] lineBuffer = new byte[lineSize];
                for (var py = image.Height - 1; py >= 0; py--)
                {
                    image.Extract(lineBuffer, 0, py, image.Width, 1, image.Bpp, 4);
                    bw.Write(lineBuffer, 0, lineSize);
                }
            }
        }

        public bool IsIndex()
        {
            BITMAPHEADER bmh = new BITMAPHEADER();

            using (var binaryFile = BinaryFile.Open(this.fileName, FileMode.Open, FileAccess.Read))
            {
                var br = new ByteOrderBinaryReader(binaryFile, true);
                this.LoadHeader(br, bmh);
            }

            this.ValidateFormat(bmh);

            return IsIndex(bmh.bmih.biBitCount);
        }

        public bool HasValidIdentifier()
        {
            BITMAPHEADER bmh = new BITMAPHEADER();

            using (var binaryFile = BinaryFile.Open(this.fileName, FileMode.Open, FileAccess.Read))
            {
                var br = new ByteOrderBinaryReader(binaryFile, true);
                this.LoadHeader(br, bmh);
            }

            return bmh.bmfh.bfType == FileSignature;
        }

        public bool IsInvalid()
        {
            BITMAPHEADER bmh = new BITMAPHEADER();

            using (var binaryFile = BinaryFile.Open(this.fileName, FileMode.Open, FileAccess.Read))
            {
                var br = new ByteOrderBinaryReader(binaryFile, true);
                this.LoadHeader(br, bmh);
            }

            try
            {
                this.ValidateFormat(bmh);
            }
            catch (GeneralException)
            {
                return true;
            }

            return false;
        }

        private static int CalcScanLineSize(int w, int b)
        {
            return (((w * b) + 31) / 32) * 4;
        }

        private static int CalcImageSize(int w, int h, int b)
        {
            return CalcScanLineSize(w, b) * h;
        }

        private static bool IsIndex(int bpp)
        {
            //---- 8bpp 以下はインデックスカラー
            return bpp < 16;
        }

        private int CalcScanLineSize(ref BITMAPINFOHEADER bmih)
        {
            // 32ビットアライメント
            return CalcScanLineSize(bmih.biWidth, bmih.biBitCount);
        }

        private int CalcImageSize(ref BITMAPINFOHEADER bmih)
        {
            // 高さ==負 ? トップダウン: ボトムアップ
            var dataLineSize = CalcScanLineSize(ref bmih);
            return Math.Abs(bmih.biHeight) * dataLineSize;
        }

        private int CalcPaletteColorNum(ref BITMAPINFOHEADER bmih)
        {
            // bmih.biClrUsed == 0 の場合は最大色数を使用
            return bmih.biClrUsed != 0 ? (int)bmih.biClrUsed : (1 << bmih.biBitCount);
        }

        private void LoadHeader(ByteOrderBinaryReader br, BITMAPHEADER pbmh)
        {
            br.Seek(0, SeekOrigin.Begin);

            br.Read(out pbmh.bmfh);
            br.Read(out pbmh.bmih);

            Rpt._RPT1("biSize:             {0}\n", pbmh.bmih.biSize);
            Rpt._RPT2("biWidth x biHeight: {0} x {1}\n", pbmh.bmih.biWidth, pbmh.bmih.biHeight);
            Rpt._RPT1("biPlanes:           {0}\n", pbmh.bmih.biPlanes);
            Rpt._RPT1("biBitCount:         {0}\n", pbmh.bmih.biBitCount);
            Rpt._RPT1("biCompression:      {0}\n", pbmh.bmih.biCompression != 0 ? "yes" : "no");
            Rpt._RPT1("biSizeImage:        {0}\n", pbmh.bmih.biSizeImage);
            Rpt._RPT1("biXPelsPerMeter:    {0}\n", pbmh.bmih.biXPelsPerMeter);
            Rpt._RPT1("biYPelsPerMeter:    {0}\n", pbmh.bmih.biYPelsPerMeter);
            Rpt._RPT1("biClrUsed:          {0}\n", pbmh.bmih.biClrUsed);
            Rpt._RPT1("biClrImportant:     {0}\n", pbmh.bmih.biClrImportant);
        }

        private void LoadColorTable(ByteOrderBinaryReader br, IndexImage image, int bpp)
        {
            IntColor[] colorTable;
            int bufColorNum = 1 << bpp;

            //---- ファイルはオープン済み、シーク済み
            colorTable = new IntColor[bufColorNum];
            for (int i = 0; i < bufColorNum; ++i)
            {
                RGBQUAD rgb;
                br.Read(out rgb);
                colorTable[i] = GlCm.BMP_RGBA(rgb.rgbRed, rgb.rgbGreen, rgb.rgbBlue, rgb.rgbReserved);
            }

            image.SetColorTable(colorTable, bufColorNum);
        }

        private void ValidateFormat(BITMAPHEADER bmh)
        {
            // check bitmap file header
            if (bmh.bmfh.bfType != FileSignature
                || bmh.bmfh.bfReserved1 != 0
                || bmh.bmfh.bfReserved2 != 0
                || bmh.bmfh.bfOffBits < Marshal.SizeOf(bmh.bmfh.GetType()) + Marshal.SizeOf(bmh.bmih.GetType())
                || bmh.bmih.biSize < Marshal.SizeOf(bmh.bmih.GetType()))
            {
                throw GlCm.ErrMsg(ErrorType.Bmp, Strings.IDS_ERR_INVALID_BMP_FILE, this.fileName);
            }

            // check width & height
            if (bmh.bmih.biWidth <= 0 || bmh.bmih.biHeight == 0)
            {
                throw GlCm.ErrMsg(ErrorType.Bmp, Strings.IDS_ERR_ILLEGAL_IMAGE_SIZE, this.fileName, bmh.bmih.biWidth, bmh.bmih.biHeight);
            }

            // check image size
            {
                var dataSize = CalcImageSize(ref bmh.bmih);

                // check image size
                if (dataSize > bmh.bmih.biSizeImage)
                {
                    GlCm.ErrMsg(ErrorType.Bmp, Strings.IDS_ERR_ILLEGAL_IMAGE_SIZE, this.fileName, bmh.bmih.biSizeImage, dataSize);
                }
            }

            // check image position
            {
                var position = Marshal.SizeOf(bmh.bmfh.GetType()) + Marshal.SizeOf(bmh.bmih.GetType());

                if (bmh.bmih.biBitCount <= 8)
                {
                    var colorNum = this.CalcPaletteColorNum(ref bmh.bmih);
                    position += Marshal.SizeOf(typeof(RGBQUAD)) * colorNum;
                }
            }
        }

        private void ValidateFuture(BITMAPHEADER bmh)
        {
            // check bpp
            if (bmh.bmih.biBitCount != 1
                && bmh.bmih.biBitCount != 4
                && bmh.bmih.biBitCount != 8
                //// && bmh.bmih.biBitCount != 16
                && bmh.bmih.biBitCount != 24
                && bmh.bmih.biBitCount != 32)
            {
                throw GlCm.ErrMsg(
                    ErrorType.Bmp,
                    Strings.IDS_ERR_BMP_UNSUPPORTED_BPP,
                    this.fileName,
                    bmh.bmih.biBitCount);
            }

            // check compression & number of planes
            if (bmh.bmih.biCompression != GdiDef.BI_RGB || bmh.bmih.biPlanes != 1)
            {
                throw GlCm.ErrMsg(ErrorType.Bmp, Strings.IDS_ERR_UNSUPPORTED_BMP, this.fileName);
            }
        }

        private class BITMAPHEADER
        {
            public BITMAPFILEHEADER bmfh;
            public BITMAPINFOHEADER bmih;
        }
    }
}
