﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//------------------------------------------------------------------------------
//  EffectViewerGfx のビルド時コンフィグ
//  インクルードの制御を行うので、必ず先頭で #include する
//------------------------------------------------------------------------------
#include <Config.h>

//------------------------------------------------------------------------------
//  SDK ヘッダの include
//------------------------------------------------------------------------------
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/vfx.h>
#include <nn/vi.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/gfx/gfx_Variation-api.gl.h>

#define NN_PERF_PROFILE_ENABLED
//#include <nn/perf.h>
#include <nn/perf/perf_Profile.h>
#include <nn/perf/perf_LoadMeter.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <nn/nn_Windows.h>
#include <GL/glew.h>
#endif

#include <nn/htcs.h>

#if NN_VFX_VIEWER_USE_FONT
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <filesystem>
#endif
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
    #if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
        #include <nvnTool/nvnTool_GlslcInterface.h>
    #endif
#endif

//------------------------------------------------------------------------------
//  デモ用ヘッダの #include
//------------------------------------------------------------------------------
#include <nns/nac/nac_Pad.h>
#include <nns/nac/nac_Mouse.h>
#include <nns/nac/nac_File.h>

//------------------------------------------------------------------------------
//  EffectViewerGfx 固有ヘッダの #include
//------------------------------------------------------------------------------
#include <Grid.h>
#include <Viewer.h>
#include <Camera.h>
#include <CaptureFrame.h>
#include <ViewerLaunchArg.h>
#include <detail/util_GpuStatistics.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_FrameBuffer.h>
#include <nns/gfx/gfx_ColorBuffer.h>
#include <nns/gfx/gfx_DepthStencilBuffer.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <detail/util_PngIO.h>
#endif

//------------------------------------------------------------------------------
//  定数・マクロ定義
//------------------------------------------------------------------------------
#ifndef _MAX_PATH
#define _MAX_PATH 260  //NOLINT(preprocessor/const)
#endif
static const uint32_t MemSizeForVfx            = 256 * 1024 * 1024;     // Vfx 用メモリサイズ
static const uint32_t MemSizeForGfx            = 128 * 1024 * 1024;     // Gfx 用メモリサイズ

//------------------------------------------------------------------------------
//  グローバル変数定義
//------------------------------------------------------------------------------
nn::gfx::DescriptorPool                 g_SamplerDescriptorPool;
nn::gfx::DescriptorPool                 g_TextureDescriptorPool;

nn::gfx::TextureView                    g_ColorTextureView;
nn::gfx::DescriptorSlot                 g_ColorTextureDescSlot;
nn::gfx::ColorTargetView                g_ColorCopyTargetView;
nn::gfx::ColorTargetView                g_2DTargetView;

nn::gfx::TextureView                    g_DepthStencilTextureView;
nn::gfx::DescriptorSlot                 g_DepthStencilTextureDescSlot;
nn::gfx::ColorTargetView                g_DepthStencilCopyTargetView;

nns::gfx::FrameBuffer                   g_CopyRenderTarget;
nn::gfx::DescriptorSlot                 g_ColorCopyTextureDescSlot;
nn::gfx::DescriptorSlot                 g_DepthStencilCopyTextureDescSlot;

nns::gfx::FrameBuffer                   g_2DRenderTarget;
nn::gfx::DescriptorSlot                 g_2DTextureDescSlot;

Viewer                                  g_Viewer;
vfxdemo::DrawParam                      g_DrawParam;
Camera                                  g_DemoCamera;

GpuStatistics                           g_GpuStatistics;                // GPU 情報取得

nn::util::Vector3fType                  g_CameraPosition;               // カメラ位置
nn::util::Vector3fType                  g_CameraLookAt;                 // カメラ視点
nn::util::Matrix4x4fType                g_ProjectionMatrix;             // プロジェクション

int                                     g_ScreenWidth;                  // 画面の横幅です。
int                                     g_ScreenHeight;                 // 画面の縦幅です。
int                                     g_WindowWidth;                  // ウインドウの横幅です。
int                                     g_WindowHeight;                 // ウインドウの縦幅です。


nns::gfx::GraphicsFramework::FrameworkMode  g_DrawFrameworkMode;
vfxdemo::TextureDescriptorIndexAllocator g_TextureDescPoolAllocator;
vfxdemo::SamplerDescriptorIndexAllocator g_SamplerDescPoolAllocator;

nn::gfx::Buffer                         g_CaptureBuffer;
int                                     g_CaptureIndex;
int                                     g_CapturedFrameNumber;

void*                                   g_PerfMemory = nullptr;
//------------------------------------------------------------------------------
//  フォント用定義
//------------------------------------------------------------------------------
#if NN_VFX_VIEWER_USE_FONT
nns::gfx::GraphicsFramework::DebugFontTextWriter g_Writer;
#endif

//------------------------------------------------------------------------------
//  プラットフォーム固有の変数定義
//------------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_WIN )
HWND                                    g_HWnd;
WNDPROC                                 g_CustomWndProc;
nns::nac::Mouse                         g_Mouse;
char                                    g_WorkingDir[ _MAX_PATH ];
nn::lmem::HeapHandle                    g_HeapHandle;
#endif

nns::nac::Pad                           g_Pad;                          // デバッグ用パッド管理クラスのインスタンスへのポインタです。
bool                                    g_IsSetFocus;                   // ウィンドウのフォーカス監視

FrameCapture                            g_FrameCapture;
int                                     g_BufferIndex = 0;
char                                    g_ScreenCapturedFilePath[_MAX_PATH];
bool                                    g_ReservationScreenCapture = false;


//------------------------------------------------------------------------------
//  GraphicsFramework
//------------------------------------------------------------------------------
nns::gfx::GraphicsFramework::FrameworkInfo g_GraphicsFrameworkInfo;
nns::gfx::GraphicsFramework                g_GraphicsFramework;

#if defined( NN_BUILD_CONFIG_OS_WIN )
//------------------------------------------------------------------------------
//  Perf のコア番号を返すコールバック間数
//------------------------------------------------------------------------------
// windows 版では nn::os::SetThreadCoreMask による設定が効かない場合があるため
// メインスレッドしか使わない場合は windows 版では コア番号は常に 0 を返すようにする。
int GetFixedCoreNumber() NN_NOEXCEPT
{
    return 0;
}
#endif

//------------------------------------------------------------------------------
//  固定テクスチャディスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateStaticTextureDescriptorSlot_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView,  void* pUserData) NN_NOEXCEPT
{
    return g_TextureDescPoolAllocator.AllocateStaticTextureDescriptorSlot(pDstSlot, textureView, pUserData);
}

//------------------------------------------------------------------------------
//  固定テクスチャディスクリプタスロットを開放
//------------------------------------------------------------------------------
void FreeStaticTextureDescriptorSlot_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pDstSlot);
    NN_UNUSED(textureView);
    NN_UNUSED(pUserData);
}

//------------------------------------------------------------------------------
//  トランジェントテクスチャディスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateTransientTextureDescriptorSlot_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView,  void* pUserData) NN_NOEXCEPT
{
    return g_TextureDescPoolAllocator.AllocateTransientTextureDescriptorSlot(pDstSlot, textureView, pUserData);
}

//------------------------------------------------------------------------------
//  トランジェントテクスチャディスクリプタスロットを開放
//------------------------------------------------------------------------------
void FreeTransientTextureDescriptorSlot_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pDstSlot);
    NN_UNUSED(textureView);
    NN_UNUSED(pUserData);
}

//------------------------------------------------------------------------------
//  固定サンプラディスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateStaticSamplerDescriptorSlot_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData ) NN_NOEXCEPT
{
    return g_SamplerDescPoolAllocator.AllocateStaticSamplerDescriptorSlot(pDstSlot, sampler, pUserData);
}

//------------------------------------------------------------------------------
//  固定サンプラディスクリプタスロットを開放

//------------------------------------------------------------------------------
void FreeStaticSamplerDescriptorSlot_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData ) NN_NOEXCEPT
{
    NN_UNUSED(pDstSlot);
    NN_UNUSED(sampler);
    NN_UNUSED(pUserData);
}

//------------------------------------------------------------------------------
//  トランジェント・サンプラディスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateTransientSamplerDescriptorSlot_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData ) NN_NOEXCEPT
{
    return g_SamplerDescPoolAllocator.AllocateTransientSamplerDescriptorSlot(pDstSlot, sampler, pUserData);
}

//------------------------------------------------------------------------------
//  トランジェント・サンプラディスクリプタスロットを開放
//------------------------------------------------------------------------------
void FreeTransientSamplerDescriptorSlot_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData ) NN_NOEXCEPT
{
    NN_UNUSED(pDstSlot);
    NN_UNUSED(sampler);
    NN_UNUSED(pUserData);
}

#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
//------------------------------------------------------------------------------
// GLSLC 用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* GlslcAllocateFunction(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

static void GlslcFreeFunction(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

static void* GlslcReallocateFunction(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

//!--------------------------------------------------------------------------------------
//! @brief GpuBuffer 用メモリ確保関数
//!--------------------------------------------------------------------------------------
void* GpuBufferAllocateFunction(size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);

    // ユーザーデータポインタでアロケータを渡してもらう.
    nns::nac::MemoryAllocator*    pAllocator = static_cast<nns::nac::MemoryAllocator*>(pUserData);

    return pAllocator->Alloc(size, alignment);
}

//!--------------------------------------------------------------------------------------
//! @brief GpuBuffer 用メモリ開放関数
//!--------------------------------------------------------------------------------------
void GpuBufferDeallocateFunction(void* ptr, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);

    // ユーザーデータポインタでアロケータを渡してもらう.
    nns::nac::MemoryAllocator*    pAllocator = static_cast<nns::nac::MemoryAllocator*>(pUserData);

    pAllocator->Free(ptr);
}

//------------------------------------------------------------------------------
//  エフェクトビューアのコールバック関数の定義
//------------------------------------------------------------------------------
void _vfxDrawPostCallback( const void* pParam ) NN_NOEXCEPT
{
    NN_UNUSED( pParam );
#if NN_GFX_IS_TARGET_GL
    glBindVertexArray( 0 );
#endif
}

//------------------------------------------------------------------------------
//  入力関連 初期化処理
//------------------------------------------------------------------------------
void InitializeInputInterface() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_Pad.Initialize( g_HWnd );
    g_Pad.SetUseKeyboardForcus( true );
#else
    g_Pad.Initialize();
#endif

#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_Mouse.Initialize( g_HWnd );
    g_Mouse.SetUseKeyboardForcus( true );
#endif
}

//------------------------------------------------------------------------------
//  入力関連 更新処理
//------------------------------------------------------------------------------
void UpdateInputInterface() NN_NOEXCEPT
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    g_Pad.SetPadState( g_Viewer.GetPadState() );
#endif
    g_Pad.Update(g_IsSetFocus);

#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_Mouse.Update(g_IsSetFocus);
#endif
}

//------------------------------------------------------------------------------
//  入力関連 終了処理
//------------------------------------------------------------------------------
void FinalizeInputInterface() NN_NOEXCEPT
{
    g_Pad.Finalize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_Mouse.Finalize();
#endif
}

//---------------------------------------------------------------------------
//  Pad 管理クラスのインスタンスへのポインタを取得します。
//---------------------------------------------------------------------------
nns::nac::Pad* GetPad() NN_NOEXCEPT { return &g_Pad; }

//---------------------------------------------------------------------------
//  マウス管理クラスのインスタンスへのポインタを取得します。
//---------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_WIN )
nns::nac::Mouse* GetMouse() NN_NOEXCEPT { return &g_Mouse; }
#else
nns::nac::Mouse* GetMouse() NN_NOEXCEPT { return NULL; }
#endif

//------------------------------------------------------------------------------
//  メモリ関数の定義
//------------------------------------------------------------------------------
template< typename T >
T* AlignedAlloc( void** ppMemory, size_t size, size_t alignment = 1 ) NN_NOEXCEPT
{
    *ppMemory = reinterpret_cast< void* >( (
        reinterpret_cast< uintptr_t >( *ppMemory ) + alignment - 1 ) & ~( alignment - 1 ) );
    T* ret    = static_cast< T* >( *ppMemory );
    *ppMemory = static_cast< uint8_t* >( *ppMemory ) + size;
    return ret;
}

#if defined( NN_BUILD_CONFIG_OS_WIN )

//------------------------------------------------------------------------------
//  ウィンドウサイズの変更(WM_SIZE)
//------------------------------------------------------------------------------
LRESULT ResizeWindowProc(HWND hWnd, WPARAM wp, LPARAM lp) NN_NOEXCEPT
{
    NN_UNUSED( wp );
    NN_UNUSED( lp );

    RECT rc;
    GetClientRect( hWnd, &rc );
    g_WindowWidth  = rc.right;
    g_WindowHeight = rc.bottom;
    return 0;
}

LRESULT CALLBACK CustomWindowProc( HWND hWnd, UINT msg, WPARAM wp, LPARAM lp ) NN_NOEXCEPT
{
    switch ( msg )
    {
    case WM_SIZE:
        return ResizeWindowProc( hWnd, wp, lp );
    case WM_KILLFOCUS:
        g_IsSetFocus = false;
        break;
    case WM_SETFOCUS:
        g_IsSetFocus = true;
        break;
    case WM_ERASEBKGND:
        return 1;
    default:
        break;
    }
    return CallWindowProc( g_CustomWndProc, hWnd, msg, wp, lp );
}

#endif

//------------------------------------------------------------------------------
//  Gfx初期処理
//------------------------------------------------------------------------------
void InitializeGfx( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator, size_t poolSize , int width, int height ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pAllocator );
    NN_SDK_ASSERT_GREATER( width, 0 );
    NN_SDK_ASSERT_GREATER( width, 0 );
    NN_SDK_ASSERT_GREATER( height, 0 );
    NN_UNUSED( poolSize );

#if defined( NN_BUILD_CONFIG_OS_WIN )
    nn::vi::GetNativeWindow( ( nn::vi::NativeWindowHandle* )&g_HWnd, g_GraphicsFramework.GetLayer() );

    RECT rw, rc;
    ::GetWindowRect( g_HWnd, &rw );
    ::GetClientRect( g_HWnd, &rc );

    int new_width = (rw.right - rw.left) - (rc.right - rc.left) + width;
    int new_height = (rw.bottom - rw.top) - (rc.bottom - rc.top) + height;

    // 2015/4/1現在、gfxではウインドウサイズが設定出来ないようなので
    // Win32APIを呼び出して強制的にウインドウサイズを設定
    // 将来的には削除予定
    SetWindowPos(static_cast<HWND>( g_HWnd ), NULL, 0, 0, new_width, new_height, SWP_NOMOVE | SWP_NOZORDER);

    // ウィンドウのサブクラス化
    HWND hWnd = static_cast<HWND>( g_HWnd );
    g_CustomWndProc = reinterpret_cast<WNDPROC>(SetWindowLongPtr(hWnd, GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(CustomWindowProc)));

    // ウィンドウ名を設定
    wchar_t winTitle[256];
    wchar_t platformName[64];
    swprintf_s( (wchar_t*)platformName, 64, L"%s", L"Windows" );

#if defined( NN_BUILD_CONFIG_SPEC_GENERIC )
    #if defined( NN_BUILD_CONFIG_ADDRESS_SUPPORTS_32 )
    swprintf_s( (wchar_t*)platformName, 64, L"%s", L"Windows32" );
    #endif
#elif defined( NN_BUILD_CONFIG_SPEC_NX )
    #if defined( NN_BUILD_CONFIG_ADDRESS_SUPPORTS_32 )
    swprintf_s( (wchar_t*)platformName, 64, L"%s", L"NX32" );
    #else
    swprintf_s( (wchar_t*)platformName, 64, L"%s", L"NX" );
    #endif
#endif
    swprintf_s( (wchar_t*)&winTitle, 256, L"%s%s", L"EffectViewer ", platformName );
    SetWindowText(hWnd, (wchar_t*)&winTitle );
#endif


    // テクスチャサンプラプールを初期化
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler );
        info.SetSlotCount( 1024 );
        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( pDevice, info );
        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            size,
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( pDevice, info ) );

        g_SamplerDescriptorPool.Initialize( pDevice, info,
            g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ), offset, size );

        // サンプラ・ディスクリプタスロット・アロケータの初期化
        int baseIndex = 256;
        int staticSlotCount = 64;
        int transientSlotCount   = 256;
        g_SamplerDescPoolAllocator.Initialize( &g_SamplerDescriptorPool, baseIndex, staticSlotCount, transientSlotCount );
    }

    // テクスチャデスクリプタプールを初期化
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
        info.SetSlotCount( 1024 * 32 );
        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( pDevice, info );
        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            size,
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( pDevice, info ) );

        g_TextureDescriptorPool.Initialize( pDevice, info,
            g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ), offset, size );

        // テクスチャ・ディスクリプタスロット・アロケータの初期化
        int baseIndex          = 256;
        int staticSlotCount    = 64;
        int transientSlotCount = 1024;
        g_TextureDescPoolAllocator.Initialize( &g_TextureDescriptorPool, baseIndex, staticSlotCount, transientSlotCount );
    }

   // カラーバッファのテクスチャビューを初期化
    {
        nn::gfx::TextureView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
        viewInfo.SetImageFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
        viewInfo.SetTexturePtr( g_GraphicsFramework.GetColorBuffer() );
        g_ColorTextureView.Initialize( pDevice, viewInfo );

        // デスクリプタスロット設定
        g_TextureDescPoolAllocator.AllocateStaticTextureDescriptorSlot( &g_ColorTextureDescSlot, g_ColorTextureView, NULL );
    }


    // カラーコピーバッファのテクスチャ
    {
        // nn::gfx::Texture::InfoType info;
        nns::gfx::FrameBuffer::InfoType info( g_ScreenWidth, g_ScreenHeight );
        info.SetDefault();
        info.SetColorBufferFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
        info.SetDepthBufferEnabled();
        info.SetDepthBufferFormat( nn::gfx::ImageFormat_R32_Float );
        info.SetWidth( g_ScreenWidth );
        info.SetHeight( g_ScreenHeight );

        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,
            nns::gfx::FrameBuffer::GetRequiredMemoryPoolSize( pDevice, info ),
            nns::gfx::FrameBuffer::GetMemoryPoolAlignment( pDevice, info ) );

        g_CopyRenderTarget.Initialize( pDevice, &info,
            g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget ),
            offset, GpuBufferAllocateFunction, GpuBufferDeallocateFunction, pAllocator );

        // カラーコピーターゲットビューを初期化
        nn::gfx::ColorTargetView::InfoType targetInfo;
        targetInfo.SetDefault();
        targetInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
        targetInfo.SetImageFormat(    nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
        targetInfo.SetTexturePtr( g_CopyRenderTarget.GetColorBuffer()->GetTexture() );
        g_ColorCopyTargetView.Initialize( pDevice, targetInfo );

        // デスクリプタスロット設定
        g_TextureDescPoolAllocator.AllocateStaticTextureDescriptorSlot( &g_ColorCopyTextureDescSlot, *g_CopyRenderTarget.GetColorBuffer()->GetTextureView(), NULL );
        g_TextureDescPoolAllocator.AllocateStaticTextureDescriptorSlot( &g_DepthStencilCopyTextureDescSlot, *g_CopyRenderTarget.GetDepthBuffer()->GetTextureView(), NULL );
    }


    // 2D用のレンダーターゲット
    {
        // nn::gfx::Texture::InfoType info;
        nns::gfx::FrameBuffer::InfoType info( g_ScreenWidth, g_ScreenHeight );
        info.SetDefault();
        info.SetColorBufferFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
        info.SetDepthBufferDisabled();
        info.SetWidth( g_ScreenWidth );
        info.SetHeight( g_ScreenHeight );

        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,
            nns::gfx::FrameBuffer::GetRequiredMemoryPoolSize( pDevice, info ),
            nns::gfx::FrameBuffer::GetMemoryPoolAlignment( pDevice, info ) );

        g_2DRenderTarget.Initialize( pDevice, &info,
            g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget ),
            offset, GpuBufferAllocateFunction, GpuBufferDeallocateFunction, pAllocator );

        // カラーコピーターゲットビューを初期化
        nn::gfx::ColorTargetView::InfoType targetInfo;
        targetInfo.SetDefault();
        targetInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
        targetInfo.SetImageFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
        targetInfo.SetTexturePtr( g_2DRenderTarget.GetColorBuffer()->GetTexture() );
        g_2DTargetView.Initialize( pDevice, targetInfo );

        // デスクリプタスロット設定
        g_TextureDescPoolAllocator.AllocateStaticTextureDescriptorSlot( &g_2DTextureDescSlot, *g_2DRenderTarget.GetColorBuffer()->GetTextureView(), NULL );
    }

    // 深度ステンシルテクスチャビューを初期化
    {
        nn::gfx::TextureView::InfoType texViewInfo;
        texViewInfo.SetDefault();
        texViewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
        texViewInfo.SetImageFormat( nn::gfx::ImageFormat_D32_Float );
        texViewInfo.SetTexturePtr( g_GraphicsFramework.GetDepthStencilBuffer() );
        g_DepthStencilTextureView.Initialize( pDevice, texViewInfo );

        // デスクリプタスロット設定
        g_TextureDescPoolAllocator.AllocateStaticTextureDescriptorSlot( &g_DepthStencilTextureDescSlot, g_DepthStencilTextureView, NULL );
    }

    {
        // 深度ステンシルコピーターゲットビューを初期化
        nn::gfx::ColorTargetView::InfoType targetInfo;
        targetInfo.SetDefault();
        targetInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
        targetInfo.SetImageFormat(    nn::gfx::ImageFormat_R32_Float );
        targetInfo.SetTexturePtr( g_CopyRenderTarget.GetDepthBuffer()->GetTexture() );
        g_DepthStencilCopyTargetView.Initialize( pDevice, targetInfo );
    }

#if defined( NN_BUILD_CONFIG_OS_WIN )
    // キャプチャフレーム用のバッファ初期化
    {
        g_CaptureIndex        = 0;
        g_CapturedFrameNumber = 0;

        size_t imageBufferSize = g_ScreenWidth * g_ScreenHeight * 16;
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetSize(imageBufferSize);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);

        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            info.GetSize(),
            nn::gfx::Buffer::GetBufferAlignment( pDevice, info ) );

        g_CaptureBuffer.Initialize(
            pDevice,
            info,
            g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ),
            offset,
            g_GraphicsFramework.GetMemoryPoolAllocator( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer )->GetSize() );
    }
#endif
}

//------------------------------------------------------------------------------
//  Gfx終了処理
//------------------------------------------------------------------------------
void FinalizeGfx( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator ) NN_NOEXCEPT
{
    g_DepthStencilTextureView.Finalize( pDevice );
    g_ColorTextureView.Finalize( pDevice );
    g_TextureDescriptorPool.Finalize( pDevice );
    g_SamplerDescriptorPool.Finalize( pDevice );
    g_DepthStencilCopyTargetView.Finalize( pDevice );
    g_ColorCopyTargetView.Finalize( pDevice );
    g_2DRenderTarget.Finalize( pDevice, GpuBufferAllocateFunction, GpuBufferDeallocateFunction, pAllocator );
    g_2DTargetView.Finalize( pDevice );
    g_CopyRenderTarget.Finalize( pDevice, GpuBufferAllocateFunction, GpuBufferDeallocateFunction, pAllocator );
#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_CaptureBuffer.Finalize( pDevice );
#endif
}

//------------------------------------------------------------------------------
//  PrimitiveRenderer 初期化処理
//------------------------------------------------------------------------------
nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;
void InitializePrimitiveRenderer( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_SDK_ASSERT_NOT_NULL( pAllocator );

    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(GpuBufferAllocateFunction, pAllocator);
    info.SetAdditionalBufferSize(1024 * 1024);

    // ダブルバッファで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファをダブルにしたうえで、
    // g_pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )する
    info.SetMultiBufferQuantity( 2 );

    // PrimitiveRendererのインスタンス
    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer( pDevice, info);

    g_pPrimitiveRenderer->SetScreenWidth( g_ScreenWidth );
    g_pPrimitiveRenderer->SetScreenHeight( g_ScreenHeight );
}

//------------------------------------------------------------------------------
//  PrimitiveRenderer 破棄処理
//------------------------------------------------------------------------------
void FinalizePrimitiveRenderer( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );

    nns::gfx::PrimitiveRenderer::DestroyRenderer( g_pPrimitiveRenderer, pDevice, GpuBufferDeallocateFunction, pAllocator );
}

void MakeCommandCallback( nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData ) NN_NOEXCEPT
{
    NN_UNUSED( pGraphicsFramework );
    NN_UNUSED( pUserData );
    ViewerLaunchArg* pViewerLaunchArg = reinterpret_cast<ViewerLaunchArg*>( pUserData );

    nn::gfx::CommandBuffer*    pCommandBuffer   = g_GraphicsFramework.GetRootCommandBuffer( bufferIndex );
    nn::gfx::ColorTargetView*  pColorTargetView = g_GraphicsFramework.GetColorTargetView();
    nn::gfx::DepthStencilView* pDepthStecilView = g_GraphicsFramework.GetDepthStencilView();

    nn::gfx::Texture* pColorBuffer = g_GraphicsFramework.GetColorBuffer();
    nn::gfx::Texture* pDepthStencilBuffer = g_GraphicsFramework.GetDepthStencilBuffer();

    // Maker からのキャプチャメッセージが発行されているか
#if defined( NN_BUILD_CONFIG_OS_WIN )
    bool isViewerCapMessage = strlen( g_Viewer.GetCaptureFilePath() ) > 0 ? true : false;
#endif

    g_GraphicsFramework.BeginFrame( bufferIndex );
    {
        // カラーバッファ・デプスバッファのクリア
        // カラーバッファは背景色で更新
        const float* pBgColor = g_Viewer.GetBackgroundColor();
        pCommandBuffer->ClearColor( pColorTargetView, pBgColor[0], pBgColor[1], pBgColor[2], pBgColor[3], NULL );
        pCommandBuffer->ClearDepthStencil( pDepthStecilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );

        // MEMO: nn::gfx::ComanndBuffer::Clear~ はRenderTarget の割り当てを破棄するので、SetRenderTargetはそのあとに呼ぶ。

        pCommandBuffer->SetRenderTargets( 1, &pColorTargetView, pDepthStecilView );
        pCommandBuffer->InvalidateMemory( nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode );
        pCommandBuffer->SetDescriptorPool( &g_TextureDescriptorPool );
        pCommandBuffer->SetDescriptorPool( &g_SamplerDescriptorPool );
        pCommandBuffer->SetViewportScissorState( g_GraphicsFramework.GetViewportScissorState() );

        pCommandBuffer->SetRasterizerState( g_GraphicsFramework.GetRasterizerState( nns::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone ) );

        g_pPrimitiveRenderer->SetBlendState( pCommandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Opacity );
        g_pPrimitiveRenderer->SetDepthStencilState( pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthWriteTest );

        // エフェクト用のコールバックを設定
        // TODO: 読みづらい
        g_DrawParam.m_ProjMtx                       = g_ProjectionMatrix;
        g_DrawParam.m_ViewMtx                       = g_DemoCamera.GetMatrix();
        g_DrawParam.m_CameraPos                     = g_CameraPosition;
        g_DrawParam.m_NearZ                         = g_Viewer.GetNearZ();
        g_DrawParam.m_FarZ                          = g_Viewer.GetFarZ();
        g_DrawParam.m_CommandBuffer                 = pCommandBuffer;
        g_DrawParam.m_PrimitiveRenderer             = g_pPrimitiveRenderer;
        g_DrawParam.m_BufferWidth                   = g_ScreenWidth;
        g_DrawParam.m_BufferHeight                  = g_ScreenHeight;
        g_DrawParam.m_ColorTexture                  = pColorBuffer;
        g_DrawParam.m_ColorTargetView               = pColorTargetView;
        g_DrawParam.m_ColorDescSlot                 = g_ColorTextureDescSlot;
        g_DrawParam.m_DepthStencilTexture           = pDepthStencilBuffer;
        g_DrawParam.m_DepthStencilView		        = pDepthStecilView;
        g_DrawParam.m_DepthStencilDescSlot          = g_DepthStencilTextureDescSlot;
        g_DrawParam.m_ColorCopyTexture              = g_CopyRenderTarget.GetColorBuffer()->GetTexture();
        g_DrawParam.m_ColorCopyTargetView           = &g_ColorCopyTargetView;
        g_DrawParam.m_ColorCopyTextureView          = g_CopyRenderTarget.GetColorBuffer()->GetTextureView();
        g_DrawParam.m_DepthStencilCopyTexture       = g_CopyRenderTarget.GetDepthBuffer()->GetTexture();
        g_DrawParam.m_DepthStencilCopyTargetView    = &g_DepthStencilCopyTargetView;
        g_DrawParam.m_DepthStencilCopyTextureView	= g_CopyRenderTarget.GetDepthBuffer()->GetTextureView();
        g_DrawParam.m_ColorCopyDescSlot             = g_ColorCopyTextureDescSlot;
        g_DrawParam.m_DepthStencilCopyDescSlot      = g_DepthStencilCopyTextureDescSlot;
        g_DrawParam.m_BaseFovy                      = nn::util::DegreeToRadian( 60.0f ); // EffectMaker の標準の画角を EffectViewer の基準画角とする
        g_DrawParam.m_CurrentFovy                   = g_Viewer.GetFovy();

        g_pPrimitiveRenderer->Update( bufferIndex );

#if defined( NN_BUILD_CONFIG_OS_WIN )
        g_Viewer.AdjustCameraFromUserData( &g_Pad, &g_DrawParam );
#endif

#if NN_GFX_IS_TARGET_NVN
        GpuStatistics* gpuStatistics = &g_GpuStatistics;
        g_Viewer.ProcDraw( &g_DrawParam, gpuStatistics, pViewerLaunchArg );
#else
        g_Viewer.ProcDraw(&g_DrawParam, NULL, pViewerLaunchArg );
#endif

        // デバッグ表示
#if defined( NN_BUILD_CONFIG_OS_WIN )
        if ( ( g_Viewer.DoDebugDraw() ) && !g_Pad.IsHold( nns::nac::Pad::MASK_R ) && !isViewerCapMessage )
#else
        if ( ( g_Viewer.DoDebugDraw() ) && !g_Pad.IsHold( nns::nac::Pad::MASK_R ) )
#endif
        {
            nn::gfx::util::DebugFontTextWriter* pDebugFontWriter = &g_Writer.object;

            // テキストを固定幅に設定
            float scaleX = pDebugFontWriter->GetScaleX();
            float fixedWidth = 12.f * scaleX;
            pDebugFontWriter->SetFixedWidthEnabled( true );
            pDebugFontWriter->SetFixedWidth( fixedWidth );

            // ここまでの文字を描画
            g_Writer.object.Draw( pCommandBuffer );

            // 2D用のフレームバッファに切り替え
            nn::gfx::ColorTargetView*  p2DTargetView = &g_2DTargetView;
            pCommandBuffer->ClearColor( p2DTargetView, 0.0f, 0.0f, 0.0f, 0.f, NULL );
            pCommandBuffer->SetRenderTargets( 1, &p2DTargetView, nullptr );
            {
                pDebugFontWriter->SetCursor( 0, 0 );
                g_pPrimitiveRenderer->SetBlendState( pCommandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Normal );
                g_pPrimitiveRenderer->SetDepthStencilState( pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthWriteTest );
                g_pPrimitiveRenderer->SetLineWidth( 1.f );
                g_Viewer.DebugDraw( pDebugFontWriter, pCommandBuffer, g_pPrimitiveRenderer, &g_GpuStatistics );

                // g_Viewer.DebugDraw 内のデバッグ文字を描画
                g_Writer.object.Draw( pCommandBuffer );
            }
            // カラーバッファを戻して
            pCommandBuffer->SetRenderTargets( 1, &pColorTargetView, pDepthStecilView );

            // 上書き位置
            nn::util::Matrix4x3fType modelMatrix;
            nn::util::MatrixIdentity( &modelMatrix );
            nn::util::Vector3fType scale = NN_UTIL_VECTOR_3F_INITIALIZER( 1280.f / g_WindowWidth, 720.f / g_WindowHeight, 1.f );
            nn::util::MatrixSetScale( &modelMatrix, scale );
            nn::util::Vector3fType trans = NN_UTIL_VECTOR_3F_INITIALIZER( 1280.f / g_WindowWidth - 1.f, -( 720.f / g_WindowHeight - 1.f ), 0.f );
            nn::util::MatrixSetTranslate( &modelMatrix, trans );

            // 2D を上書き
            g_pPrimitiveRenderer->SetModelMatrix( &modelMatrix );
            g_pPrimitiveRenderer->SetBlendState( pCommandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Normal );
            nn::util::Uint8x4 alpha = { { 255, 255, 255, 200 } };
            g_pPrimitiveRenderer->SetColor( alpha );
            g_pPrimitiveRenderer->DrawScreenQuadYFlip( pCommandBuffer, g_2DTextureDescSlot, g_Viewer.GetTextureSampler() );
            nn::util::MatrixIdentity( &modelMatrix );
            g_pPrimitiveRenderer->SetModelMatrix( &modelMatrix );
        }

#if NN_GFX_IS_TARGET_GL
        //------------------------------------------
        // OpenGL 使用の場合、ここでユーザーコマンドを呼ばせる
        //------------------------------------------
        pCommandBuffer->Gl4SetUserCommand( _vfxDrawPostCallback, pViewerLaunchArg );
#endif

        // カラーバッファ、クエリバッファをフラッシュ
        pCommandBuffer->FlushMemory( nn::gfx::GpuAccess_ColorBuffer || nn::gfx::GpuAccess_QueryBuffer );

#if defined( NN_BUILD_CONFIG_OS_WIN )
        //------------------------------------------
        // Windows のみ：画面のキャプチャコマンドを追加
        //------------------------------------------

        if ( g_ReservationScreenCapture )
        {
            g_FrameCapture.PushCaptureCommand( pCommandBuffer, &g_CaptureBuffer,
                                                  g_GraphicsFramework.GetSwapChain()->AcquireNextScanBuffer(), g_ScreenWidth, g_ScreenHeight );
            g_BufferIndex = 1 - bufferIndex;
            g_ReservationScreenCapture = false;
        }

        bool holdR = g_Pad.IsHold( nns::nac::Pad::MASK_R );

        if ( isViewerCapMessage || holdR )
        {
            g_ReservationScreenCapture = true;
        }
#endif
    }
    g_GraphicsFramework.EndFrame( bufferIndex );

    return;
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ取得用関数
//------------------------------------------------------------------------------
void* Allocate( size_t size ) NN_NOEXCEPT
{
    return malloc( size );
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ解放用関数
//------------------------------------------------------------------------------
void Deallocate( void* p, size_t size ) NN_NOEXCEPT
{
    NN_UNUSED( size );
    free( p );
}

//------------------------------------------------------------------------------
//  メイン関数
//------------------------------------------------------------------------------
extern "C" void nnMain() NN_NOEXCEPT
{
    //------------------------------------------
    // 起動引数判定
    //------------------------------------------
    ViewerLaunchArg viewerLaunchArg( nn::os::GetHostArgc(), nn::os::GetHostArgv() );

    //------------------------------------------
    // ファイルシステムのアロケータ設定
    //------------------------------------------
    nn::fs::SetAllocator( Allocate, Deallocate );
    nn::fs::MountHostRoot();

#if defined( NN_BUILD_CONFIG_OS_WIN ) && defined( NN_SDK_BUILD_DEBUG )
    _CrtSetDbgFlag(_CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF);
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    static const size_t   GraphicsSystemMemorySize = 8 * 1024 * 1024;
    g_GraphicsFramework.InitializeGraphicsSystem( GraphicsSystemMemorySize );
#endif

#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    // GLSLC のためのメモリアロケータを設定します。
    glslcSetAllocator( GlslcAllocateFunction, GlslcFreeFunction, GlslcReallocateFunction, NULL );
#endif

    //------------------------------------------
    // アロケータ初期化
    //------------------------------------------
    nns::nac::MemoryAllocator allocatorForGfx;
    uint8_t* const pMemForGfx = new uint8_t[ MemSizeForGfx ];
    allocatorForGfx.Initialize( pMemForGfx, MemSizeForGfx );

    nns::nac::MemoryAllocator allocatorForEffectSystem;
    uint8_t* const pMemForVfx = new uint8_t[ MemSizeForVfx ];
    allocatorForEffectSystem.Initialize( pMemForVfx, MemSizeForVfx );

    g_ScreenWidth  = 1280;
    g_ScreenHeight = 720;
    g_WindowWidth  = g_ScreenWidth;
    g_WindowHeight = g_ScreenHeight;


    //--------------------------　----------------
    // 描画フレームワーク設定
    //------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_DrawFrameworkMode = nns::gfx::GraphicsFramework::FrameworkMode_Immediate;
#else
    g_DrawFrameworkMode = nns::gfx::GraphicsFramework::FrameworkMode_DeferredExecution;
#endif

    //------------------------------------------
    // GraphicsFrameWorkの初期化
    //------------------------------------------
    g_GraphicsFrameworkInfo.SetDefault();
    g_GraphicsFrameworkInfo.SetDisplayWidth( g_ScreenWidth );
    g_GraphicsFrameworkInfo.SetDisplayHeight( g_ScreenHeight );
    if ( g_DrawFrameworkMode == nns::gfx::GraphicsFramework::FrameworkMode_DeferredExecution )
    {
        g_GraphicsFrameworkInfo.SetBufferCount( 2 );
        g_GraphicsFrameworkInfo.SetSwapChainBufferCount( 2 );
    }
    else
    {
        g_GraphicsFrameworkInfo.SetBufferCount( 1 );
        g_GraphicsFrameworkInfo.SetSwapChainBufferCount( 1 );
    }
    g_GraphicsFrameworkInfo.SetColorBufferFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
    g_GraphicsFrameworkInfo.SetDepthStencilBufferFormat( nn::gfx::ImageFormat_D32_Float );
    ViewerLaunchArg::DepthMode depthMode = viewerLaunchArg.GetDepthMode();
#if NN_GFX_IS_TARGET_NVN
    if ( depthMode == ViewerLaunchArg::DepthMode_Near_Is_Zero )
    {
        g_GraphicsFrameworkInfo.SetDepthMode( nns::gfx::GraphicsFramework::DepthMode::DepthMode_Near_Is_Zero );
    }
    else if ( depthMode == ViewerLaunchArg::DepthMode_Near_Is_Minus_W )
    {
        g_GraphicsFrameworkInfo.SetDepthMode( nns::gfx::GraphicsFramework::DepthMode::DepthMode_Near_Is_Minus_W  );
    }
#endif

#if NN_GFX_IS_TARGET_GL
    if ( depthMode == ViewerLaunchArg::DepthMode_Near_Is_Zero )
    {
        g_GraphicsFrameworkInfo.SetDepthMode( nns::gfx::GraphicsFramework::DepthMode::DepthMode_Near_Is_Zero  );
    }
#endif
    g_GraphicsFrameworkInfo.SetMemoryPoolSize(  nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 1024 * 1024 * 64 );
    g_GraphicsFrameworkInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, 1024 * 1024 * 64 );

    if ( viewerLaunchArg.GetSwapChainMode() == ViewerLaunchArg::SwapChainMode_Linear )
    {
        g_GraphicsFrameworkInfo.SetSwapChainFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    }
    else
    {
        g_GraphicsFrameworkInfo.SetSwapChainFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    }

    //g_GraphicsFrameworkInfo.SetDebugMode( nn::gfx::DebugMode_Enable );

    g_GraphicsFramework.Initialize( g_GraphicsFrameworkInfo );

    g_GraphicsFramework.SetMakeCommandCallback( MakeCommandCallback, &viewerLaunchArg );
    g_GraphicsFramework.SetFrameworkMode( g_DrawFrameworkMode );

    //------------------------------------------
    // nn::gfx の初期化
    //------------------------------------------
    InitializeGfx( g_GraphicsFramework.GetDevice(), &allocatorForGfx, MemSizeForGfx, g_ScreenWidth, g_ScreenHeight );

    // debug フォントの初期化
    nn::util::Unorm8x4 color0 = { { 255, 255, 255, 255 } };
    const float ScaleW = 0.8f;
    const float ScaleH = 1.0f;
    g_GraphicsFramework.InitializeDebugFontTextWriter( &g_Writer, 1024 );
    g_Writer.object.SetScale( ScaleW, ScaleH );
    g_Writer.object.SetTextColor( color0 );
    g_Writer.object.SetFixedWidthEnabled( true );
    g_Writer.object.SetTextureDescriptor( &g_TextureDescriptorPool, g_TextureDescPoolAllocator.GetStaticDescriptorPoolIndex( 1 ) );
    g_Writer.object.SetSamplerDescriptor( &g_SamplerDescriptorPool, g_SamplerDescPoolAllocator.GetStaticDescriptorPoolIndex( 1 ) );

#if NN_VFX_VIEWER_USE_PRIMITIVE_RENDERER
    //------------------------------------------
    // PrimitiveRenderer の初期化
    //------------------------------------------
    InitializePrimitiveRenderer( g_GraphicsFramework.GetDevice(), &allocatorForGfx );
#endif

    //------------------------------------------
    // 通信の初期化(アプリ側の義務になった)
    //------------------------------------------
#if (defined NN_ENABLE_HTC)
    if( !nn::htcs::IsInitialized() )
    {
        nn::htcs::Initialize(Allocate, Deallocate);
    }
#endif

#if NN_VFX_VIEWER_USE_PAD
    //------------------------------------------
    // 入力インターフェースの初期化
    //------------------------------------------
    InitializeInputInterface();
#endif
    //------------------------------------------
    // Effectの初期化
    //------------------------------------------
    g_Viewer.Initialize( g_GraphicsFramework.GetDevice(),
                         &allocatorForEffectSystem,
                         &g_TextureDescPoolAllocator,
                         &g_SamplerDescPoolAllocator,
                         g_ScreenWidth, g_ScreenHeight,
                         AllocateStaticTextureDescriptorSlot_,
                         AllocateStaticSamplerDescriptorSlot_,
                         viewerLaunchArg );
    g_Viewer.SetCamera( &g_DemoCamera );
    g_Viewer.SetProjectionMatrix( &g_ProjectionMatrix );
    if ( viewerLaunchArg.GetSwapChainMode() == ViewerLaunchArg::SwapChainMode_Linear )
    {
        g_Viewer.SetLinearMode( true );
    }
    else
    {
        g_Viewer.SetLinearMode( false );
    }

    // カメラを初期化
    nn::util::VectorSet( &g_CameraPosition, 0.0f, 30.0f, 50.0f );
    nn::util::VectorSet( &g_CameraLookAt,   0.0f,  0.0f,  0.0f );
    g_DemoCamera.SetPos( g_CameraPosition );
    g_DemoCamera.SetLookAtPos( g_CameraLookAt );
    g_DemoCamera.Preset();

    // プロジェクションを初期化
    const float fovy = nn::util::FloatPi / 3.0f;
    const float aspect = static_cast< float >( g_WindowWidth ) / static_cast< float >( g_WindowHeight );
    nn::util::MatrixPerspectiveFieldOfViewRightHanded( &g_ProjectionMatrix, fovy, aspect, 0.1f, 1000.f );

    g_Viewer.SetFovy( fovy );

    //------------------------------------------
    // スクリーンキャプチャの初期化
    //------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_WIN )
    // スクリーンキャプチャ用のフォルダを削除
    RemoveDirectory( L"ScreenCap" );

    // 起動フォルダパスを取得
    TCHAR baseDirectory[ _MAX_PATH ];
    GetCurrentDirectory( _MAX_PATH, baseDirectory );
    WideCharToMultiByte( CP_ACP, 0, baseDirectory, -1, g_WorkingDir, sizeof(g_WorkingDir), NULL, NULL );

    const uint32_t memSizeCapture   = 16 * 1024 * 1024;
    uint8_t* pMemForScreenCapture   = new uint8_t[ memSizeCapture ];
    g_HeapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(pMemForScreenCapture), memSizeCapture, nn::lmem::CreationOption_NoOption);
    g_FrameCapture.SetHeap( &g_HeapHandle );
#endif

    //------------------------------------------
    // 処理メータ( nn::perf )初期化
    //------------------------------------------
    {
        nn::perf::LoadMeterCenterInfo info;
        info.SetCoreCount( 1 );
        info.SetCpuSectionCountMax( 2048 );
        info.SetGpuSectionCountMax( 2048 );
        if ( g_DrawFrameworkMode == nns::gfx::GraphicsFramework::FrameworkMode_Immediate )
        {
            // 逐次描画モードの場合
            // 同フレーム内で CPU と GPU の処理が完了するため、バッファ数を等しく設定
            info.SetCpuBufferCount( 2 );
            info.SetGpuBufferCount( 2 );
        }
        else if ( g_DrawFrameworkMode == nns::gfx::GraphicsFramework::FrameworkMode_DeferredExecution )
        {
            // 遅延描画モードの場合
            // 1フレーム遅延して GPU の処理が完了するため、GPU のバッファ数を 1つ多く設定
            info.SetCpuBufferCount( 2 );
            info.SetGpuBufferCount( 3 );
        }

        size_t meterMemorySize = NN_PERF_GET_BUFFER_SIZE( info );
        size_t meterMemoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
        g_PerfMemory = GpuBufferAllocateFunction( meterMemorySize, meterMemoryAlignment, &allocatorForGfx );

        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            NN_PERF_GET_MEMORY_POOL_SIZE( g_GraphicsFramework.GetDevice(), info ),
            NN_PERF_GET_MEMORY_POOL_ALIGNMENT( g_GraphicsFramework.GetDevice(), info ) );

        NN_PERF_INITIALIZE_METER( g_GraphicsFramework.GetDevice(), info,
            g_PerfMemory, meterMemorySize,
            g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ),
            offset,
            g_GraphicsFramework.GetMemoryPoolAllocator( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer )->GetSize() );

        // 現在のスレッド(メインスレッド)を 0 番コアに割り当て
        nn::os::SetThreadCoreMask( nn::os::GetCurrentThread(), 0, 1 );

#if defined( NN_BUILD_CONFIG_OS_WIN )
        NN_PERF_SET_GET_CORE_NUMBER_FUNCTION(GetFixedCoreNumber);
#endif
        NN_PERF_SET_COLOR_GPU( nn::util::Color4u8::Blue() );
    }

    //------------------------------------------
    // GpuStatistics 初期化
    //------------------------------------------
    {
        GpuStatisticsInfo statisticsInfo;
        statisticsInfo.SetBufferCount( 2 );

        size_t gpuStatisticsMemorySize = GpuStatistics::CalculateBufferSize();
        size_t gpuStatisticsMemoryAlignment = GpuStatistics::GetBufferAlignment();
        void* gpuStatisticsMemory = GpuBufferAllocateFunction( gpuStatisticsMemorySize, gpuStatisticsMemoryAlignment, &allocatorForGfx );

        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            GpuStatistics::CalculateQueryBufferSize( g_GraphicsFramework.GetDevice(), statisticsInfo ),
            GpuStatistics::GetQueryBufferAlignment( g_GraphicsFramework.GetDevice(), statisticsInfo ) );

        g_GpuStatistics.Initialize( g_GraphicsFramework.GetDevice(), statisticsInfo,
            gpuStatisticsMemory, gpuStatisticsMemorySize,
            g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ),
            offset,
            g_GraphicsFramework.GetMemoryPoolAllocator( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer )->GetSize() );
    }

#if defined( NN_BUILD_CONFIG_OS_WIN )
    // ウインドウサイズを可変に。
    LONG lStyle = GetWindowLong( g_HWnd, GWL_STYLE );
    lStyle |= WS_SIZEBOX;
    lStyle |= WS_THICKFRAME;
    lStyle = SetWindowLong( g_HWnd, GWL_STYLE, lStyle );
#endif

    //------------------------------------------
    // ＜初期化ここまで＞
    //------------------------------------------

#if defined( NN_BUILD_CONFIG_OS_WIN )
    MSG  msg;
#endif

    //------------------------------------------
    // 毎フレームのレンダリング
    //------------------------------------------
    for( ; ; )
    {
        NN_PERF_BEGIN_FRAME();
        NN_PERF_SET_COLOR( nn::util::Color4u8::Gray() );

#if defined( NN_BUILD_CONFIG_OS_WIN )
        //------------------------------------------
        // Windows メッセージ処理
        //------------------------------------------
        if( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE ) )
        {
            TranslateMessage( &msg );
            if( msg.message == WM_QUIT )
            {
                break;
            }
            DispatchMessage( &msg );
        }
#endif

        //------------------------------------------
        // システムの状態更新
        //------------------------------------------
        {
            // フレームレート設定
            g_GraphicsFramework.SetPresentInterval( g_Viewer.GetSwapInterval() );

#if NN_VFX_VIEWER_USE_PAD
            // システム デモパッドの更新
            UpdateInputInterface();

            // カメラの更新
            g_DemoCamera.UpdateCamera( GetPad(), GetMouse(), GetMouse()->IsPointerOn(), true );
#endif
            g_DemoCamera.GetPos( &g_CameraPosition );
            g_DemoCamera.GetLookAtPos( &g_CameraLookAt );

            g_Viewer.UpdateProjection( static_cast< float >( g_WindowWidth ) / static_cast< float >( g_WindowHeight ) );
        }

#if defined( NN_BUILD_CONFIG_OS_WIN )
        g_Viewer.ProcCalc( &g_Mouse, &g_Pad, &g_DrawParam, AllocateTransientTextureDescriptorSlot_ );
#else
        g_Viewer.ProcCalc( &g_Pad, &g_DrawParam, AllocateTransientTextureDescriptorSlot_ );
#endif

#if defined( NN_BUILD_CONFIG_OS_WIN )
        if ( g_FrameCapture.IsCaptureCommandPushed() )
        {
            //nn::TimeSpan timeOut;
            //timeOut.FromMilliSeconds( 32 );
            //g_GraphicsFramework.GetGpuFence( g_BufferIndex )->Sync( timeOut );

            if ( strlen( g_Viewer.GetCaptureFilePath() ) > 0 )
            {
                g_FrameCapture.CaptureFrameWithFilePath( g_Viewer.GetCaptureFilePath() );
                g_Viewer.SetCaptureFilePath( nullptr );
            }
            else
            {
                int interval = 4;
                if ( g_Pad.IsHold( nns::nac::Pad::MASK_LS_SHIFT ) ||
                     g_Pad.IsHold( nns::nac::Pad::MASK_RS_SHIFT ) )
                {
                    interval = 1;
                }

                g_FrameCapture.CaptureFrameWithDirectoryPath( g_WorkingDir, interval );
            }
        }
        else
        {
            g_FrameCapture.ClearIndex();
        }
#endif

        // グラフィックスフレームワーク定期処理
        g_GraphicsFramework.ProcessFrame();

        NN_PERF_END_FRAME();
        g_GpuStatistics.Next();
    }

    //------------------------------------------
    // 各オブジェクトを破棄
    //------------------------------------------
    g_GpuStatistics.Finalize( g_GraphicsFramework.GetDevice() );
    NN_PERF_FINALIZE_METER( g_GraphicsFramework.GetDevice() );
    GpuBufferDeallocateFunction( g_PerfMemory, &allocatorForGfx );
    g_Viewer.Finalize( g_GraphicsFramework.GetDevice(), &allocatorForEffectSystem, FreeStaticTextureDescriptorSlot_,  FreeStaticSamplerDescriptorSlot_ );
    FinalizeInputInterface();

#if NN_VFX_VIEWER_USE_PRIMITIVE_RENDERER
    FinalizePrimitiveRenderer( g_GraphicsFramework.GetDevice(), &allocatorForGfx );
#endif

    FinalizeGfx( g_GraphicsFramework.GetDevice(), &allocatorForGfx );

    nn::fs::UnmountHostRoot();

#if (defined NN_ENABLE_HTC)
    nn::htcs::Finalize();
#endif

#if defined( NN_BUILD_CONFIG_OS_WIN )
    delete[] pMemForScreenCapture;
#endif

    delete[] pMemForVfx;
    delete[] pMemForGfx;

    return;
}

